@file:Suppress("DEPRECATION", "unused")

package com.paymentpage.msdk.core.network

import android.content.Context
import android.net.ConnectivityManager
import android.net.Network
import android.net.NetworkCapabilities
import android.net.NetworkRequest
import android.os.Build
import com.paymentpage.msdk.core.appContext
import com.paymentpage.msdk.core.utils.Log
import kotlinx.coroutines.flow.MutableStateFlow

internal actual class ConnectivityStatus {
    actual val networkState = MutableStateFlow(ConnectivityStatusState.UNKNOWN)

    private val connectivityManager: ConnectivityManager? = null
    private val networkCallback = object : ConnectivityManager.NetworkCallback() {
        override fun onAvailable(network: Network) {
            Log.d("ConnectivityStatus") { "Connected" }
            networkState.value = ConnectivityStatusState.CONNECTED
        }

        override fun onLost(network: Network) {
            Log.d("ConnectivityStatus") { "Disconnected" }
            networkState.value = ConnectivityStatusState.DISCONNECTED
        }
    }

    init {
        try {
            val connectivityManager =
                appContext?.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                // API 24 and above
                connectivityManager.registerDefaultNetworkCallback(networkCallback)

                val currentNetwork = connectivityManager.activeNetwork

                if (currentNetwork == null) {
                    networkState.value = ConnectivityStatusState.DISCONNECTED
                    Log.d("ConnectivityStatus") { "Disconnected" }
                }
            } else {
                // API 23 and below
                val networkRequest = NetworkRequest.Builder().apply {
                    addCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET)
                    addCapability(NetworkCapabilities.NET_CAPABILITY_NOT_RESTRICTED)
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                        addCapability(NetworkCapabilities.NET_CAPABILITY_VALIDATED)
                    }
                }.build()

                connectivityManager.registerNetworkCallback(networkRequest, networkCallback)

                val currentNetwork = connectivityManager.activeNetworkInfo

                if (currentNetwork == null || (
                            currentNetwork.type != ConnectivityManager.TYPE_ETHERNET &&
                                    currentNetwork.type != ConnectivityManager.TYPE_WIFI &&
                                    currentNetwork.type != ConnectivityManager.TYPE_MOBILE
                            )
                ) {
                    networkState.value = ConnectivityStatusState.DISCONNECTED
                    Log.d("ConnectivityStatus") { "Disconnected" }
                }
            }
        } catch (e: Exception) {
            Log.d("ConnectivityStatus") { "Disconnected" }
            Log.w("ConnectivityStatus") { e.stackTrace }
            networkState.value = ConnectivityStatusState.DISCONNECTED
        }
    }

//    actual fun stop() {
//        connectivityManager?.unregisterNetworkCallback(networkCallback)
//    }
}
