package com.paymentpage.msdk.core.network.webSocket

import com.paymentpage.msdk.core.base.Response
import com.paymentpage.msdk.core.utils.Log
import com.paymentpage.msdk.core.utils.extensions.safeRun
import io.ktor.client.*
import io.ktor.client.plugins.websocket.*
import io.ktor.http.*
import io.ktor.websocket.*
import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.cancel
import kotlinx.coroutines.flow.*

internal actual class PlatformWebSocketEngine actual constructor(
    private val httpClient: HttpClient,
    private val apiUrl: String
) : WebSocketEngine {
    private var wsSession: WebSocketSession? = null

    private val _state = MutableStateFlow<WebSocketEngine.State>(WebSocketEngine.State.Disconnected)
    private val _messages = MutableSharedFlow<String>()

    override val state = _state.asStateFlow()
    override val messages = _messages.asSharedFlow()

    override suspend fun connect() {
        val session =
            safeRun { httpClient.webSocketSession { this@webSocketSession.url.takeFrom(apiUrl) } }
        when (session) {
            is Response.Success -> {
                wsSession = session.value
                _state.value = WebSocketEngine.State.Connected
                startListenWebSocket()
            }
            is Response.Error -> _state.value = WebSocketEngine.State.Error(session.exception)
        }
    }

    override suspend fun disconnect() {
        wsSession?.close()
        clearSession()
    }

    override suspend fun send(message: String) {
        wsSession?.send(message)
    }

    private fun startListenWebSocket() {
        val session = wsSession ?: return

        session.incoming.receiveAsFlow()
            .map { frame -> frame.data.decodeToString() }
            .filter { message -> message.isNotEmpty() }
            .onEach { message ->
                Log.d { "Ws message: $message" }
                _messages.emit(message)
            }
            .catch { error -> Log.d { "Ws error: $error" } }
            .onCompletion { error ->
                if (error is CancellationException) clearSession() else clearSession(
                    error
                )
            }
            .launchIn(session)
    }

    private fun clearSession(error: Throwable? = null) {
        _state.value =
            if (error == null) WebSocketEngine.State.Disconnected else WebSocketEngine.State.Error(
                error
            )
        wsSession?.cancel()
        wsSession = null
    }
}