package kh.org.nbc.bakong_khqr;

import kh.org.nbc.bakong_khqr.exception.KHQRException;
import kh.org.nbc.bakong_khqr.model.*;
import kh.org.nbc.bakong_khqr.presenter.GenerateDeepLinkPresenter;
import kh.org.nbc.bakong_khqr.presenter.MerchantPresentedDecodeMode;
import kh.org.nbc.bakong_khqr.presenter.MerchantPresentedMode;
import kh.org.nbc.bakong_khqr.utils.StringUtils;

import java.math.BigInteger;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import static kh.org.nbc.bakong_khqr.model.Constant.ERROR_CODE;
import static kh.org.nbc.bakong_khqr.model.Constant.SUCCESS_CODE;
import static kh.org.nbc.bakong_khqr.model.KHQRErrorCode.*;
import static kh.org.nbc.bakong_khqr.utils.BakongKHQRUtils.getChecksumResult;

public class BakongKHQR {

    private static final String PAYLOAD_FORMAT_INDICATOR = "01";
    private static final String MERCHANT_CATEGORY_CODE = "5999";
    private static final String COUNTRY_CODE = "KH";

    public static KHQRResponse<KHQRData> generateIndividual(IndividualInfo individualInfo) {
        KHQRResponse<KHQRData> khqrResponse = new KHQRResponse<>();
        KHQRStatus KHQRStatus = new KHQRStatus();
        try {
            KHQRData khqrData = new KHQRData();
            String qr = getIndividualQr(individualInfo);
            khqrData.setQr(qr);
            khqrData.setMd5(hashTextToMd5(qr));
            khqrResponse.setData(khqrData);
        } catch (KHQRException exception) {
            KHQRStatus.setCode(ERROR_CODE);
            KHQRStatus.setErrorCode(exception.getErrorCode());
            KHQRStatus.setMessage(exception.getMessage());
        }
        khqrResponse.setKHQRStatus(KHQRStatus);
        return khqrResponse;
    }

    public static KHQRResponse<KHQRData> generateMerchant(MerchantInfo merchantInfo) {
        KHQRResponse<KHQRData> khqrResponse = new KHQRResponse<>();
        KHQRStatus KHQRStatus = new KHQRStatus();
        try {
            KHQRData khqrData = new KHQRData();
            String qr = getMerchantQr(merchantInfo);
            khqrData.setQr(qr);
            khqrData.setMd5(hashTextToMd5(qr));
            khqrResponse.setData(khqrData);
        } catch (KHQRException exception) {
            KHQRStatus.setCode(ERROR_CODE);
            KHQRStatus.setErrorCode(exception.getErrorCode());
            KHQRStatus.setMessage(exception.getMessage());
        }
        khqrResponse.setKHQRStatus(KHQRStatus);
        return khqrResponse;
    }

    public static KHQRResponse<CRCValidation> verify(String khqrCode) {
        KHQRStatus khqrStatus = new KHQRStatus();
        KHQRResponse<CRCValidation> khqrResponse = new KHQRResponse<>();
        if (StringUtils.isBlank(khqrCode) || khqrCode.trim().length() < 8) {
            setStatusInvalidQr(khqrStatus, khqrResponse);
        } else {
            KHQRResponse<KHQRDecodeData> khqrDecodeData = decodeToVerify(khqrCode);
            CRCValidation crcValidation = new CRCValidation();
            if (khqrDecodeData.getKHQRStatus().getCode() == SUCCESS_CODE) {
                khqrResponse.setKHQRStatus(khqrStatus);
                khqrCode = khqrCode.trim();
                String checksum = khqrCode.substring(khqrCode.length() - 4).toUpperCase();
                String dataPayload = khqrCode.substring(0, khqrCode.length() - 4);
                crcValidation.setValid(checksum.equals(getChecksumResult(dataPayload)));
            } else {
                khqrResponse.setKHQRStatus(khqrDecodeData.getKHQRStatus());
                crcValidation.setValid(false);
            }
            khqrResponse.setData(crcValidation);
        }
        return khqrResponse;
    }

    public static KHQRResponse<KHQRDeepLinkData> generateDeepLink(String url, String qr, SourceInfo sourceInfo) {
        // check null or valid url
        if (KHQRValidation.isEmpty(url) || KHQRValidation.isUrlInvalid(url)) {
            return GenerateDeepLinkPresenter.responseError(INVALID_DEEP_LINK_URL);
        }
        // verify
        KHQRResponse<CRCValidation> crcValidation = verify(qr);
        if (crcValidation.getKHQRStatus().getCode() == ERROR_CODE || !crcValidation.getData().isValid()) {
            KHQRStatus status = crcValidation.getKHQRStatus();
            if (status.getErrorCode() == null || status.getMessage() == null) {
                return GenerateDeepLinkPresenter.responseError(KHQR_INVALID);
            } else {
                return GenerateDeepLinkPresenter.responseError(status.getErrorCode());
            }
        }
        // check source info
        if (KHQRValidation.isSourceInfoInvalid(sourceInfo)) {
            return GenerateDeepLinkPresenter.responseError(INVALID_DEEP_LINK_SOURCE_INFO);
        }
        // generate deep link
        try {
            return new GenerateDeepLinkPresenter(url, qr, sourceInfo).generate();
        } catch (KHQRException exception) {
            return GenerateDeepLinkPresenter.responseError(exception.getErrorCode());
        }
    }

    public static KHQRResponse<KHQRDecodeData> decode(String qr) {
        KHQRResponse<KHQRDecodeData> khqrResponse = new KHQRResponse<>();
        KHQRStatus khqrStatus = new KHQRStatus();
        try {
            KHQRDecodeData khqrDecodeData = new MerchantPresentedDecodeMode().decode(qr, false);
            khqrResponse.setKHQRStatus(khqrStatus);
            khqrResponse.setData(khqrDecodeData);
        } catch (KHQRException e) {
            e.printStackTrace();
        }
        return khqrResponse;
    }
    private static KHQRResponse<KHQRDecodeData> decodeToVerify(String qr) {
        KHQRResponse<KHQRDecodeData> khqrResponse = new KHQRResponse<>();
        KHQRStatus khqrStatus = new KHQRStatus();
        if (StringUtils.isBlank(qr) || qr.trim().length() < 8) {
            setInvalidQrData(khqrResponse, khqrStatus);
        } else {
            try {
                KHQRDecodeData khqrDecodeData = new MerchantPresentedDecodeMode().decode(qr, true);
                khqrResponse.setKHQRStatus(khqrStatus);
                khqrResponse.setData(khqrDecodeData);
            } catch (KHQRException e) {
                khqrStatus.setCode(ERROR_CODE);
                khqrStatus.setErrorCode(e.getErrorCode());
                khqrStatus.setMessage(e.getMessage());
                khqrResponse.setKHQRStatus(khqrStatus);
            }
        }
        return khqrResponse;
    }

    private static void setInvalidQrData(KHQRResponse<KHQRDecodeData> khqrResponse, KHQRStatus khqrStatus) {
        khqrStatus.setErrorCode(KHQR_INVALID);
        khqrStatus.setCode(ERROR_CODE);
        khqrStatus.setMessage(KHQRErrorCode.errorCodeMap.get(KHQR_INVALID));
        khqrResponse.setKHQRStatus(khqrStatus);
    }


    private static void setStatusInvalidQr(KHQRStatus khqrStatus, KHQRResponse<CRCValidation> khqrResponse) {
        khqrStatus.setErrorCode(KHQR_INVALID);
        khqrStatus.setCode(ERROR_CODE);//0->Success and 1->Error
        khqrStatus.setMessage(KHQRErrorCode.errorCodeMap.get(KHQR_INVALID));
        khqrResponse.setKHQRStatus(khqrStatus);
    }

    private static String getMerchantQr(MerchantInfo merchantInfo) throws KHQRException {
        MerchantPresentedMode merchantPresentedMode = new MerchantPresentedMode();
        merchantPresentedMode.addMerchantAccountInformation(getMerchantAccountInfo(merchantInfo));
        merchantPresentedMode.setAdditionalDataField(getAdditionalDataFieldTemplate(merchantInfo.getTerminalLabel(), merchantInfo.getStoreLabel(), merchantInfo.getBillNumber(), merchantInfo.getMobileNumber()));
        return getMerchantPresentedMode(merchantPresentedMode, merchantInfo.getMerchantName(), merchantInfo.getAmount(), merchantInfo.getCurrency(), merchantInfo.getMerchantCity());
    }

    private static String getIndividualQr(IndividualInfo individualInfo) throws KHQRException {
        MerchantPresentedMode merchantPresentedMode = new MerchantPresentedMode();
        merchantPresentedMode.addMerchantAccountInformation(getIndividualAccountInfo(individualInfo));
        merchantPresentedMode.setAdditionalDataField(getAdditionalDataFieldTemplate(individualInfo.getTerminalLabel(), individualInfo.getStoreLabel(), individualInfo.getBillNumber(), individualInfo.getMobileNumber()));
        return getMerchantPresentedMode(merchantPresentedMode, individualInfo.getMerchantName(), individualInfo.getAmount(), individualInfo.getCurrency(), individualInfo.getMerchantCity());
    }

    private static String getMerchantPresentedMode(MerchantPresentedMode merchantPresentedMode, String merchantName, Double amount, KHQRCurrency currency, String merchantCity) throws KHQRException {

        merchantPresentedMode.setMerchantName(KHQRValidation.getMerchantName(merchantName));
        merchantPresentedMode.setTransactionCurrency(currency.getValue());
        merchantPresentedMode.setTransactionAmount(KHQRValidation.getAmount(amount, currency));
        merchantPresentedMode.setMerchantCity(KHQRValidation.getMerchantCity(merchantCity));
        merchantPresentedMode.setCountryCode(COUNTRY_CODE);
        merchantPresentedMode.setPayloadFormatIndicator(PAYLOAD_FORMAT_INDICATOR);
        merchantPresentedMode.setPointOfInitiationMethod(KHQRValidation.getQrType(amount));
        merchantPresentedMode.setMerchantCategoryCode(MERCHANT_CATEGORY_CODE);
        return merchantPresentedMode.toString();
    }

    private static MerchantAccountInformationTemplate getMerchantAccountInfo(MerchantInfo merchantInfo) throws KHQRException {
        String bakongAccountID = KHQRValidation.getBakongAccountId(merchantInfo.getBakongAccountId());
        final MerchantAccountInformation merchantAccountInformationValue = new MerchantAccountInformation();
        merchantAccountInformationValue.setGloballyUniqueIdentifier(bakongAccountID);
        merchantAccountInformationValue.setMerchantId(KHQRValidation.getMerchantId(merchantInfo.getMerchantId()));
        merchantAccountInformationValue.setAcquiringBank(KHQRValidation.getAcquiringBankForMerchant(merchantInfo.getAcquiringBank()));
        final MerchantAccountInformationTemplate merchantAccountInformation = new MerchantAccountInformationTemplate();
        merchantAccountInformation.setTag(KHQRMerchantType.MERCHANT.getType());
        merchantAccountInformation.setValue(merchantAccountInformationValue);
        return merchantAccountInformation;
    }

    private static MerchantAccountInformationTemplate getIndividualAccountInfo(IndividualInfo individualInfo) throws KHQRException {
        String bakongAccountID = KHQRValidation.getBakongAccountId(individualInfo.getBakongAccountId());
        final MerchantAccountInformation merchantAccountInformationValue = new MerchantAccountInformation();
        merchantAccountInformationValue.setGloballyUniqueIdentifier(bakongAccountID);
        merchantAccountInformationValue.setAccountInformation(KHQRValidation.getAccountInformation(individualInfo.getAccountInformation()));
        merchantAccountInformationValue.setAcquiringBank(KHQRValidation.getAcquiringBankForIndividual(individualInfo.getAcquiringBank()));
        final MerchantAccountInformationTemplate merchantAccountInformation = new MerchantAccountInformationTemplate();
        merchantAccountInformation.setTag(KHQRMerchantType.INDIVIDUAL.getType());
        merchantAccountInformation.setValue(merchantAccountInformationValue);
        return merchantAccountInformation;
    }

    private static AdditionalDataFieldTemplate getAdditionalDataFieldTemplate(String terminalLabel, String storeLabel, String billNumber, String mobileNumber) throws KHQRException {
        final AdditionalDataField additionalDataField = new AdditionalDataField();
        additionalDataField.setTerminalLabel(KHQRValidation.getTerminalLabel(terminalLabel));
        additionalDataField.setStoreLabel(KHQRValidation.getStoreLabel(storeLabel));
        additionalDataField.setBillNumber(KHQRValidation.getBillNumber(billNumber));
        additionalDataField.setMobileNumber(KHQRValidation.getMobileNumber(mobileNumber));
        final AdditionalDataFieldTemplate additionalDataFieldTemplate = new AdditionalDataFieldTemplate();
        additionalDataFieldTemplate.setValue(additionalDataField);
        return additionalDataFieldTemplate;
    }

    private static String hashTextToMd5(String qr) {
        MessageDigest m;
        try {
            m = MessageDigest.getInstance("MD5");
            m.reset();
            m.update(qr.getBytes());
            byte[] digest = m.digest();
            BigInteger bigInt = new BigInteger(1, digest);
            StringBuilder hashText = new StringBuilder(bigInt.toString(16));
            while (hashText.length() < 32) {
                hashText.insert(0, "0");
            }
            return hashText.toString();
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }
        return null;
    }

}
