package kh.org.nbc.bakong_khqr.model;


import kh.org.nbc.bakong_khqr.exception.KHQRException;
import kh.org.nbc.bakong_khqr.utils.BakongKHQRUtils;
import kh.org.nbc.bakong_khqr.utils.StringUtils;
import org.apache.commons.validator.routines.UrlValidator;

import java.math.BigDecimal;
import java.util.regex.Pattern;

import static kh.org.nbc.bakong_khqr.exception.KHQRException.throwCustomerException;

public class KHQRValidation {

    private static int MERCHANT_LENGTH = 25;
    private static int AMOUNT_LENGTH = 13;
    private static int STORE_LABEL_LENGTH = 25;
    private static int TERMINAL_LABEL_LENGTH = 25;
    private static int MOBILE_NUMBER_LENGTH = 12;
    private static int BILL_NUMBER_LENGTH = 25;
    private static int ACCOUNT_ID_LENGTH = 32;
    private static int PAYLOAD_FORMAT_LENGTH = 2;
    private static int CATEGORY_CODE_LENGTH = 4;
    private static int TRANSACTION_CURRENCY = 3;
    private static int CRC_LENGTH = 4;
    private static int MERCHANT_CITY_LENGTH = 15;
    private static int MERCHANT_ID_LENGTH = 32;
    private static int ACCOUNT_INFORMATION_LENGTH = 32;
    private static int ACQUIRING_BANK_LENGTH = 32;
    private static int COUNTRY_CODE_LENGTH = 2;
    private static String FORMAT_ACCOUNT_ID = "^[^@]+[@][^@]+$";


    public static boolean isMerchantNameEmpty(String merchantName) {
        return StringUtils.isBlank(merchantName);
    }

    public static boolean isMerchantNameLengthInvalid(String merchantName) {
        return merchantName.length() > MERCHANT_LENGTH;
    }

    public static boolean isMerchantIdEmpty(String merchantId) {
        return StringUtils.isBlank(merchantId);
    }

    public static boolean isMerchantIdLengthInvalid(String merchantId) {
        return merchantId.length() > MERCHANT_ID_LENGTH;
    }

    public static boolean isAccountInformationLengthInvalid(String accountInformation) {
        return accountInformation.length() > ACCOUNT_INFORMATION_LENGTH;
    }

    public static boolean isAcquiringBankEmpty(String acquiringBank) {
        return StringUtils.isBlank(acquiringBank);
    }

    public static boolean isAcquiringBankLengthInvalid(String acquiringBank) {
        return acquiringBank.length() > ACQUIRING_BANK_LENGTH;
    }


    public static boolean isMerchantTypeIsEmpty(KHQRMerchantType merchantType) {
        return merchantType == null;
    }

    public static boolean isAmountInvalid(Double amount, KHQRCurrency currency) {
        if (amount != null) {

            BigDecimal bd = BigDecimal.valueOf(amount);
            String valueAmount = bd.stripTrailingZeros().toPlainString();
            int index = valueAmount.indexOf(".");
            int decimalCount = index < 0 ? 0 : valueAmount.length() - index - 1;

            if (amount < 0 || valueAmount.length() > AMOUNT_LENGTH) {
                return true;
            }
            if (currency == KHQRCurrency.USD) {
                //0.##
                return decimalCount >= 3;

            } else {
                //0
                return decimalCount != 0;
            }
        } else return false;
    }

    public static boolean isStoreLabelLengthInvalid(String storeLabel) {
        return storeLabel != null && storeLabel.length() > STORE_LABEL_LENGTH;

    }

    public static boolean isStoreLabelIsNotEmpty(String storeLabel) {
        return StringUtils.isNoneBlank(storeLabel);
    }

    public static boolean isTerminalLabelLengthInvalid(String terminalLabel) {
        return terminalLabel != null && terminalLabel.length() > TERMINAL_LABEL_LENGTH;

    }

    public static boolean isTerminalLabelIsNotEmpty(String terminalLabel) {
        return StringUtils.isNoneBlank(terminalLabel);
    }

    public static boolean isBillNumberLengthInvalid(String billNumber) {
        return billNumber != null && billNumber.length() > BILL_NUMBER_LENGTH;

    }

    public static boolean isBillNumberIsNotEmpty(String billNumber) {
        return StringUtils.isNoneBlank(billNumber);
    }

    public static boolean isMobileNumberIsNotEmpty(String mobileNumber) {
        return StringUtils.isNoneBlank(mobileNumber);
    }

    public static boolean isMobileNumberLengthInvalid(String mobileNumber) {
        return mobileNumber != null && mobileNumber.length() > MOBILE_NUMBER_LENGTH;

    }

    public static boolean isAccountIdIsEmpty(String accountId) {
        return StringUtils.isBlank(accountId);
    }

    public static boolean isAccountIdLengthInvalid(String accountId) {
        return accountId.length() > ACCOUNT_ID_LENGTH;
    }

    public static boolean isAccountIdInvalidFormat(String accountId) {
        return !Pattern.matches(FORMAT_ACCOUNT_ID, accountId);
    }

    public static boolean isPayLoadFormatIndicatorLengthInvalid(String payload) {
        return payload.length() > PAYLOAD_FORMAT_LENGTH;
    }

    public static boolean isMerchantCategoryCodeLengthInvalid(String categoryCode) {
        return categoryCode.length() > CATEGORY_CODE_LENGTH;
    }

    public static boolean isTransactionCurrencyLengthInvalid(String transactionCurrency) {
        return transactionCurrency.length() > TRANSACTION_CURRENCY;
    }

    public static boolean isCrcLengthInvalid(String crc) {
        return crc.length() > CRC_LENGTH;
    }

    public static boolean isQrValidCrc(String qr) {
        String expectCrc = qr.substring(qr.length() - 4).toUpperCase();
        String qrWithoutCrc = qr.substring(0, qr.length() - 8);
        String crc = BakongKHQRUtils.getCRC(qrWithoutCrc).toUpperCase();
        return expectCrc.equals(crc);
    }
    public static boolean isContainCrcTagValue(String qr) {
        String crc = qr.substring(qr.length() - 8).toUpperCase();
        return Pattern.matches("6304[A-F0-9]{4}$", crc);
    }

    public static boolean isMerchantCityLengthInvalid(String merchantCity) {
        return merchantCity.length() > MERCHANT_CITY_LENGTH;
    }

    public static boolean isCountryCodeLengthInvalid(String countryCode) {
        return countryCode.length() > COUNTRY_CODE_LENGTH;
    }

    public static boolean isUrlInvalid(String url) {
        String[] schemes = {"http", "https"}; // DEFAULT schemes = "http", "https", "ftp"
        UrlValidator urlValidator = new UrlValidator(schemes);
        return !urlValidator.isValid(url);
    }

    public static boolean isEmpty(String text) {
        return text == null || text.isEmpty();
    }

    public static boolean isSourceInfoInvalid(SourceInfo sourceInfo) {
        if (sourceInfo == null) return false;
        String appName = sourceInfo.getAppName();
        String appDeepLinkCallback = sourceInfo.getAppDeepLinkCallback();
        String appIconUrl = sourceInfo.getAppIconUrl();

        return isEmpty(appName) || isEmpty(appDeepLinkCallback) || isEmpty(appIconUrl);
    }

    /// new validation

    public static String getMerchantName(String merchantName) throws KHQRException {
        if (isMerchantNameEmpty(merchantName)) {
            throwCustomerException(KHQRErrorCode.MERCHANT_NAME_REQUIRED);
        } else if (isMerchantNameLengthInvalid(merchantName)) {
            throwCustomerException(KHQRErrorCode.MERCHANT_NAME_LENGHT_INVALID);
        }
        return merchantName;
    }

    public static String getAmount(Double amount, KHQRCurrency currency) throws KHQRException {

        if (isAmountInvalid(amount, currency)) {
            throwCustomerException(KHQRErrorCode.TRANSACTION_AMOUNT_INVALID);
        } else if (amount == null || amount == 0) {
            return null;
        }

        BigDecimal bd = BigDecimal.valueOf(amount);
        return bd.stripTrailingZeros().toPlainString();
    }

    public static String getMerchantCity(String merchantCity) throws KHQRException {
        if (StringUtils.isBlank(merchantCity)) {
            return "Phnom Penh";
        } else if (isMerchantCityLengthInvalid(merchantCity)) {
            throwCustomerException(KHQRErrorCode.MERCHANT_CITY_LENGTH_INVALID);
        }
        return merchantCity;
    }

    public static String getQrType(Double amount) {
        if (amount == null || amount == 0) {
            return KHQRType.STATIC.getType();
        } else return KHQRType.DYNAMIC.getType();
    }

    public static String getBakongAccountId(String bakongAccountID) throws KHQRException {
        if (isAccountIdIsEmpty(bakongAccountID)) {
            throwCustomerException(KHQRErrorCode.BAKONG_ACCOUNT_ID_REQUIRED);
        } else if (isAccountIdLengthInvalid(bakongAccountID)) {
            throwCustomerException(KHQRErrorCode.BAKONG_ACCOUNT_ID_LENGTH_INVALID);
        } else if (isAccountIdInvalidFormat(bakongAccountID)) {
            throwCustomerException(KHQRErrorCode.BAKONG_ACCOUNT_ID_INVALID);
        }
        return bakongAccountID;
    }

    public static String getTerminalLabel(String terminalLabel) throws KHQRException {
        if (isTerminalLabelIsNotEmpty(terminalLabel)) {
            if (isTerminalLabelLengthInvalid(terminalLabel)) {
                throwCustomerException(KHQRErrorCode.TERMINAL_LABEL_LENGTH_INVALID);
            } else {
                return terminalLabel;
            }
        }
        return null;
    }

    public static String getStoreLabel(String storeLabel) throws KHQRException {
        if (KHQRValidation.isStoreLabelIsNotEmpty(storeLabel)) {
            if (isStoreLabelLengthInvalid(storeLabel)) {
                throwCustomerException(KHQRErrorCode.STORE_LABEL_LENGTH_INVALID);
            } else {
                return storeLabel;
            }
        }
        return null;
    }

    public static String getBillNumber(String billNumber) throws KHQRException {
        if (isBillNumberIsNotEmpty(billNumber)) {
            if (isBillNumberLengthInvalid(billNumber)) {
                throwCustomerException(KHQRErrorCode.BILL_NUMBER_LENGTH_INVALID);
            } else {
                return billNumber;
            }
        }
        return null;
    }

    public static String getMobileNumber(String mobileNumber) throws KHQRException {
        if (isMobileNumberIsNotEmpty(mobileNumber)) {
            if (isBillNumberLengthInvalid(mobileNumber)) {
                throwCustomerException(KHQRErrorCode.MOBILE_NUMBER_LENGTH_INVALID);
            } else {
                return mobileNumber;
            }
        }
        return null;
    }

    public static String getMerchantId(String merchantId) throws KHQRException {
        if (isMerchantIdEmpty(merchantId)) {
            throwCustomerException(KHQRErrorCode.MERCHANT_ID_REQUIRED);
        } else if (isMerchantIdLengthInvalid(merchantId)) {
            throwCustomerException(KHQRErrorCode.MERCHANT_ID_LENGTH_INVALID);
        }
        return merchantId;
    }

    public static String getAccountInformation(String accountInformation) throws KHQRException {
        if (accountInformation != null && isAccountInformationLengthInvalid(accountInformation)) {
            throwCustomerException(KHQRErrorCode.ACCOUNT_INFORMATION_LENGTH_INVALID);
        }
        return accountInformation;
    }

    public static String getAcquiringBankForMerchant(String acquiringBank) throws KHQRException {
        if (isMerchantIdEmpty(acquiringBank)) {
            throwCustomerException(KHQRErrorCode.ACQUIRING_BANK_REQUIRED);
        } else if (isMerchantIdLengthInvalid(acquiringBank)) {
            throwCustomerException(KHQRErrorCode.ACQUIRING_BANK_LENGTH_INVALID);
        }
        return acquiringBank;
    }

    public static String getAcquiringBankForIndividual(String acquiringBank) throws KHQRException {
        if (acquiringBank != null && isMerchantIdLengthInvalid(acquiringBank)) {
            throwCustomerException(KHQRErrorCode.ACQUIRING_BANK_LENGTH_INVALID);
        }
        return acquiringBank;
    }

}
