package kh.org.nbc.bakong_khqr.presenter;

import com.google.gson.Gson;
import kh.org.nbc.bakong_khqr.exception.KHQRException;
import kh.org.nbc.bakong_khqr.model.*;
import okhttp3.*;

import javax.net.ssl.*;
import java.net.ConnectException;
import java.net.SocketTimeoutException;
import java.util.concurrent.TimeUnit;

import static kh.org.nbc.bakong_khqr.model.Constant.ERROR_CODE;
import static kh.org.nbc.bakong_khqr.model.Constant.SUCCESS_CODE;

public class GenerateDeepLinkPresenter {

    private static final int CONNECTION_TIME_OUT = 45000;

    private String qr;
    private SourceInfo sourceInfo;
    private String baseUrl;

    public GenerateDeepLinkPresenter(String url, String qr, SourceInfo sourceInfo) {
        this.qr = qr;
        this.sourceInfo = sourceInfo;
        this.baseUrl = url;
    }

    public KHQRResponse<KHQRDeepLinkData> generate() throws KHQRException {

        KHQRGenerateDeepLinkResponse baseResponse;
        Gson gson = new Gson();
        GenerateDeepLinkBody requestBody = new GenerateDeepLinkBody();
        requestBody.setQr(qr);
        if (sourceInfo != null)
            requestBody.setSourceInfo(sourceInfo);

        OkHttpClient client = getUnsafeOkHttpClient().newBuilder().connectTimeout(CONNECTION_TIME_OUT, TimeUnit.MILLISECONDS)
                .build();
        MediaType mediaType = MediaType.parse("application/json");
        RequestBody body = RequestBody.create(mediaType, gson.toJson(requestBody));
        Request request = new Request.Builder()
                .url(baseUrl)
                .method("POST", body)
                .addHeader("Content-Type", "application/json")
                .build();

        try {
            Response response = client.newCall(request).execute();
            if (response.isSuccessful()) {
                if (response.body() != null) {
                    ResponseBody responseBody = response.body();
                    if (responseBody != null) {
                        String json = responseBody.string();
                        baseResponse = gson.fromJson(
                                json,
                                KHQRGenerateDeepLinkResponse.class
                        );
                        return responseMapping(baseResponse);
                    }
                }
            } else {
                // 404, 500 or else
                return responseParser(new KHQRGenerateDeepLinkResponse(null, ERROR_CODE, KHQRErrorCode.CONNECTION_TIMEOUT,
                        KHQRErrorCode.errorCodeMap.get(KHQRErrorCode.CONNECTION_TIMEOUT)));
            }

        } catch (Exception e) {
            if (e instanceof ConnectException || e instanceof SocketTimeoutException) {
                throwCustomerException(KHQRErrorCode.CONNECTION_TIMEOUT);
            } else {
                //if response is not instance of KHQRGenerateDeepLinkResponse
                throwCustomerException(KHQRErrorCode.INTERNAL_SERVER_ERROR);
            }
        }
        return null;
    }

    private void throwCustomerException(int errorCode) throws KHQRException {
        throw new KHQRException(errorCode);
    }

    private KHQRResponse<KHQRDeepLinkData> responseMapping(KHQRGenerateDeepLinkResponse response) {
        KHQRResponse<KHQRDeepLinkData> baseResponse;
        if (response.getResponseCode() == SUCCESS_CODE) {
            baseResponse = responseParser(response);
        } else {
            switch (response.getErrorCode()) {
                case ApiErrorCode.REQUIRED_FIELD_MISSING: {
                    response.setErrorCode(KHQRErrorCode.INVALID_DEEP_LINK_SOURCE_INFO);
                    response.setResponseMessage(KHQRErrorCode.errorCodeMap.get(KHQRErrorCode.INVALID_DEEP_LINK_SOURCE_INFO));
                    baseResponse = responseParser(response);
                    break;
                }

                case ApiErrorCode.DEEP_LINK_GENERATE_FAILED: {
                    response.setErrorCode(KHQRErrorCode.INTERNAL_SERVER_ERROR);
                    response.setResponseMessage(KHQRErrorCode.errorCodeMap.get(KHQRErrorCode.INTERNAL_SERVER_ERROR));
                    baseResponse = responseParser(response);
                    break;
                }
                default: {
                    baseResponse = responseParser(response);
                }
            }
        }
        return baseResponse;
    }

    private KHQRResponse<KHQRDeepLinkData> responseParser(KHQRGenerateDeepLinkResponse response) {

        KHQRResponse<KHQRDeepLinkData> mResponse = new KHQRResponse<>();
        if (response != null) {
            KHQRStatus khqrStatus = new KHQRStatus();
            khqrStatus.setCode(response.getResponseCode());
            khqrStatus.setErrorCode(response.getErrorCode());
            khqrStatus.setMessage(response.getResponseMessage());
            mResponse.setKHQRStatus(khqrStatus);

            if (response.getData() != null && !KHQRValidation.isEmpty(response.getData().getShortLink())) {
                KHQRDeepLinkData khqrDeepLinkData = new KHQRDeepLinkData();
                khqrDeepLinkData.setShortLink(response.getData().getShortLink());
                mResponse.setData(khqrDeepLinkData);
            }
        }
        return mResponse;
    }

    public static KHQRResponse<KHQRDeepLinkData> responseError(int errorCode) {
        KHQRResponse<KHQRDeepLinkData> response = new KHQRResponse<>();
        KHQRStatus status = new KHQRStatus();
        status.setErrorCode(errorCode);
        status.setCode(ERROR_CODE);
        status.setMessage(KHQRErrorCode.errorCodeMap.get(errorCode));
        response.setKHQRStatus(status);
        return response;
    }

    private static OkHttpClient getUnsafeOkHttpClient() {
        try {
            // Create a trust manager that does not validate certificate chains
            final TrustManager[] trustAllCerts = new TrustManager[]{
                    new X509TrustManager() {
                        @Override
                        public void checkClientTrusted(java.security.cert.X509Certificate[] chain, String authType) {
                        }

                        @Override
                        public void checkServerTrusted(java.security.cert.X509Certificate[] chain, String authType) {
                        }

                        @Override
                        public java.security.cert.X509Certificate[] getAcceptedIssuers() {
                            return new java.security.cert.X509Certificate[]{};
                        }
                    }
            };

            // Install the all-trusting trust manager
            final SSLContext sslContext = SSLContext.getInstance("SSL");
            sslContext.init(null, trustAllCerts, new java.security.SecureRandom());
            // Create an ssl socket factory with our all-trusting manager
            final SSLSocketFactory sslSocketFactory = sslContext.getSocketFactory();

            OkHttpClient.Builder builder = new OkHttpClient.Builder();
            builder.sslSocketFactory(sslSocketFactory, (X509TrustManager) trustAllCerts[0]);
            builder.hostnameVerifier((hostname, session) -> true);

            return builder.build();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }
}

