package kh.org.nbc.bakong_khqr.presenter;

import kh.org.nbc.bakong_khqr.core.TagLengthString;
import kh.org.nbc.bakong_khqr.exception.KHQRException;
import kh.org.nbc.bakong_khqr.model.*;
import kh.org.nbc.bakong_khqr.utils.NumberUtils;
import kh.org.nbc.bakong_khqr.utils.StringUtils;

import java.util.HashMap;
import java.util.Objects;

import static kh.org.nbc.bakong_khqr.exception.KHQRException.throwCustomerException;
import static kh.org.nbc.bakong_khqr.model.KHQRMerchantPresentedCodes.*;

public class MerchantPresentedDecodeMode {

    private static final int TAG_ID_COUNT = 2;
    private static final int TAG_LENGTH_WORD_COUNT = 2;
    private static String khqr;
    private static int startTag;
    private static boolean mIsRestrict;


    private String payloadFormatIndicator;//m
    private String merchantCategoryCode;//m
    private String countryCode;//m
    private String merchantName;//m
    private String merchantCity;//m
    private String transactionCurrency;//m
    private String crc;//m
    private String bakongAccountId;//m
    private String merchantType;//m
    private String billNumber;//o
    private String storeLabel;//o
    private String terminalLabel;//o
    private String transactionAmount;//o
    private String pointOfInitiationMethod;//o
    private String timestamp;//o
    private String mobileNumber;//o'
    private String merchantId; //m for merchant type
    private String acquiringBank; //m for merchant type
    private String accountInformation; // o
    private KHQRCurrency currency;


    public KHQRDecodeData decode(String qr, Boolean isRestrict) throws KHQRException {
        khqr = qr;
        mIsRestrict = isRestrict;
        MerchantPresentedDecodeMode merchantPresentedDecodeMode = new MerchantPresentedDecodeMode();

        int currentIndex = 0;
        startTag = -1;

        if (mIsRestrict) {
            if (!KHQRValidation.isContainCrcTagValue(qr)) {
                throwCustomerException(KHQRErrorCode.CRC_REQUIRED);
            } else if (!KHQRValidation.isQrValidCrc(qr)) {
                throwCustomerException(KHQRErrorCode.KHQR_INVALID);
            } else {
                while (currentIndex < qr.length()) {
                    TagLengthString tagLengthString = getSpecificTagValue(qr, currentIndex);
                    //check tag order
                    //checkLengthOrder(startTag, tagLengthString.getTag());
                    currentIndex += TAG_ID_COUNT + TAG_LENGTH_WORD_COUNT + tagLengthString.getLength();
                    merchantPresentedDecodeMode.assignValue(tagLengthString);
                }
            }
        } else {
            while (currentIndex < qr.length()) {
                TagLengthString tagLengthString = getSpecificTagValue(qr, currentIndex);
                currentIndex += TAG_ID_COUNT + TAG_LENGTH_WORD_COUNT + tagLengthString.getLength();
                merchantPresentedDecodeMode.assignValue(tagLengthString);
            }
        }
        return merchantPresentedDecodeMode.getDecodeData();
    }

    private HashMap<String, TagLengthString> decodeChildTag(String qr) throws KHQRException {
        HashMap<String, TagLengthString> tagLengthStrings = new HashMap<>();
        int currentIndex = 0;
        startTag = -1;

        if (mIsRestrict) {
            while (currentIndex < qr.length()) {
                TagLengthString tagLengthString = getSpecificTagValue(qr, currentIndex);
                //check tag order
                //checkLengthOrder(startTag, tagLengthString.getTag());
                currentIndex += TAG_ID_COUNT + TAG_LENGTH_WORD_COUNT + tagLengthString.getLength();
                tagLengthStrings.put(tagLengthString.getTag(), tagLengthString);
            }
        } else {
            while (currentIndex < qr.length()) {
                TagLengthString tagLengthString = getSpecificTagValue(qr, currentIndex);
                currentIndex += TAG_ID_COUNT + TAG_LENGTH_WORD_COUNT + tagLengthString.getLength();
                tagLengthStrings.put(tagLengthString.getTag(), tagLengthString);
            }
        }
        return tagLengthStrings;
    }

    private TagLengthString getSpecificTagValue(String source, int currentIndex) {
        int tagCount = currentIndex + TAG_ID_COUNT;
        int lengthCount = currentIndex + TAG_LENGTH_WORD_COUNT;

        String tag;
        String value;
        try {
            tag = source.substring(currentIndex, tagCount);
        } catch (Exception e) {
            tag = "";
        }
        try {
            int length = Integer.parseInt(source.substring(lengthCount, lengthCount + TAG_LENGTH_WORD_COUNT));

            int dataStartLength = currentIndex + TAG_ID_COUNT + TAG_LENGTH_WORD_COUNT;
            int dataLength = currentIndex + TAG_LENGTH_WORD_COUNT + TAG_ID_COUNT + length;
            value = source.substring(dataStartLength, dataLength);
        } catch (Exception e) {
            value = "";
        }

        return new TagLengthString(tag, value);
    }

    private void assignValue(TagLengthString tagLengthString) throws KHQRException {

        String tag = tagLengthString.getTag();
        String value = tagLengthString.getValue();

        switch (tag) {
            case PAYLOAD_FORMAT_INDICATOR: {
                setPayloadFormatIndicator(value);
                break;
            }
            case MERCHANT_ACCOUNT_INFORMATION_INDIVIDUAL:
            case MERCHANT_ACCOUNT_INFORMATION_MERCHANT: {
                HashMap<String, TagLengthString> internalTagLengthString;
                internalTagLengthString = decodeChildTag(value);

                if (internalTagLengthString.get(MERCHANT_ACCOUNT_INFORMATION_GLOBALLY_UNIQUE_IDENTIFIER) != null) {
                    String accountId = Objects.requireNonNull(internalTagLengthString.get(MERCHANT_ACCOUNT_INFORMATION_GLOBALLY_UNIQUE_IDENTIFIER)).getValue();
                    setBakongAccountId(accountId);
                }
                if (internalTagLengthString.get(MERCHANT_ACCOUNT_INFORMATION_MERCHANT_ID) != null) {
                    String merchantInfo = Objects.requireNonNull(internalTagLengthString.get(MERCHANT_ACCOUNT_INFORMATION_MERCHANT_ID)).getValue();
                    if (tag.equals(MERCHANT_ACCOUNT_INFORMATION_MERCHANT)) {
                        setMerchantId(merchantInfo);
                    } else {
                        setAccountInformation(merchantInfo);
                    }
                }
                if (internalTagLengthString.get(MERCHANT_ACCOUNT_INFORMATION_ACQUIRING_BANK) != null) {
                    String acquiringBank = Objects.requireNonNull(internalTagLengthString.get(MERCHANT_ACCOUNT_INFORMATION_ACQUIRING_BANK)).getValue();
                    setAcquiringBank(acquiringBank);
                }

                setMerchantType(tag);
                break;
            }
            case MERCHANT_CATEGORY_CODE: {
                setMerchantCategoryCode(value);
                break;
            }
            case COUNTRY_CODE: {
                setCountryCode(value);
                break;
            }
            case MERCHANT_NAME: {
                setMerchantName(value);
                break;
            }
            case MERCHANT_CITY: {
                setMerchantCity(value);
                break;
            }
            case TRANSACTION_CURRENCY: {
                setTransactionCurrency(value);
                break;
            }
            case CRC: {
                setCrc(value);
                break;
            }
            case ADDITIONAL_DATA_FIELD_TEMPLATE: {
                HashMap<String, TagLengthString> internalTagLengthString;

                internalTagLengthString = decodeChildTag(value);

                if (internalTagLengthString.get(ADDITIONAL_DATA_FIELD_BILL_NUMBER) != null) {
                    String billNumber = Objects.requireNonNull(internalTagLengthString.get(ADDITIONAL_DATA_FIELD_BILL_NUMBER)).getValue();
                    setBillNumber(billNumber);
                }
                if (internalTagLengthString.get(ADDITIONAL_DATA_FIELD_STORE_LABEL) != null) {
                    String storeLabel = Objects.requireNonNull(internalTagLengthString.get(ADDITIONAL_DATA_FIELD_STORE_LABEL)).getValue();
                    setStoreLabel(storeLabel);
                }
                if (internalTagLengthString.get(ADDITIONAL_DATA_FIELD_TERMINAL_LABEL) != null) {
                    String terminalLabel = Objects.requireNonNull(internalTagLengthString.get(ADDITIONAL_DATA_FIELD_TERMINAL_LABEL)).getValue();
                    setTerminalLabel(terminalLabel);
                }
                if (internalTagLengthString.get(ADDITIONAL_DATA_FIELD_MOBILE_NUMBER) != null) {
                    String mobileNumber = Objects.requireNonNull(internalTagLengthString.get(ADDITIONAL_DATA_FIELD_MOBILE_NUMBER)).getValue();
                    setMobileNumber(mobileNumber);
                }
                break;
            }
            case RFU_FOR_KHQR: {
                TagLengthString tagLengthStringRfu = getSpecificTagValue(value, 0);
                if (tagLengthStringRfu.getTag().equals(TIMESTAMP)) {
                    setTimestamp(tagLengthStringRfu.getValue());
                }
                break;
            }
            case POINT_OF_INITIATION_METHOD: {
                setPointOfInitiationMethod(value);
                break;
            }
            case TRANSACTION_AMOUNT: {
                setTransactionAmount(value);
                break;
            }
        }
    }

    private KHQRDecodeData getDecodeData() throws KHQRException {
        KHQRDecodeData khqrDecodeData = new KHQRDecodeData();
        khqrDecodeData.setCrc(mIsRestrict ? getCrc() : crc);
        khqrDecodeData.setPayloadFormatIndicator(mIsRestrict ? getPayloadFormatIndicator() : payloadFormatIndicator);
        khqrDecodeData.setPointOfInitiationMethod(mIsRestrict ? getPointOfInitiationMethod() : pointOfInitiationMethod);
        khqrDecodeData.setMerchantType(mIsRestrict ? getMerchantType() : merchantType);
        khqrDecodeData.setBakongAccountID(mIsRestrict ? getBakongAccountId() : bakongAccountId);
        khqrDecodeData.setMerchantId(mIsRestrict ? getMerchantId() : merchantId);
        khqrDecodeData.setAccountInformation(mIsRestrict ? getAccountInformation() : accountInformation);
        khqrDecodeData.setAcquiringBank(mIsRestrict ? getAcquiringBank() : acquiringBank);
        khqrDecodeData.setMerchantCategoryCode(mIsRestrict ? getMerchantCategoryCode() : merchantCategoryCode);
        khqrDecodeData.setTransactionCurrency(mIsRestrict ? getTransactionCurrency() : transactionCurrency);
        khqrDecodeData.setTransactionAmount(mIsRestrict ? getTransactionAmount(currency) : transactionAmount);
        khqrDecodeData.setCountryCode(mIsRestrict ? getCountryCode() : countryCode);
        khqrDecodeData.setMerchantName(mIsRestrict ? getMerchantName() : merchantName);
        khqrDecodeData.setMerchantCity(mIsRestrict ? getMerchantCity() : merchantCity);
        khqrDecodeData.setBillNumber(mIsRestrict ? getBillNumber() : billNumber);
        khqrDecodeData.setMobileNumber(mIsRestrict ? getMobileNumber() : mobileNumber);
        khqrDecodeData.setStoreLabel(mIsRestrict ? getStoreLabel() : storeLabel);
        khqrDecodeData.setTerminalLabel(mIsRestrict ? getTerminalLabel() : terminalLabel);
        khqrDecodeData.setTimestamp(mIsRestrict ? getTimestamp() : timestamp);
        return khqrDecodeData;
    }

    private String getTimestamp() {
        return timestamp;
    }

    private void setTimestamp(String timestamp) {
        this.timestamp = timestamp;
    }

    private String getMerchantType() throws KHQRException {
        if (StringUtils.isBlank(merchantType))
            throwCustomerException(KHQRErrorCode.MERCHANT_TYPE_REQUIRED);
        return merchantType;
    }

    private void setMerchantType(String merchantType) {
        this.merchantType = merchantType;
    }

    private String getPayloadFormatIndicator() throws KHQRException {
        if (StringUtils.isBlank(payloadFormatIndicator)) {
            throwCustomerException(KHQRErrorCode.PAYLOAD_FORMAT_INDICATOR_REQUIRED);
        } else if (KHQRValidation.isPayLoadFormatIndicatorLengthInvalid(payloadFormatIndicator)) {
            throwCustomerException(KHQRErrorCode.PAYLOAD_FORMAT_INDICATOR_LENGTH_INVALID);
        }
        return payloadFormatIndicator;
    }

    private void setPayloadFormatIndicator(String payloadFormatIndicator) {
        this.payloadFormatIndicator = payloadFormatIndicator;
    }

    private String getMerchantCategoryCode() throws KHQRException {
        if (StringUtils.isBlank(merchantCategoryCode))
            throwCustomerException(KHQRErrorCode.MERCHANT_CATEGORY_CODE_REQUIRED);
        else if (KHQRValidation.isMerchantCategoryCodeLengthInvalid(merchantCategoryCode)) {
            throwCustomerException(KHQRErrorCode.MERCHANT_CATEGORY_CODE_LENGTH_INVALID);
        }
        return merchantCategoryCode;
    }

    private void setMerchantCategoryCode(String merchantCategoryCode) {
        this.merchantCategoryCode = merchantCategoryCode;
    }

    private String getCountryCode() throws KHQRException {
        if (StringUtils.isBlank(countryCode)) {
            throwCustomerException(KHQRErrorCode.COUNTRY_CODE_REQUIRED);
        } else if (KHQRValidation.isCountryCodeLengthInvalid(countryCode)) {
            throwCustomerException(KHQRErrorCode.COUNTRY_CODE_LENGTH_INVALID);
        }
        return countryCode;
    }

    private void setCountryCode(String countryCode) {
        this.countryCode = countryCode;
    }

    private String getMerchantName() throws KHQRException {
        if (StringUtils.isBlank(merchantName)) {
            throwCustomerException(KHQRErrorCode.MERCHANT_NAME_REQUIRED);
        } else if (KHQRValidation.isMerchantNameLengthInvalid(merchantName)) {
            throwCustomerException(KHQRErrorCode.MERCHANT_NAME_LENGHT_INVALID);
        }
        return merchantName;
    }

    private void setMerchantName(String merchantName) {
        this.merchantName = merchantName;
    }

    private String getMerchantCity() throws KHQRException {
        if (StringUtils.isBlank(merchantCity)) {
            throwCustomerException(KHQRErrorCode.MERCHANT_CITY_REQUIRED);
        } else if (KHQRValidation.isMerchantCityLengthInvalid(merchantCity)) {
            throwCustomerException(KHQRErrorCode.MERCHANT_CITY_LENGTH_INVALID);
        }
        return merchantCity;
    }

    private String getAcquiringBank() throws KHQRException {
        if (acquiringBank != null && KHQRValidation.isAcquiringBankLengthInvalid(acquiringBank)) {
            throwCustomerException(KHQRErrorCode.ACQUIRING_BANK_LENGTH_INVALID);
        } else return acquiringBank;
        return null;
    }

    private String getMerchantId() throws KHQRException {
        if (merchantType != null && merchantType.equals(KHQRMerchantType.MERCHANT.getType())) {
            if (StringUtils.isBlank(merchantId)) {
                throwCustomerException(KHQRErrorCode.MERCHANT_ID_REQUIRED);
            } else if (KHQRValidation.isAcquiringBankLengthInvalid(merchantId)) {
                throwCustomerException(KHQRErrorCode.MERCHANT_ID_LENGTH_INVALID);
            } else return merchantId;
        }
        return null;
    }

    private String getAccountInformation() throws KHQRException {
        if (accountInformation != null && KHQRValidation.isAccountInformationLengthInvalid(accountInformation)) {
            throwCustomerException(KHQRErrorCode.ACCOUNT_INFORMATION_LENGTH_INVALID);
        } else return accountInformation;
        return null;
    }

    private void setMerchantCity(String merchantCity) {
        this.merchantCity = merchantCity;
    }

    private String getTransactionCurrency() throws KHQRException {
        if (StringUtils.isBlank(transactionCurrency)) {
            throwCustomerException(KHQRErrorCode.CURRENCY_TYPE_REQUIRED);
        } else if (KHQRValidation.isTransactionCurrencyLengthInvalid(transactionCurrency)) {
            throwCustomerException(KHQRErrorCode.TRANSACTION_CURRENCY_LENGTH_INVALID);
        }
        if (!KHQRCurrency.USD.getValue().equals(transactionCurrency) && !KHQRCurrency.KHR.getValue().equals(transactionCurrency)) {
            throwCustomerException(KHQRErrorCode.UNSUPPORTED_TRANSACTION_CURRENCY);
        }
        return transactionCurrency;
    }

    private void setTransactionCurrency(String transactionCurrency) {
        if (transactionCurrency.equals(KHQRCurrency.KHR.getValue())) {
            currency = KHQRCurrency.KHR;
        } else if (transactionCurrency.equals(KHQRCurrency.USD.getValue())) {
            currency = KHQRCurrency.USD;
        }

        this.transactionCurrency = transactionCurrency;
    }

    private String getCrc() throws KHQRException {
        if (StringUtils.isBlank(crc)) {
            throwCustomerException(KHQRErrorCode.CRC_REQUIRED);
        } else if (KHQRValidation.isCrcLengthInvalid(crc)) {
            throwCustomerException(KHQRErrorCode.CRC_LENGTH_INVALID);
        } else if (!KHQRValidation.isQrValidCrc(khqr)) {
            throwCustomerException(KHQRErrorCode.KHQR_INVALID);
        }
        return crc;
    }

    private void setCrc(String crc) {
        this.crc = crc;
    }

    private String getBillNumber() throws KHQRException {
        if (KHQRValidation.isBillNumberLengthInvalid(billNumber)) {
            throwCustomerException(KHQRErrorCode.BILL_NUMBER_LENGTH_INVALID);
        }
        return billNumber;
    }

    private String getMobileNumber() throws KHQRException {
        if (KHQRValidation.isMobileNumberLengthInvalid(mobileNumber)) {
            throwCustomerException(KHQRErrorCode.MOBILE_NUMBER_LENGTH_INVALID);
        }
        return mobileNumber;
    }

    private void setBillNumber(String billNumber) {
        this.billNumber = billNumber;
    }

    private String getStoreLabel() throws KHQRException {
        if (KHQRValidation.isStoreLabelLengthInvalid(storeLabel)) {
            throwCustomerException(KHQRErrorCode.STORE_LABEL_LENGTH_INVALID);
        }
        return storeLabel;
    }

    private void setStoreLabel(String storeLabel) {
        this.storeLabel = storeLabel;
    }

    private String getTerminalLabel() throws KHQRException {
        if (KHQRValidation.isTerminalLabelLengthInvalid(terminalLabel)) {
            throwCustomerException(KHQRErrorCode.TERMINAL_LABEL_LENGTH_INVALID);
        }
        return terminalLabel;
    }

    private void setTerminalLabel(String terminalLabel) {
        this.terminalLabel = terminalLabel;
    }

    private void setMobileNumber(String mobileNumber) {
        this.mobileNumber = mobileNumber;
    }

    private String getTransactionAmount(KHQRCurrency currency) throws KHQRException {
        if (transactionAmount == null) {
            return null;
        } else if (!NumberUtils.isCreatable(transactionAmount) || KHQRValidation.isAmountInvalid(Double.parseDouble(transactionAmount), currency)) {
            throwCustomerException(KHQRErrorCode.TRANSACTION_AMOUNT_INVALID);
        }
        return transactionAmount;
    }

    private void setTransactionAmount(String transactionAmount) {
        this.transactionAmount = transactionAmount;
    }

    private String getPointOfInitiationMethod() throws KHQRException {
        if (StringUtils.isNotBlank(pointOfInitiationMethod) && pointOfInitiationMethod.length() > 2) {
            throwCustomerException(KHQRErrorCode.POINT_OF_INITIATION_METHOD_LENGTH_INVALID);
        }
        return pointOfInitiationMethod;
    }

    private void setPointOfInitiationMethod(String pointOfInitiationMethod) {
        this.pointOfInitiationMethod = pointOfInitiationMethod;
    }

    private String getBakongAccountId() throws KHQRException {
        if (StringUtils.isBlank(bakongAccountId)) {
            throwCustomerException(KHQRErrorCode.BAKONG_ACCOUNT_ID_REQUIRED);
        } else if (KHQRValidation.isAccountIdLengthInvalid(bakongAccountId)) {
            throwCustomerException(KHQRErrorCode.BAKONG_ACCOUNT_ID_LENGTH_INVALID);
        } else if (KHQRValidation.isAccountIdInvalidFormat(bakongAccountId)) {
            throwCustomerException(KHQRErrorCode.BAKONG_ACCOUNT_ID_INVALID);
        }
        return bakongAccountId;
    }

    private void setBakongAccountId(String bakongAccountId) {
        this.bakongAccountId = bakongAccountId;
    }

    private void setMerchantId(String merchantId) {
        this.merchantId = merchantId;
    }

    private void setAcquiringBank(String acquiringBank) {
        this.acquiringBank = acquiringBank;
    }

    private void setAccountInformation(String accountInformation) {
        this.accountInformation = accountInformation;
    }

//    private static void checkLengthOrder(int oldTag, String newTag) throws KHQRException {
//        if (NumberUtils.isCreatable(newTag)) {
//            int mNewTag = Integer.parseInt(newTag);
//            if (oldTag >= mNewTag && mNewTag != Integer.parseInt(CRC)) {
//                throwCustomerException(KHQRErrorCode.TAG_NOT_IN_ORDER);
//            }
//            startTag = mNewTag;
//        }
//    }


}