/*
 * Decompiled with CFR 0.152.
 */
package life.expert.value.numeric.amount;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Objects;
import life.expert.common.async.LogUtils;
import life.expert.value.numeric.amount.Quantity;
import life.expert.value.numeric.context.AmountContext;
import life.expert.value.numeric.operators.Operator;
import life.expert.value.numeric.unit.Piece;
import life.expert.value.numeric.unit.Unit;
import life.expert.value.numeric.utils.AmountParseException;
import life.expert.value.numeric.utils.DefaultNumberValue;
import life.expert.value.numeric.utils.NumberUtils;
import life.expert.value.numeric.utils.NumberValue;
import life.expert.value.numeric.utils.ValueException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class Amount
implements Quantity,
Comparable<Quantity> {
    private static final Logger logger_ = LoggerFactory.getLogger(LogUtils.class);
    private final Unit unit;
    private final long number;
    private static final int SCALE = 5;
    public static final AmountContext DEFAULT_CONTEXT = AmountContext.of(Amount.class, 19, true, 5, RoundingMode.HALF_EVEN);
    public static final AmountContext MAX_CONTEXT = AmountContext.of(Amount.class, 19, true, 5, RoundingMode.HALF_EVEN);
    public static final Amount MAX_VALUE = new Amount(Long.MAX_VALUE, Piece.builder().code("piece").build());
    private static final BigDecimal MAX_BD = MAX_VALUE.getBigDecimal();
    public static final Amount MIN_VALUE = new Amount(Long.MIN_VALUE, Piece.builder().code("piece").build());
    private static final BigDecimal MIN_BD = MIN_VALUE.getBigDecimal();

    private static final void log_(String format, Object ... arguments) {
        logger_.info(format, arguments);
    }

    private static final void logAtError_(String format, Object ... arguments) {
        logger_.error(format, arguments);
    }

    private static final void logAtWarning_(String format, Object ... arguments) {
        logger_.warn(format, arguments);
    }

    private long getInternalNumber(Number number, boolean allowInternalRounding) {
        BigDecimal bd = NumberUtils.getBigDecimal(number);
        if (!allowInternalRounding && bd.scale() > 5) {
            throw new ArithmeticException(number + " can not be represented by this class, scale > 5");
        }
        if (bd.compareTo(MIN_BD) < 0) {
            throw new ArithmeticException("Overflow: " + number + " < " + MIN_BD);
        }
        if (bd.compareTo(MAX_BD) > 0) {
            throw new ArithmeticException("Overflow: " + number + " > " + MAX_BD);
        }
        return bd.movePointRight(5).longValue();
    }

    private void checkAmountParameter(Quantity amount) {
        NumberUtils.checkAmountParameter(amount, this.unit);
        if (amount.getNumber().getScale() > 5) {
            throw new ArithmeticException("Parameter exceeds maximal scale: 5");
        }
        if (amount.getNumber().getPrecision() > MAX_BD.precision()) {
            throw new ArithmeticException("Parameter exceeds maximal precision: 5");
        }
    }

    protected void checkNumber(Number number) {
        Objects.requireNonNull(number, "Number is required.");
        if (number.longValue() > MAX_BD.longValue()) {
            throw new ArithmeticException("Value exceeds maximal value: " + MAX_BD);
        }
        BigDecimal bd = NumberUtils.getBigDecimal(number);
        if (bd.precision() > MAX_BD.precision()) {
            throw new ArithmeticException("Precision exceeds maximal precision: " + MAX_BD.precision());
        }
        if (bd.scale() > 5) {
            throw new ArithmeticException("Scale of " + bd + " exceeds maximal scale: 5");
        }
    }

    private boolean isOne(Number number) {
        BigDecimal bd = NumberUtils.getBigDecimal(number);
        try {
            return bd.scale() == 0 && bd.longValueExact() == 1L;
        }
        catch (Exception e) {
            return false;
        }
    }

    private BigDecimal getBigDecimal() {
        return BigDecimal.valueOf(this.number).movePointLeft(5);
    }

    public final boolean isLessThan(Number number) {
        this.checkNumber(number);
        return this.getBigDecimal().compareTo(NumberUtils.getBigDecimal(number)) < 0;
    }

    public final boolean isLessThanOrEqualTo(Number number) {
        this.checkNumber(number);
        return this.getBigDecimal().compareTo(NumberUtils.getBigDecimal(number)) <= 0;
    }

    public final boolean isGreaterThan(Number number) {
        this.checkNumber(number);
        return this.getBigDecimal().compareTo(NumberUtils.getBigDecimal(number)) > 0;
    }

    public final boolean isGreaterThanOrEqualTo(Number number) {
        this.checkNumber(number);
        return this.getBigDecimal().compareTo(NumberUtils.getBigDecimal(number)) >= 0;
    }

    public final boolean hasSameNumberAs(Number number) {
        this.checkNumber(number);
        try {
            return this.number == this.getInternalNumber(number, false);
        }
        catch (ArithmeticException e) {
            return false;
        }
    }

    private Amount(Number number, Unit unit, boolean allowInternalRounding) {
        Objects.requireNonNull(unit, "Unit is required.");
        this.unit = unit;
        Objects.requireNonNull(number, "Number is required.");
        this.number = this.getInternalNumber(number, allowInternalRounding);
    }

    private Amount(NumberValue numberValue, Unit unit, boolean allowInternalRounding) {
        Objects.requireNonNull(unit, "Unit is required.");
        this.unit = unit;
        Objects.requireNonNull(numberValue, "Number is required.");
        this.number = this.getInternalNumber(numberValue.numberValue(BigDecimal.class), allowInternalRounding);
    }

    private Amount(long number, Unit unit) {
        Objects.requireNonNull(unit, "Unit is required.");
        this.unit = unit;
        this.number = number;
    }

    public static Amount of(NumberValue numberBinding, Unit unit) {
        return new Amount(numberBinding, unit, false);
    }

    public static Amount of(Number number, Unit unit) {
        return new Amount(number, unit, false);
    }

    public static Amount of(Number number, String unitCode) {
        Piece unit = Piece.of(unitCode);
        return Amount.of(number, (Unit)unit);
    }

    public static Amount of(Number number) {
        Piece unit = Piece.of("piece");
        return Amount.of(number, (Unit)unit);
    }

    public static Amount zero(Unit unit) {
        return Amount.of((Number)BigDecimal.ZERO, unit);
    }

    public static Amount ofMinor(Unit unit, long amountMinor) {
        return Amount.ofMinor(unit, amountMinor, unit.getDefaultFractionDigits());
    }

    public static Amount ofMinor(Unit unit, long amountMinor, int factionDigits) {
        if (factionDigits < 0) {
            throw new IllegalArgumentException("The factionDigits cannot be negative");
        }
        return Amount.of((Number)BigDecimal.valueOf(amountMinor, factionDigits), unit);
    }

    public static Amount from(Quantity amount) {
        if (Amount.class.isInstance(amount)) {
            return (Amount)Amount.class.cast(amount);
        }
        return new Amount(amount.getNumber(), amount.getUnit(), false);
    }

    public static Amount parse(CharSequence text) {
        String[] array = Objects.requireNonNull(text).toString().split(" ");
        if (array.length != 2) {
            throw new AmountParseException("An error happened when try to parse the Amount.", text, 0);
        }
        String parsed_unit = array[0];
        BigDecimal number = new BigDecimal(array[1]);
        return Amount.of((Number)number, (Unit)Piece.of(parsed_unit));
    }

    @Override
    public Unit getUnit() {
        return this.unit;
    }

    @Override
    public AmountContext getContext() {
        return DEFAULT_CONTEXT;
    }

    @Override
    public int compareTo(Quantity o) {
        Objects.requireNonNull(o);
        int compare = this.getUnit().getCode().compareTo(o.getUnit().getCode());
        if (compare == 0) {
            compare = this.getNumber().numberValue(BigDecimal.class).compareTo(o.getNumber().numberValue(BigDecimal.class));
        }
        return compare;
    }

    public int hashCode() {
        return Objects.hash(this.unit, this.number);
    }

    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if (obj instanceof Amount) {
            Amount other = (Amount)obj;
            return Objects.equals(this.unit, other.unit) && Objects.equals(this.number, other.number);
        }
        return false;
    }

    @Override
    public Amount abs() {
        if (this.isPositiveOrZero()) {
            return this;
        }
        return this.negate();
    }

    @Override
    public Amount add(Quantity amount) {
        this.checkAmountParameter(amount);
        if (amount.isZero()) {
            return this;
        }
        return new Amount(Math.addExact(this.number, this.getInternalNumber(amount.getNumber(), false)), this.getUnit());
    }

    @Override
    public Amount divide(Number divisor) {
        if (NumberUtils.isInfinityAndNotNaN(divisor)) {
            return new Amount(0L, this.getUnit());
        }
        this.checkNumber(divisor);
        if (this.isOne(divisor)) {
            return this;
        }
        return new Amount(Math.round((double)this.number / divisor.doubleValue()), this.getUnit());
    }

    public Amount[] divideAndRemainder(Number divisor) {
        if (NumberUtils.isInfinityAndNotNaN(divisor)) {
            Amount zero = new Amount(0L, this.getUnit());
            return new Amount[]{zero, zero};
        }
        this.checkNumber(divisor);
        BigDecimal div = NumberUtils.getBigDecimal(divisor);
        BigDecimal[] res = this.getBigDecimal().divideAndRemainder(div);
        return new Amount[]{new Amount(res[0], this.getUnit(), true), new Amount(res[1], this.getUnit(), true)};
    }

    @Override
    public Amount divideToIntegralValue(Number divisor) {
        if (NumberUtils.isInfinityAndNotNaN(divisor)) {
            return new Amount(0L, this.getUnit());
        }
        this.checkNumber(divisor);
        if (this.isOne(divisor)) {
            return this;
        }
        BigDecimal div = NumberUtils.getBigDecimal(divisor);
        return new Amount(this.getBigDecimal().divideToIntegralValue(div), this.getUnit(), false);
    }

    @Override
    public Amount multiply(Number multiplicand) {
        NumberUtils.checkNoInfinityOrNaN(multiplicand);
        this.checkNumber(multiplicand);
        if (this.isOne(multiplicand)) {
            return this;
        }
        return new Amount(Math.multiplyExact(this.number, this.getInternalNumber(multiplicand, false)) / 100000L, this.getUnit());
    }

    @Override
    public Amount negate() {
        return new Amount(Math.multiplyExact(this.number, -1), this.getUnit());
    }

    @Override
    public Amount plus() {
        return this;
    }

    @Override
    public Amount subtract(Quantity amount) {
        this.checkAmountParameter(amount);
        if (amount.isZero()) {
            return this;
        }
        return new Amount(Math.subtractExact(this.number, this.getInternalNumber(amount.getNumber(), false)), this.getUnit());
    }

    @Override
    public Amount remainder(Number divisor) {
        this.checkNumber(divisor);
        return new Amount(this.number % this.getInternalNumber(divisor, false), this.getUnit());
    }

    @Override
    public Amount scaleByPowerOfTen(int power) {
        return new Amount(this.getNumber().numberValue(BigDecimal.class).scaleByPowerOfTen(power), this.getUnit(), true);
    }

    @Override
    public boolean isZero() {
        return this.number == 0L;
    }

    @Override
    public boolean isPositive() {
        return this.number > 0L;
    }

    @Override
    public boolean isPositiveOrZero() {
        return this.number >= 0L;
    }

    @Override
    public boolean isNegative() {
        return this.number < 0L;
    }

    @Override
    public boolean isNegativeOrZero() {
        return this.number <= 0L;
    }

    public int getScale() {
        return 5;
    }

    public int getPrecision() {
        return this.getNumber().numberValue(BigDecimal.class).precision();
    }

    @Override
    public int signum() {
        if (this.number < 0L) {
            return -1;
        }
        if (this.number == 0L) {
            return 0;
        }
        return 1;
    }

    @Override
    public boolean isLessThan(Quantity amount) {
        this.checkAmountParameter(amount);
        return this.getBigDecimal().compareTo(amount.getNumber().numberValue(BigDecimal.class)) < 0;
    }

    @Override
    public boolean isLessThanOrEqualTo(Quantity amount) {
        this.checkAmountParameter(amount);
        return this.getBigDecimal().compareTo(amount.getNumber().numberValue(BigDecimal.class)) <= 0;
    }

    @Override
    public boolean isGreaterThan(Quantity amount) {
        this.checkAmountParameter(amount);
        return this.getBigDecimal().compareTo(amount.getNumber().numberValue(BigDecimal.class)) > 0;
    }

    @Override
    public boolean isGreaterThanOrEqualTo(Quantity amount) {
        this.checkAmountParameter(amount);
        return this.getBigDecimal().compareTo(amount.getNumber().numberValue(BigDecimal.class)) >= 0;
    }

    @Override
    public boolean isEqualTo(Quantity amount) {
        this.checkAmountParameter(amount);
        return this.getBigDecimal().compareTo(amount.getNumber().numberValue(BigDecimal.class)) == 0;
    }

    @Override
    public NumberValue getNumber() {
        return new DefaultNumberValue(this.getBigDecimal());
    }

    public String toString() {
        return this.unit.toString() + " " + this.getBigDecimal();
    }

    @Override
    public Amount with(Operator operator) {
        Objects.requireNonNull(operator);
        try {
            return (Amount)Amount.class.cast(operator.apply(this));
        }
        catch (ArithmeticException e) {
            throw e;
        }
        catch (Exception e) {
            throw new ValueException("Operator failed: " + operator, e);
        }
    }

    @Override
    public Amount multiply(double multiplicand) {
        NumberUtils.checkNoInfinityOrNaN(multiplicand);
        if (multiplicand == 1.0) {
            return this;
        }
        if (multiplicand == 0.0) {
            return new Amount(0L, this.unit);
        }
        return new Amount(Math.round((double)this.number * multiplicand), this.unit);
    }

    @Override
    public Amount divide(long divisor) {
        if (divisor == 1L) {
            return this;
        }
        return new Amount(this.number / divisor, this.unit);
    }

    @Override
    public Amount divide(double divisor) {
        if (NumberUtils.isInfinityAndNotNaN(divisor)) {
            return new Amount(0L, this.getUnit());
        }
        if (divisor == 1.0) {
            return this;
        }
        return new Amount(Math.round((double)this.number / divisor), this.getUnit());
    }

    @Override
    public Amount remainder(long divisor) {
        return this.remainder(BigDecimal.valueOf(divisor));
    }

    @Override
    public Amount remainder(double divisor) {
        if (NumberUtils.isInfinityAndNotNaN(divisor)) {
            return new Amount(0L, this.getUnit());
        }
        return this.remainder(new BigDecimal(String.valueOf(divisor)));
    }

    public Amount[] divideAndRemainder(long divisor) {
        return this.divideAndRemainder(BigDecimal.valueOf(divisor));
    }

    public Amount[] divideAndRemainder(double divisor) {
        if (NumberUtils.isInfinityAndNotNaN(divisor)) {
            Amount zero = new Amount(0L, this.getUnit());
            return new Amount[]{zero, zero};
        }
        if (Double.isNaN(divisor)) {
            throw new ArithmeticException("Not a number: NaN.");
        }
        return this.divideAndRemainder(new BigDecimal(String.valueOf(divisor)));
    }

    @Override
    public Amount stripTrailingZeros() {
        return this;
    }

    @Override
    public Amount multiply(long multiplicand) {
        if (multiplicand == 1L) {
            return this;
        }
        if (multiplicand == 0L) {
            return new Amount(0L, this.unit);
        }
        return new Amount(Math.multiplyExact(multiplicand, this.number), this.unit);
    }

    @Override
    public Amount divideToIntegralValue(long divisor) {
        if (divisor == 1L) {
            return this;
        }
        return this.divideToIntegralValue(NumberUtils.getBigDecimal(divisor));
    }

    @Override
    public Amount divideToIntegralValue(double divisor) {
        if (NumberUtils.isInfinityAndNotNaN(divisor)) {
            return new Amount(0L, this.getUnit());
        }
        if (divisor == 1.0) {
            return this;
        }
        return this.divideToIntegralValue(NumberUtils.getBigDecimal(divisor));
    }
}

