/*
 * $Id$
 */

package org.apache.maven.model.io.stax;

  //---------------------------------/
 //- Imported classes and packages -/
//---------------------------------/

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.text.DateFormat;
import java.text.ParsePosition;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.xml.stream.*;
import org.apache.maven.model.Branch;
import org.apache.maven.model.Build;
import org.apache.maven.model.BuildBase;
import org.apache.maven.model.Contributor;
import org.apache.maven.model.Dependency;
import org.apache.maven.model.Developer;
import org.apache.maven.model.FileSet;
import org.apache.maven.model.License;
import org.apache.maven.model.MailingList;
import org.apache.maven.model.Model;
import org.apache.maven.model.ModelBase;
import org.apache.maven.model.Organization;
import org.apache.maven.model.PackageGroup;
import org.apache.maven.model.PatternSet;
import org.apache.maven.model.PluginConfiguration;
import org.apache.maven.model.PluginContainer;
import org.apache.maven.model.Repository;
import org.apache.maven.model.Resource;
import org.apache.maven.model.SourceModification;
import org.apache.maven.model.UnitTest;
import org.apache.maven.model.Version;
import org.codehaus.plexus.util.xml.Xpp3Dom;

/**
 * Class MavenStaxReader.
 * 
 * @version $Revision$ $Date$
 */
public class MavenStaxReader {


      //-----------/
     //- Methods -/
    //-----------/

    /**
     * Method buildDom
     * 
     * @param xmlStreamReader
     */
    public Xpp3Dom buildDom(XMLStreamReader xmlStreamReader)
        throws XMLStreamException
    {
        java.util.List elements = new java.util.ArrayList();
        java.util.List values = new java.util.ArrayList();
        int eventType = xmlStreamReader.getEventType();
        while ( xmlStreamReader.hasNext() )
        {
            if ( eventType == XMLStreamConstants.START_ELEMENT )
            {
                String rawName = xmlStreamReader.getLocalName();
                Xpp3Dom childConfiguration = new Xpp3Dom( rawName );
                int depth = elements.size();
                if ( depth > 0 )
                {
                    Xpp3Dom parent = (Xpp3Dom) elements.get( depth - 1 );
                    parent.addChild( childConfiguration );
                }
                elements.add( childConfiguration );
                if ( xmlStreamReader.isEndElement() )
                {
                    values.add( null );
                }
                else
                {
                    values.add( new StringBuffer() );
                }
                int attributesSize = xmlStreamReader.getAttributeCount();
                for ( int i = 0; i < attributesSize; i++ )
                {
                    String name = xmlStreamReader.getAttributeLocalName( i );
                    String value = xmlStreamReader.getAttributeValue( i );
                    childConfiguration.setAttribute( name, value );
                }
            }
            else if ( eventType == XMLStreamConstants.CHARACTERS )
            {
                int depth = values.size() - 1;
                StringBuffer valueBuffer = (StringBuffer) values.get( depth );
                String text = xmlStreamReader.getText();
                text = text.trim();
                valueBuffer.append( text );
            }
            else if ( eventType == XMLStreamConstants.END_ELEMENT )
            {
                int depth = elements.size() - 1;
                Xpp3Dom finishedConfiguration = (Xpp3Dom) elements.remove( depth );
                // this Object could be null if it is a singleton tag
                Object accumulatedValue = values.remove( depth );
                if ( finishedConfiguration.getChildCount() == 0 )
                {
                    if ( accumulatedValue == null )
                    {
                        finishedConfiguration.setValue( null );
                    }
                    else
                    {
                        finishedConfiguration.setValue( accumulatedValue.toString() );
                    }
                }
                if ( depth == 0 )
                {
                    return finishedConfiguration;
                }
            }
            eventType = xmlStreamReader.next();
        }
        throw new IllegalStateException( "End of document found before returning to 0 depth" );
    } //-- Xpp3Dom buildDom(XMLStreamReader) 

    /**
     * Method getBooleanValue
     * 
     * @param s
     * @param xmlStreamReader
     * @param attribute
     */
    public boolean getBooleanValue(String s, String attribute, XMLStreamReader xmlStreamReader)
        throws XMLStreamException
    {
        if ( s != null )
        {
            return Boolean.valueOf( s ).booleanValue();
        }
        return false;
    } //-- boolean getBooleanValue(String, String, XMLStreamReader) 

    /**
     * Method getCharacterValue
     * 
     * @param s
     * @param xmlStreamReader
     * @param attribute
     */
    public char getCharacterValue(String s, String attribute, XMLStreamReader xmlStreamReader)
        throws XMLStreamException
    {
        if ( s != null )
        {
            return s.charAt( 0 );
        }
        return 0;
    } //-- char getCharacterValue(String, String, XMLStreamReader) 

    /**
     * Method getDateValue
     * 
     * @param s
     * @param xmlStreamReader
     * @param dateFormat
     * @param attribute
     */
    public java.util.Date getDateValue(String s, String attribute, String dateFormat, XMLStreamReader xmlStreamReader)
        throws XMLStreamException
    {
        if ( s != null )
        {
            if ( dateFormat == null )
            {
                return new java.util.Date( Long.valueOf( s ).longValue() );
            }
            else
            {
                DateFormat dateParser = new java.text.SimpleDateFormat( dateFormat, Locale.US );
                try
                {
                    return dateParser.parse( s );
                }
                catch ( java.text.ParseException e )
                {
                    throw new XMLStreamException( e.getMessage() );
                }
            }
        }
        return null;
    } //-- java.util.Date getDateValue(String, String, String, XMLStreamReader) 

    /**
     * Method getDoubleValue
     * 
     * @param s
     * @param strict
     * @param xmlStreamReader
     * @param attribute
     */
    public double getDoubleValue(String s, String attribute, XMLStreamReader xmlStreamReader, boolean strict)
        throws XMLStreamException
    {
        if ( s != null )
        {
            try
            {
                return Double.valueOf( s ).doubleValue();
            }
            catch ( NumberFormatException e )
            {
                if ( strict )
                {
                    throw new XMLStreamException( "Unable to parse element '" + attribute + "', must be a floating point number but was '" + s + "'", xmlStreamReader.getLocation() );
                }
            }
        }
        return 0;
    } //-- double getDoubleValue(String, String, XMLStreamReader, boolean) 

    /**
     * Method getFloatValue
     * 
     * @param s
     * @param strict
     * @param xmlStreamReader
     * @param attribute
     */
    public float getFloatValue(String s, String attribute, XMLStreamReader xmlStreamReader, boolean strict)
        throws XMLStreamException
    {
        if ( s != null )
        {
            try
            {
                return Float.valueOf( s ).floatValue();
            }
            catch ( NumberFormatException e )
            {
                if ( strict )
                {
                    throw new XMLStreamException( "Unable to parse element '" + attribute + "', must be a floating point number but was '" + s + "'", xmlStreamReader.getLocation() );
                }
            }
        }
        return 0;
    } //-- float getFloatValue(String, String, XMLStreamReader, boolean) 

    /**
     * Method getIntegerValue
     * 
     * @param s
     * @param strict
     * @param xmlStreamReader
     * @param attribute
     */
    public int getIntegerValue(String s, String attribute, XMLStreamReader xmlStreamReader, boolean strict)
        throws XMLStreamException
    {
        if ( s != null )
        {
            try
            {
                return Integer.valueOf( s ).intValue();
            }
            catch ( NumberFormatException e )
            {
                if ( strict )
                {
                    throw new XMLStreamException( "Unable to parse element '" + attribute + "', must be an integer but was '" + s + "'", xmlStreamReader.getLocation() );
                }
            }
        }
        return 0;
    } //-- int getIntegerValue(String, String, XMLStreamReader, boolean) 

    /**
     * Method getLongValue
     * 
     * @param s
     * @param strict
     * @param xmlStreamReader
     * @param attribute
     */
    public long getLongValue(String s, String attribute, XMLStreamReader xmlStreamReader, boolean strict)
        throws XMLStreamException
    {
        if ( s != null )
        {
            try
            {
                return Long.valueOf( s ).longValue();
            }
            catch ( NumberFormatException e )
            {
                if ( strict )
                {
                    throw new XMLStreamException( "Unable to parse element '" + attribute + "', must be a long integer but was '" + s + "'", xmlStreamReader.getLocation() );
                }
            }
        }
        return 0;
    } //-- long getLongValue(String, String, XMLStreamReader, boolean) 

    /**
     * Method getRequiredAttributeValue
     * 
     * @param s
     * @param strict
     * @param xmlStreamReader
     * @param attribute
     */
    public String getRequiredAttributeValue(String s, String attribute, XMLStreamReader xmlStreamReader, boolean strict)
        throws XMLStreamException
    {
        if ( s == null )
        {
            if ( strict )
            {
                throw new XMLStreamException( "Missing required value for attribute '" + attribute + "'", xmlStreamReader.getLocation() );
            }
        }
        return s;
    } //-- String getRequiredAttributeValue(String, String, XMLStreamReader, boolean) 

    /**
     * Method getShortValue
     * 
     * @param s
     * @param strict
     * @param xmlStreamReader
     * @param attribute
     */
    public short getShortValue(String s, String attribute, XMLStreamReader xmlStreamReader, boolean strict)
        throws XMLStreamException
    {
        if ( s != null )
        {
            try
            {
                return Short.valueOf( s ).shortValue();
            }
            catch ( NumberFormatException e )
            {
                if ( strict )
                {
                    throw new XMLStreamException( "Unable to parse element '" + attribute + "', must be a short integer but was '" + s + "'", xmlStreamReader.getLocation() );
                }
            }
        }
        return 0;
    } //-- short getShortValue(String, String, XMLStreamReader, boolean) 

    /**
     * Method getTrimmedValue
     * 
     * @param s
     */
    public String getTrimmedValue(String s)
    {
        if ( s != null )
        {
            s = s.trim();
        }
        return s;
    } //-- String getTrimmedValue(String) 

    /**
     * Method parseBranch
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param xmlStreamReader
     */
    private Branch parseBranch(String tagName, XMLStreamReader xmlStreamReader, boolean strict, String encoding)
        throws IOException, XMLStreamException
    {
        Branch branch = new Branch();
        branch.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
        {
            if ( xmlStreamReader.getLocalName().equals( "tag" )  )
            {
                if ( parsed.contains( "tag" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "tag" );
                branch.setTag( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else
            {
                if ( strict )
                {
                    throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
            }
        }
        return branch;
    } //-- Branch parseBranch(String, XMLStreamReader, boolean, String) 

    /**
     * Method parseBuild
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param xmlStreamReader
     */
    private Build parseBuild(String tagName, XMLStreamReader xmlStreamReader, boolean strict, String encoding)
        throws IOException, XMLStreamException
    {
        Build build = new Build();
        build.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
        {
            if ( xmlStreamReader.getLocalName().equals( "nagEmailAddress" )  )
            {
                if ( parsed.contains( "nagEmailAddress" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "nagEmailAddress" );
                build.setNagEmailAddress( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "sourceDirectory" )  )
            {
                if ( parsed.contains( "sourceDirectory" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "sourceDirectory" );
                build.setSourceDirectory( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "unitTestSourceDirectory" )  )
            {
                if ( parsed.contains( "unitTestSourceDirectory" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "unitTestSourceDirectory" );
                build.setUnitTestSourceDirectory( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "aspectSourceDirectory" )  )
            {
                if ( parsed.contains( "aspectSourceDirectory" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "aspectSourceDirectory" );
                build.setAspectSourceDirectory( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "integrationUnitTestSourceDirectory" )  )
            {
                if ( parsed.contains( "integrationUnitTestSourceDirectory" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "integrationUnitTestSourceDirectory" );
                build.setIntegrationUnitTestSourceDirectory( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "sourceModifications" )  )
            {
                if ( parsed.contains( "sourceModifications" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "sourceModifications" );
                java.util.List sourceModifications = new java.util.ArrayList();
                build.setSourceModifications( sourceModifications );
                while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                {
                    if ( xmlStreamReader.getLocalName().equals( "sourceModification" ) )
                    {
                        build.addSourceModification( parseSourceModification( "sourceModification", xmlStreamReader, strict, encoding ) );
                    }
                    else
                    {
                        throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                }
            }
            else if ( xmlStreamReader.getLocalName().equals( "unitTest" )  )
            {
                if ( parsed.contains( "unitTest" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "unitTest" );
                build.setUnitTest( parseUnitTest( "unitTest", xmlStreamReader, strict, encoding ) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "defaultGoal" )  )
            {
                if ( parsed.contains( "defaultGoal" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "defaultGoal" );
                build.setDefaultGoal( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "resources" )  )
            {
                if ( parsed.contains( "resources" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "resources" );
                java.util.List resources = new java.util.ArrayList();
                build.setResources( resources );
                while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                {
                    if ( xmlStreamReader.getLocalName().equals( "resource" ) )
                    {
                        build.addResource( parseResource( "resource", xmlStreamReader, strict, encoding ) );
                    }
                    else
                    {
                        throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
            }
        }
        return build;
    } //-- Build parseBuild(String, XMLStreamReader, boolean, String) 

    /**
     * Method parseBuildBase
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param xmlStreamReader
     */
    private BuildBase parseBuildBase(String tagName, XMLStreamReader xmlStreamReader, boolean strict, String encoding)
        throws IOException, XMLStreamException
    {
        BuildBase buildBase = new BuildBase();
        buildBase.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
        {
            if ( xmlStreamReader.getLocalName().equals( "defaultGoal" )  )
            {
                if ( parsed.contains( "defaultGoal" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "defaultGoal" );
                buildBase.setDefaultGoal( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "resources" )  )
            {
                if ( parsed.contains( "resources" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "resources" );
                java.util.List resources = new java.util.ArrayList();
                buildBase.setResources( resources );
                while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                {
                    if ( xmlStreamReader.getLocalName().equals( "resource" ) )
                    {
                        buildBase.addResource( parseResource( "resource", xmlStreamReader, strict, encoding ) );
                    }
                    else
                    {
                        throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
            }
        }
        return buildBase;
    } //-- BuildBase parseBuildBase(String, XMLStreamReader, boolean, String) 

    /**
     * Method parseContributor
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param xmlStreamReader
     */
    private Contributor parseContributor(String tagName, XMLStreamReader xmlStreamReader, boolean strict, String encoding)
        throws IOException, XMLStreamException
    {
        Contributor contributor = new Contributor();
        contributor.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
        {
            if ( xmlStreamReader.getLocalName().equals( "name" )  )
            {
                if ( parsed.contains( "name" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "name" );
                contributor.setName( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "email" )  )
            {
                if ( parsed.contains( "email" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "email" );
                contributor.setEmail( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "url" )  )
            {
                if ( parsed.contains( "url" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "url" );
                contributor.setUrl( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "organization" ) || xmlStreamReader.getLocalName().equals( "organisation" )  )
            {
                if ( parsed.contains( "organization" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "organization" );
                contributor.setOrganization( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "organizationUrl" ) || xmlStreamReader.getLocalName().equals( "organisationUrl" )  )
            {
                if ( parsed.contains( "organizationUrl" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "organizationUrl" );
                contributor.setOrganizationUrl( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "roles" )  )
            {
                if ( parsed.contains( "roles" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "roles" );
                java.util.List roles = new java.util.ArrayList();
                contributor.setRoles( roles );
                while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                {
                    if ( xmlStreamReader.getLocalName().equals( "role" ) )
                    {
                        roles.add( getTrimmedValue( xmlStreamReader.getElementText()) );
                    }
                    else
                    {
                        throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                }
            }
            else if ( xmlStreamReader.getLocalName().equals( "timezone" )  )
            {
                if ( parsed.contains( "timezone" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "timezone" );
                contributor.setTimezone( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "properties" )  )
            {
                if ( parsed.contains( "properties" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "properties" );
                while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                {
                    String key = xmlStreamReader.getLocalName();
                    String value = xmlStreamReader.getElementText()
                    .trim()
                    ;
                    contributor.addProperty( key, value );
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
            }
        }
        return contributor;
    } //-- Contributor parseContributor(String, XMLStreamReader, boolean, String) 

    /**
     * Method parseDependency
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param xmlStreamReader
     */
    private Dependency parseDependency(String tagName, XMLStreamReader xmlStreamReader, boolean strict, String encoding)
        throws IOException, XMLStreamException
    {
        Dependency dependency = new Dependency();
        dependency.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
        {
            if ( xmlStreamReader.getLocalName().equals( "id" )  )
            {
                if ( parsed.contains( "id" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "id" );
                dependency.setId( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "groupId" )  )
            {
                if ( parsed.contains( "groupId" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "groupId" );
                dependency.setGroupId( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "artifactId" )  )
            {
                if ( parsed.contains( "artifactId" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "artifactId" );
                dependency.setArtifactId( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "version" )  )
            {
                if ( parsed.contains( "version" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "version" );
                dependency.setVersion( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "url" )  )
            {
                if ( parsed.contains( "url" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "url" );
                dependency.setUrl( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "jar" )  )
            {
                if ( parsed.contains( "jar" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "jar" );
                dependency.setJar( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "type" )  )
            {
                if ( parsed.contains( "type" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "type" );
                dependency.setType( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "properties" )  )
            {
                if ( parsed.contains( "properties" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "properties" );
                while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                {
                    String key = xmlStreamReader.getLocalName();
                    String value = xmlStreamReader.getElementText()
                    .trim()
                    ;
                    dependency.addProperty( key, value );
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
            }
        }
        return dependency;
    } //-- Dependency parseDependency(String, XMLStreamReader, boolean, String) 

    /**
     * Method parseDeveloper
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param xmlStreamReader
     */
    private Developer parseDeveloper(String tagName, XMLStreamReader xmlStreamReader, boolean strict, String encoding)
        throws IOException, XMLStreamException
    {
        Developer developer = new Developer();
        developer.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
        {
            if ( xmlStreamReader.getLocalName().equals( "id" )  )
            {
                if ( parsed.contains( "id" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "id" );
                developer.setId( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "name" )  )
            {
                if ( parsed.contains( "name" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "name" );
                developer.setName( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "email" )  )
            {
                if ( parsed.contains( "email" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "email" );
                developer.setEmail( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "url" )  )
            {
                if ( parsed.contains( "url" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "url" );
                developer.setUrl( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "organization" ) || xmlStreamReader.getLocalName().equals( "organisation" )  )
            {
                if ( parsed.contains( "organization" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "organization" );
                developer.setOrganization( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "organizationUrl" ) || xmlStreamReader.getLocalName().equals( "organisationUrl" )  )
            {
                if ( parsed.contains( "organizationUrl" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "organizationUrl" );
                developer.setOrganizationUrl( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "roles" )  )
            {
                if ( parsed.contains( "roles" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "roles" );
                java.util.List roles = new java.util.ArrayList();
                developer.setRoles( roles );
                while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                {
                    if ( xmlStreamReader.getLocalName().equals( "role" ) )
                    {
                        roles.add( getTrimmedValue( xmlStreamReader.getElementText()) );
                    }
                    else
                    {
                        throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                }
            }
            else if ( xmlStreamReader.getLocalName().equals( "timezone" )  )
            {
                if ( parsed.contains( "timezone" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "timezone" );
                developer.setTimezone( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "properties" )  )
            {
                if ( parsed.contains( "properties" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "properties" );
                while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                {
                    String key = xmlStreamReader.getLocalName();
                    String value = xmlStreamReader.getElementText()
                    .trim()
                    ;
                    developer.addProperty( key, value );
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
            }
        }
        return developer;
    } //-- Developer parseDeveloper(String, XMLStreamReader, boolean, String) 

    /**
     * Method parseFileSet
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param xmlStreamReader
     */
    private FileSet parseFileSet(String tagName, XMLStreamReader xmlStreamReader, boolean strict, String encoding)
        throws IOException, XMLStreamException
    {
        FileSet fileSet = new FileSet();
        fileSet.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
        {
            if ( xmlStreamReader.getLocalName().equals( "directory" )  )
            {
                if ( parsed.contains( "directory" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "directory" );
                fileSet.setDirectory( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "includes" )  )
            {
                if ( parsed.contains( "includes" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "includes" );
                java.util.List includes = new java.util.ArrayList();
                fileSet.setIncludes( includes );
                while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                {
                    if ( xmlStreamReader.getLocalName().equals( "include" ) )
                    {
                        includes.add( getTrimmedValue( xmlStreamReader.getElementText()) );
                    }
                    else
                    {
                        throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                }
            }
            else if ( xmlStreamReader.getLocalName().equals( "excludes" )  )
            {
                if ( parsed.contains( "excludes" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "excludes" );
                java.util.List excludes = new java.util.ArrayList();
                fileSet.setExcludes( excludes );
                while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                {
                    if ( xmlStreamReader.getLocalName().equals( "exclude" ) )
                    {
                        excludes.add( getTrimmedValue( xmlStreamReader.getElementText()) );
                    }
                    else
                    {
                        throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
            }
        }
        return fileSet;
    } //-- FileSet parseFileSet(String, XMLStreamReader, boolean, String) 

    /**
     * Method parseLicense
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param xmlStreamReader
     */
    private License parseLicense(String tagName, XMLStreamReader xmlStreamReader, boolean strict, String encoding)
        throws IOException, XMLStreamException
    {
        License license = new License();
        license.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
        {
            if ( xmlStreamReader.getLocalName().equals( "name" )  )
            {
                if ( parsed.contains( "name" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "name" );
                license.setName( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "url" )  )
            {
                if ( parsed.contains( "url" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "url" );
                license.setUrl( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "distribution" )  )
            {
                if ( parsed.contains( "distribution" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "distribution" );
                license.setDistribution( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "comments" )  )
            {
                if ( parsed.contains( "comments" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "comments" );
                license.setComments( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else
            {
                if ( strict )
                {
                    throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
            }
        }
        return license;
    } //-- License parseLicense(String, XMLStreamReader, boolean, String) 

    /**
     * Method parseMailingList
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param xmlStreamReader
     */
    private MailingList parseMailingList(String tagName, XMLStreamReader xmlStreamReader, boolean strict, String encoding)
        throws IOException, XMLStreamException
    {
        MailingList mailingList = new MailingList();
        mailingList.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
        {
            if ( xmlStreamReader.getLocalName().equals( "name" )  )
            {
                if ( parsed.contains( "name" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "name" );
                mailingList.setName( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "subscribe" )  )
            {
                if ( parsed.contains( "subscribe" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "subscribe" );
                mailingList.setSubscribe( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "unsubscribe" )  )
            {
                if ( parsed.contains( "unsubscribe" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "unsubscribe" );
                mailingList.setUnsubscribe( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "post" )  )
            {
                if ( parsed.contains( "post" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "post" );
                mailingList.setPost( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "archive" )  )
            {
                if ( parsed.contains( "archive" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "archive" );
                mailingList.setArchive( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "otherArchives" )  )
            {
                if ( parsed.contains( "otherArchives" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "otherArchives" );
                java.util.List otherArchives = new java.util.ArrayList();
                mailingList.setOtherArchives( otherArchives );
                while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                {
                    if ( xmlStreamReader.getLocalName().equals( "otherArchive" ) )
                    {
                        otherArchives.add( getTrimmedValue( xmlStreamReader.getElementText()) );
                    }
                    else
                    {
                        throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
            }
        }
        return mailingList;
    } //-- MailingList parseMailingList(String, XMLStreamReader, boolean, String) 

    /**
     * Method parseModel
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param xmlStreamReader
     */
    private Model parseModel(String tagName, XMLStreamReader xmlStreamReader, boolean strict, String encoding)
        throws IOException, XMLStreamException
    {
        Model model = new Model();
        model.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        boolean foundRoot = false;
        while ( xmlStreamReader.hasNext() )
        {
            int eventType = xmlStreamReader.next();
            if ( eventType == XMLStreamConstants.START_ELEMENT )
            {
                if ( xmlStreamReader.getLocalName().equals( tagName ) )
                {
                    foundRoot = true;
                }
                else if ( xmlStreamReader.getLocalName().equals( "extend" )  )
                {
                    if ( parsed.contains( "extend" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "extend" );
                    model.setExtend( getTrimmedValue( xmlStreamReader.getElementText()) );
                }
                else if ( xmlStreamReader.getLocalName().equals( "pomVersion" )  )
                {
                    if ( parsed.contains( "pomVersion" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "pomVersion" );
                    model.setPomVersion( getTrimmedValue( xmlStreamReader.getElementText()) );
                }
                else if ( xmlStreamReader.getLocalName().equals( "groupId" )  )
                {
                    if ( parsed.contains( "groupId" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "groupId" );
                    model.setGroupId( getTrimmedValue( xmlStreamReader.getElementText()) );
                }
                else if ( xmlStreamReader.getLocalName().equals( "artifactId" )  )
                {
                    if ( parsed.contains( "artifactId" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "artifactId" );
                    model.setArtifactId( getTrimmedValue( xmlStreamReader.getElementText()) );
                }
                else if ( xmlStreamReader.getLocalName().equals( "id" )  )
                {
                    if ( parsed.contains( "id" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "id" );
                    model.setId( getTrimmedValue( xmlStreamReader.getElementText()) );
                }
                else if ( xmlStreamReader.getLocalName().equals( "name" )  )
                {
                    if ( parsed.contains( "name" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "name" );
                    model.setName( getTrimmedValue( xmlStreamReader.getElementText()) );
                }
                else if ( xmlStreamReader.getLocalName().equals( "currentVersion" )  )
                {
                    if ( parsed.contains( "currentVersion" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "currentVersion" );
                    model.setCurrentVersion( getTrimmedValue( xmlStreamReader.getElementText()) );
                }
                else if ( xmlStreamReader.getLocalName().equals( "shortDescription" )  )
                {
                    if ( parsed.contains( "shortDescription" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "shortDescription" );
                    model.setShortDescription( getTrimmedValue( xmlStreamReader.getElementText()) );
                }
                else if ( xmlStreamReader.getLocalName().equals( "description" )  )
                {
                    if ( parsed.contains( "description" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "description" );
                    model.setDescription( getTrimmedValue( xmlStreamReader.getElementText()) );
                }
                else if ( xmlStreamReader.getLocalName().equals( "url" )  )
                {
                    if ( parsed.contains( "url" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "url" );
                    model.setUrl( getTrimmedValue( xmlStreamReader.getElementText()) );
                }
                else if ( xmlStreamReader.getLocalName().equals( "logo" )  )
                {
                    if ( parsed.contains( "logo" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "logo" );
                    model.setLogo( getTrimmedValue( xmlStreamReader.getElementText()) );
                }
                else if ( xmlStreamReader.getLocalName().equals( "issueTrackingUrl" )  )
                {
                    if ( parsed.contains( "issueTrackingUrl" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "issueTrackingUrl" );
                    model.setIssueTrackingUrl( getTrimmedValue( xmlStreamReader.getElementText()) );
                }
                else if ( xmlStreamReader.getLocalName().equals( "inceptionYear" )  )
                {
                    if ( parsed.contains( "inceptionYear" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "inceptionYear" );
                    model.setInceptionYear( getTrimmedValue( xmlStreamReader.getElementText()) );
                }
                else if ( xmlStreamReader.getLocalName().equals( "gumpRepositoryId" )  )
                {
                    if ( parsed.contains( "gumpRepositoryId" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "gumpRepositoryId" );
                    model.setGumpRepositoryId( getTrimmedValue( xmlStreamReader.getElementText()) );
                }
                else if ( xmlStreamReader.getLocalName().equals( "siteAddress" )  )
                {
                    if ( parsed.contains( "siteAddress" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "siteAddress" );
                    model.setSiteAddress( getTrimmedValue( xmlStreamReader.getElementText()) );
                }
                else if ( xmlStreamReader.getLocalName().equals( "siteDirectory" )  )
                {
                    if ( parsed.contains( "siteDirectory" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "siteDirectory" );
                    model.setSiteDirectory( getTrimmedValue( xmlStreamReader.getElementText()) );
                }
                else if ( xmlStreamReader.getLocalName().equals( "distributionSite" )  )
                {
                    if ( parsed.contains( "distributionSite" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "distributionSite" );
                    model.setDistributionSite( getTrimmedValue( xmlStreamReader.getElementText()) );
                }
                else if ( xmlStreamReader.getLocalName().equals( "distributionDirectory" )  )
                {
                    if ( parsed.contains( "distributionDirectory" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "distributionDirectory" );
                    model.setDistributionDirectory( getTrimmedValue( xmlStreamReader.getElementText()) );
                }
                else if ( xmlStreamReader.getLocalName().equals( "mailingLists" )  )
                {
                    if ( parsed.contains( "mailingLists" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "mailingLists" );
                    java.util.List mailingLists = new java.util.ArrayList();
                    model.setMailingLists( mailingLists );
                    while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                    {
                        if ( xmlStreamReader.getLocalName().equals( "mailingList" ) )
                        {
                            model.addMailingList( parseMailingList( "mailingList", xmlStreamReader, strict, encoding ) );
                        }
                        else
                        {
                            throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                        }
                    }
                }
                else if ( xmlStreamReader.getLocalName().equals( "developers" )  )
                {
                    if ( parsed.contains( "developers" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "developers" );
                    java.util.List developers = new java.util.ArrayList();
                    model.setDevelopers( developers );
                    while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                    {
                        if ( xmlStreamReader.getLocalName().equals( "developer" ) )
                        {
                            model.addDeveloper( parseDeveloper( "developer", xmlStreamReader, strict, encoding ) );
                        }
                        else
                        {
                            throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                        }
                    }
                }
                else if ( xmlStreamReader.getLocalName().equals( "contributors" )  )
                {
                    if ( parsed.contains( "contributors" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "contributors" );
                    java.util.List contributors = new java.util.ArrayList();
                    model.setContributors( contributors );
                    while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                    {
                        if ( xmlStreamReader.getLocalName().equals( "contributor" ) )
                        {
                            model.addContributor( parseContributor( "contributor", xmlStreamReader, strict, encoding ) );
                        }
                        else
                        {
                            throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                        }
                    }
                }
                else if ( xmlStreamReader.getLocalName().equals( "licenses" )  )
                {
                    if ( parsed.contains( "licenses" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "licenses" );
                    java.util.List licenses = new java.util.ArrayList();
                    model.setLicenses( licenses );
                    while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                    {
                        if ( xmlStreamReader.getLocalName().equals( "license" ) )
                        {
                            model.addLicense( parseLicense( "license", xmlStreamReader, strict, encoding ) );
                        }
                        else
                        {
                            throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                        }
                    }
                }
                else if ( xmlStreamReader.getLocalName().equals( "versions" )  )
                {
                    if ( parsed.contains( "versions" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "versions" );
                    java.util.List versions = new java.util.ArrayList();
                    model.setVersions( versions );
                    while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                    {
                        if ( xmlStreamReader.getLocalName().equals( "version" ) )
                        {
                            model.addVersion( parseVersion( "version", xmlStreamReader, strict, encoding ) );
                        }
                        else
                        {
                            throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                        }
                    }
                }
                else if ( xmlStreamReader.getLocalName().equals( "branches" )  )
                {
                    if ( parsed.contains( "branches" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "branches" );
                    java.util.List branches = new java.util.ArrayList();
                    model.setBranches( branches );
                    while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                    {
                        if ( xmlStreamReader.getLocalName().equals( "branch" ) )
                        {
                            model.addBranch( parseBranch( "branch", xmlStreamReader, strict, encoding ) );
                        }
                        else
                        {
                            throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                        }
                    }
                }
                else if ( xmlStreamReader.getLocalName().equals( "packageGroups" )  )
                {
                    if ( parsed.contains( "packageGroups" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "packageGroups" );
                    java.util.List packageGroups = new java.util.ArrayList();
                    model.setPackageGroups( packageGroups );
                    while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                    {
                        if ( xmlStreamReader.getLocalName().equals( "packageGroup" ) )
                        {
                            model.addPackageGroup( parsePackageGroup( "packageGroup", xmlStreamReader, strict, encoding ) );
                        }
                        else
                        {
                            throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                        }
                    }
                }
                else if ( xmlStreamReader.getLocalName().equals( "reports" )  )
                {
                    if ( parsed.contains( "reports" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "reports" );
                    java.util.List reports = new java.util.ArrayList();
                    model.setReports( reports );
                    while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                    {
                        if ( xmlStreamReader.getLocalName().equals( "report" ) )
                        {
                            reports.add( getTrimmedValue( xmlStreamReader.getElementText()) );
                        }
                        else
                        {
                            throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                        }
                    }
                }
                else if ( xmlStreamReader.getLocalName().equals( "repository" )  )
                {
                    if ( parsed.contains( "repository" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "repository" );
                    model.setRepository( parseRepository( "repository", xmlStreamReader, strict, encoding ) );
                }
                else if ( xmlStreamReader.getLocalName().equals( "organization" ) || xmlStreamReader.getLocalName().equals( "organisation" )  )
                {
                    if ( parsed.contains( "organization" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "organization" );
                    model.setOrganization( parseOrganization( "organization", xmlStreamReader, strict, encoding ) );
                }
                else if ( xmlStreamReader.getLocalName().equals( "properties" )  )
                {
                    if ( parsed.contains( "properties" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "properties" );
                    while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                    {
                        String key = xmlStreamReader.getLocalName();
                        String value = xmlStreamReader.getElementText()
                        .trim()
                        ;
                        model.addProperty( key, value );
                    }
                }
                else if ( xmlStreamReader.getLocalName().equals( "package" )  )
                {
                    if ( parsed.contains( "package" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "package" );
                    model.setPackageName( getTrimmedValue( xmlStreamReader.getElementText()) );
                }
                else if ( xmlStreamReader.getLocalName().equals( "build" )  )
                {
                    if ( parsed.contains( "build" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "build" );
                    model.setBuild( parseBuild( "build", xmlStreamReader, strict, encoding ) );
                }
                else if ( xmlStreamReader.getLocalName().equals( "dependencies" )  )
                {
                    if ( parsed.contains( "dependencies" ) )
                    {
                        throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                    parsed.add( "dependencies" );
                    java.util.List dependencies = new java.util.ArrayList();
                    model.setDependencies( dependencies );
                    while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                    {
                        if ( xmlStreamReader.getLocalName().equals( "dependency" ) )
                        {
                            model.addDependency( parseDependency( "dependency", xmlStreamReader, strict, encoding ) );
                        }
                        else
                        {
                            throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                        }
                    }
                }
                else
                {
                    if ( foundRoot )
                    {
                        if ( strict )
                        {
                            throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                        }
                    }
                }
            }
        }
        return model;
    } //-- Model parseModel(String, XMLStreamReader, boolean, String) 

    /**
     * Method parseModelBase
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param xmlStreamReader
     */
    private ModelBase parseModelBase(String tagName, XMLStreamReader xmlStreamReader, boolean strict, String encoding)
        throws IOException, XMLStreamException
    {
        ModelBase modelBase = new ModelBase();
        modelBase.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
        {
            if ( xmlStreamReader.getLocalName().equals( "dependencies" )  )
            {
                if ( parsed.contains( "dependencies" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "dependencies" );
                java.util.List dependencies = new java.util.ArrayList();
                modelBase.setDependencies( dependencies );
                while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                {
                    if ( xmlStreamReader.getLocalName().equals( "dependency" ) )
                    {
                        modelBase.addDependency( parseDependency( "dependency", xmlStreamReader, strict, encoding ) );
                    }
                    else
                    {
                        throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
            }
        }
        return modelBase;
    } //-- ModelBase parseModelBase(String, XMLStreamReader, boolean, String) 

    /**
     * Method parseOrganization
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param xmlStreamReader
     */
    private Organization parseOrganization(String tagName, XMLStreamReader xmlStreamReader, boolean strict, String encoding)
        throws IOException, XMLStreamException
    {
        Organization organization = new Organization();
        organization.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
        {
            if ( xmlStreamReader.getLocalName().equals( "name" )  )
            {
                if ( parsed.contains( "name" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "name" );
                organization.setName( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "url" )  )
            {
                if ( parsed.contains( "url" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "url" );
                organization.setUrl( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "logo" )  )
            {
                if ( parsed.contains( "logo" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "logo" );
                organization.setLogo( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else
            {
                if ( strict )
                {
                    throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
            }
        }
        return organization;
    } //-- Organization parseOrganization(String, XMLStreamReader, boolean, String) 

    /**
     * Method parsePackageGroup
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param xmlStreamReader
     */
    private PackageGroup parsePackageGroup(String tagName, XMLStreamReader xmlStreamReader, boolean strict, String encoding)
        throws IOException, XMLStreamException
    {
        PackageGroup packageGroup = new PackageGroup();
        packageGroup.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
        {
            if ( xmlStreamReader.getLocalName().equals( "title" )  )
            {
                if ( parsed.contains( "title" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "title" );
                packageGroup.setTitle( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "packages" )  )
            {
                if ( parsed.contains( "packages" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "packages" );
                packageGroup.setPackages( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else
            {
                if ( strict )
                {
                    throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
            }
        }
        return packageGroup;
    } //-- PackageGroup parsePackageGroup(String, XMLStreamReader, boolean, String) 

    /**
     * Method parsePatternSet
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param xmlStreamReader
     */
    private PatternSet parsePatternSet(String tagName, XMLStreamReader xmlStreamReader, boolean strict, String encoding)
        throws IOException, XMLStreamException
    {
        PatternSet patternSet = new PatternSet();
        patternSet.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
        {
            if ( xmlStreamReader.getLocalName().equals( "includes" )  )
            {
                if ( parsed.contains( "includes" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "includes" );
                java.util.List includes = new java.util.ArrayList();
                patternSet.setIncludes( includes );
                while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                {
                    if ( xmlStreamReader.getLocalName().equals( "include" ) )
                    {
                        includes.add( getTrimmedValue( xmlStreamReader.getElementText()) );
                    }
                    else
                    {
                        throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                }
            }
            else if ( xmlStreamReader.getLocalName().equals( "excludes" )  )
            {
                if ( parsed.contains( "excludes" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "excludes" );
                java.util.List excludes = new java.util.ArrayList();
                patternSet.setExcludes( excludes );
                while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                {
                    if ( xmlStreamReader.getLocalName().equals( "exclude" ) )
                    {
                        excludes.add( getTrimmedValue( xmlStreamReader.getElementText()) );
                    }
                    else
                    {
                        throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
            }
        }
        return patternSet;
    } //-- PatternSet parsePatternSet(String, XMLStreamReader, boolean, String) 

    /**
     * Method parsePluginConfiguration
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param xmlStreamReader
     */
    private PluginConfiguration parsePluginConfiguration(String tagName, XMLStreamReader xmlStreamReader, boolean strict, String encoding)
        throws IOException, XMLStreamException
    {
        PluginConfiguration pluginConfiguration = new PluginConfiguration();
        pluginConfiguration.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
        {
            if ( strict )
            {
                throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
            }
        }
        return pluginConfiguration;
    } //-- PluginConfiguration parsePluginConfiguration(String, XMLStreamReader, boolean, String) 

    /**
     * Method parsePluginContainer
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param xmlStreamReader
     */
    private PluginContainer parsePluginContainer(String tagName, XMLStreamReader xmlStreamReader, boolean strict, String encoding)
        throws IOException, XMLStreamException
    {
        PluginContainer pluginContainer = new PluginContainer();
        pluginContainer.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
        {
            if ( strict )
            {
                throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
            }
        }
        return pluginContainer;
    } //-- PluginContainer parsePluginContainer(String, XMLStreamReader, boolean, String) 

    /**
     * Method parseRepository
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param xmlStreamReader
     */
    private Repository parseRepository(String tagName, XMLStreamReader xmlStreamReader, boolean strict, String encoding)
        throws IOException, XMLStreamException
    {
        Repository repository = new Repository();
        repository.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
        {
            if ( xmlStreamReader.getLocalName().equals( "connection" )  )
            {
                if ( parsed.contains( "connection" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "connection" );
                repository.setConnection( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "developerConnection" )  )
            {
                if ( parsed.contains( "developerConnection" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "developerConnection" );
                repository.setDeveloperConnection( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "url" )  )
            {
                if ( parsed.contains( "url" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "url" );
                repository.setUrl( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else
            {
                if ( strict )
                {
                    throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
            }
        }
        return repository;
    } //-- Repository parseRepository(String, XMLStreamReader, boolean, String) 

    /**
     * Method parseResource
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param xmlStreamReader
     */
    private Resource parseResource(String tagName, XMLStreamReader xmlStreamReader, boolean strict, String encoding)
        throws IOException, XMLStreamException
    {
        Resource resource = new Resource();
        resource.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
        {
            if ( xmlStreamReader.getLocalName().equals( "targetPath" )  )
            {
                if ( parsed.contains( "targetPath" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "targetPath" );
                resource.setTargetPath( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "filtering" )  )
            {
                if ( parsed.contains( "filtering" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "filtering" );
                resource.setFiltering( getBooleanValue( getTrimmedValue( xmlStreamReader.getElementText()), "filtering", xmlStreamReader ) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "directory" )  )
            {
                if ( parsed.contains( "directory" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "directory" );
                resource.setDirectory( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "includes" )  )
            {
                if ( parsed.contains( "includes" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "includes" );
                java.util.List includes = new java.util.ArrayList();
                resource.setIncludes( includes );
                while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                {
                    if ( xmlStreamReader.getLocalName().equals( "include" ) )
                    {
                        includes.add( getTrimmedValue( xmlStreamReader.getElementText()) );
                    }
                    else
                    {
                        throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                }
            }
            else if ( xmlStreamReader.getLocalName().equals( "excludes" )  )
            {
                if ( parsed.contains( "excludes" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "excludes" );
                java.util.List excludes = new java.util.ArrayList();
                resource.setExcludes( excludes );
                while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                {
                    if ( xmlStreamReader.getLocalName().equals( "exclude" ) )
                    {
                        excludes.add( getTrimmedValue( xmlStreamReader.getElementText()) );
                    }
                    else
                    {
                        throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
            }
        }
        return resource;
    } //-- Resource parseResource(String, XMLStreamReader, boolean, String) 

    /**
     * Method parseSourceModification
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param xmlStreamReader
     */
    private SourceModification parseSourceModification(String tagName, XMLStreamReader xmlStreamReader, boolean strict, String encoding)
        throws IOException, XMLStreamException
    {
        SourceModification sourceModification = new SourceModification();
        sourceModification.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
        {
            if ( xmlStreamReader.getLocalName().equals( "className" )  )
            {
                if ( parsed.contains( "className" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "className" );
                sourceModification.setClassName( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "property" )  )
            {
                if ( parsed.contains( "property" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "property" );
                sourceModification.setProperty( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "directory" )  )
            {
                if ( parsed.contains( "directory" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "directory" );
                sourceModification.setDirectory( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "includes" )  )
            {
                if ( parsed.contains( "includes" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "includes" );
                java.util.List includes = new java.util.ArrayList();
                sourceModification.setIncludes( includes );
                while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                {
                    if ( xmlStreamReader.getLocalName().equals( "include" ) )
                    {
                        includes.add( getTrimmedValue( xmlStreamReader.getElementText()) );
                    }
                    else
                    {
                        throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                }
            }
            else if ( xmlStreamReader.getLocalName().equals( "excludes" )  )
            {
                if ( parsed.contains( "excludes" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "excludes" );
                java.util.List excludes = new java.util.ArrayList();
                sourceModification.setExcludes( excludes );
                while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                {
                    if ( xmlStreamReader.getLocalName().equals( "exclude" ) )
                    {
                        excludes.add( getTrimmedValue( xmlStreamReader.getElementText()) );
                    }
                    else
                    {
                        throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
            }
        }
        return sourceModification;
    } //-- SourceModification parseSourceModification(String, XMLStreamReader, boolean, String) 

    /**
     * Method parseUnitTest
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param xmlStreamReader
     */
    private UnitTest parseUnitTest(String tagName, XMLStreamReader xmlStreamReader, boolean strict, String encoding)
        throws IOException, XMLStreamException
    {
        UnitTest unitTest = new UnitTest();
        unitTest.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
        {
            if ( xmlStreamReader.getLocalName().equals( "resources" )  )
            {
                if ( parsed.contains( "resources" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "resources" );
                java.util.List resources = new java.util.ArrayList();
                unitTest.setResources( resources );
                while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                {
                    if ( xmlStreamReader.getLocalName().equals( "resource" ) )
                    {
                        unitTest.addResource( parseResource( "resource", xmlStreamReader, strict, encoding ) );
                    }
                    else
                    {
                        throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                }
            }
            else if ( xmlStreamReader.getLocalName().equals( "includes" )  )
            {
                if ( parsed.contains( "includes" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "includes" );
                java.util.List includes = new java.util.ArrayList();
                unitTest.setIncludes( includes );
                while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                {
                    if ( xmlStreamReader.getLocalName().equals( "include" ) )
                    {
                        includes.add( getTrimmedValue( xmlStreamReader.getElementText()) );
                    }
                    else
                    {
                        throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                }
            }
            else if ( xmlStreamReader.getLocalName().equals( "excludes" )  )
            {
                if ( parsed.contains( "excludes" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "excludes" );
                java.util.List excludes = new java.util.ArrayList();
                unitTest.setExcludes( excludes );
                while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
                {
                    if ( xmlStreamReader.getLocalName().equals( "exclude" ) )
                    {
                        excludes.add( getTrimmedValue( xmlStreamReader.getElementText()) );
                    }
                    else
                    {
                        throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                    }
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
            }
        }
        return unitTest;
    } //-- UnitTest parseUnitTest(String, XMLStreamReader, boolean, String) 

    /**
     * Method parseVersion
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param xmlStreamReader
     */
    private Version parseVersion(String tagName, XMLStreamReader xmlStreamReader, boolean strict, String encoding)
        throws IOException, XMLStreamException
    {
        Version version = new Version();
        version.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( xmlStreamReader.nextTag() == XMLStreamConstants.START_ELEMENT )
        {
            if ( xmlStreamReader.getLocalName().equals( "name" )  )
            {
                if ( parsed.contains( "name" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "name" );
                version.setName( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "tag" )  )
            {
                if ( parsed.contains( "tag" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "tag" );
                version.setTag( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else if ( xmlStreamReader.getLocalName().equals( "id" )  )
            {
                if ( parsed.contains( "id" ) )
                {
                    throw new XMLStreamException( "Duplicated tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
                parsed.add( "id" );
                version.setId( getTrimmedValue( xmlStreamReader.getElementText()) );
            }
            else
            {
                if ( strict )
                {
                    throw new XMLStreamException( "Unrecognised tag: '" + xmlStreamReader.getLocalName() + "'", xmlStreamReader.getLocation() );
                }
            }
        }
        return version;
    } //-- Version parseVersion(String, XMLStreamReader, boolean, String) 

    /**
     * Method read
     * 
     * @param reader
     * @param strict
     */
    public Model read(Reader reader, boolean strict)
        throws IOException, XMLStreamException
    {
        XMLStreamReader xmlStreamReader = XMLInputFactory.newInstance().createXMLStreamReader( reader );
        
        String encoding = xmlStreamReader.getCharacterEncodingScheme();
        Model value = parseModel( "project", xmlStreamReader, strict, encoding );
        resolveReferences( value );
        return value;
    } //-- Model read(Reader, boolean) 

    /**
     * Method read
     * 
     * @param reader
     */
    public Model read(Reader reader)
        throws IOException, XMLStreamException
    {
        return read( reader, true );
    } //-- Model read(Reader) 

    /**
     * Method read
     * 
     * @param filePath
     * @param strict
     */
    public Model read(String filePath, boolean strict)
        throws IOException, XMLStreamException
    {
        File file = new File(filePath);
        XMLStreamReader xmlStreamReader = XMLInputFactory.newInstance().createXMLStreamReader( file.toURL().toExternalForm(), new FileInputStream(file) );
        
        String encoding = xmlStreamReader.getCharacterEncodingScheme();
        Model value = parseModel( "project", xmlStreamReader, strict, encoding );
        resolveReferences( value );
        return value;
    } //-- Model read(String, boolean) 

    /**
     * Method read
     * 
     * @param filePath
     */
    public Model read(String filePath)
        throws IOException, XMLStreamException
    {
        return read( filePath, true );
    } //-- Model read(String) 

    /**
     * Method resolveReferences
     * 
     * @param value
     */
    private void resolveReferences(Model value)
    {
        java.util.Map refs;
    } //-- void resolveReferences(Model) 


}
