/*
 * $Id$
 */

package org.apache.maven.project.io.xpp3;

  //---------------------------------/
 //- Imported classes and packages -/
//---------------------------------/

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.text.DateFormat;
import java.text.ParsePosition;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.maven.project.Branch;
import org.apache.maven.project.Build;
import org.apache.maven.project.BuildBase;
import org.apache.maven.project.Contributor;
import org.apache.maven.project.Dependency;
import org.apache.maven.project.Developer;
import org.apache.maven.project.FileSet;
import org.apache.maven.project.License;
import org.apache.maven.project.MailingList;
import org.apache.maven.project.Model;
import org.apache.maven.project.ModelBase;
import org.apache.maven.project.Organization;
import org.apache.maven.project.PackageGroup;
import org.apache.maven.project.PatternSet;
import org.apache.maven.project.PluginConfiguration;
import org.apache.maven.project.PluginContainer;
import org.apache.maven.project.Repository;
import org.apache.maven.project.Resource;
import org.apache.maven.project.SourceModification;
import org.apache.maven.project.UnitTest;
import org.apache.maven.project.Version;
import org.codehaus.plexus.util.IOUtil;
import org.codehaus.plexus.util.xml.pull.MXParser;
import org.codehaus.plexus.util.xml.pull.XmlPullParser;
import org.codehaus.plexus.util.xml.pull.XmlPullParserException;

/**
 * Class MavenXpp3Reader.
 * 
 * @version $Revision$ $Date$
 */
public class MavenXpp3Reader {


      //--------------------------/
     //- Class/Member Variables -/
    //--------------------------/

    /**
     * If set the parser till be loaded with all single characters
     * from the XHTML specification.
     * The entities used:
     * <ul>
     * <li>http://www.w3.org/TR/xhtml1/DTD/xhtml-lat1.ent</li>
     * <li>http://www.w3.org/TR/xhtml1/DTD/xhtml-special.ent</li>
     * <li>http://www.w3.org/TR/xhtml1/DTD/xhtml-symbol.ent</li>
     * </ul>
     */
    private boolean addDefaultEntities = true;


      //-----------/
     //- Methods -/
    //-----------/

    /**
     * Method getAddDefaultEntities
     */
    public boolean getAddDefaultEntities()
    {
        return addDefaultEntities;
    } //-- boolean getAddDefaultEntities() 

    /**
     * Method getBooleanValue
     * 
     * @param s
     * @param parser
     * @param attribute
     */
    public boolean getBooleanValue(String s, String attribute, XmlPullParser parser)
        throws XmlPullParserException
    {
        if ( s != null )
        {
            return Boolean.valueOf( s ).booleanValue();
        }
        return false;
    } //-- boolean getBooleanValue(String, String, XmlPullParser) 

    /**
     * Method getCharacterValue
     * 
     * @param s
     * @param parser
     * @param attribute
     */
    public char getCharacterValue(String s, String attribute, XmlPullParser parser)
        throws XmlPullParserException
    {
        if ( s != null )
        {
            return s.charAt( 0 );
        }
        return 0;
    } //-- char getCharacterValue(String, String, XmlPullParser) 

    /**
     * Method getDateValue
     * 
     * @param s
     * @param parser
     * @param dateFormat
     * @param attribute
     */
    public java.util.Date getDateValue(String s, String attribute, String dateFormat, XmlPullParser parser)
        throws XmlPullParserException
    {
        if ( s != null )
        {
            if ( dateFormat == null )
            {
                return new java.util.Date( Long.valueOf( s ).longValue() );
            }
            else
            {
                DateFormat dateParser = new java.text.SimpleDateFormat( dateFormat, Locale.US );
                try
                {
                    return dateParser.parse( s );
                }
                catch ( java.text.ParseException e )
                {
                    throw new XmlPullParserException( e.getMessage() );
                }
            }
        }
        return null;
    } //-- java.util.Date getDateValue(String, String, String, XmlPullParser) 

    /**
     * Method getDoubleValue
     * 
     * @param s
     * @param strict
     * @param parser
     * @param attribute
     */
    public double getDoubleValue(String s, String attribute, XmlPullParser parser, boolean strict)
        throws XmlPullParserException
    {
        if ( s != null )
        {
            try
            {
                return Double.valueOf( s ).doubleValue();
            }
            catch ( NumberFormatException e )
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unable to parse element '" + attribute + "', must be a floating point number", parser, null );
                }
            }
        }
        return 0;
    } //-- double getDoubleValue(String, String, XmlPullParser, boolean) 

    /**
     * Method getFloatValue
     * 
     * @param s
     * @param strict
     * @param parser
     * @param attribute
     */
    public float getFloatValue(String s, String attribute, XmlPullParser parser, boolean strict)
        throws XmlPullParserException
    {
        if ( s != null )
        {
            try
            {
                return Float.valueOf( s ).floatValue();
            }
            catch ( NumberFormatException e )
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unable to parse element '" + attribute + "', must be a floating point number", parser, null );
                }
            }
        }
        return 0;
    } //-- float getFloatValue(String, String, XmlPullParser, boolean) 

    /**
     * Method getIntegerValue
     * 
     * @param s
     * @param strict
     * @param parser
     * @param attribute
     */
    public int getIntegerValue(String s, String attribute, XmlPullParser parser, boolean strict)
        throws XmlPullParserException
    {
        if ( s != null )
        {
            try
            {
                return Integer.valueOf( s ).intValue();
            }
            catch ( NumberFormatException e )
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unable to parse element '" + attribute + "', must be an integer", parser, null );
                }
            }
        }
        return 0;
    } //-- int getIntegerValue(String, String, XmlPullParser, boolean) 

    /**
     * Method getLongValue
     * 
     * @param s
     * @param strict
     * @param parser
     * @param attribute
     */
    public long getLongValue(String s, String attribute, XmlPullParser parser, boolean strict)
        throws XmlPullParserException
    {
        if ( s != null )
        {
            try
            {
                return Long.valueOf( s ).longValue();
            }
            catch ( NumberFormatException e )
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unable to parse element '" + attribute + "', must be a long integer", parser, null );
                }
            }
        }
        return 0;
    } //-- long getLongValue(String, String, XmlPullParser, boolean) 

    /**
     * Method getRequiredAttributeValue
     * 
     * @param s
     * @param strict
     * @param parser
     * @param attribute
     */
    public String getRequiredAttributeValue(String s, String attribute, XmlPullParser parser, boolean strict)
        throws XmlPullParserException
    {
        if ( s == null )
        {
            if ( strict )
            {
                throw new XmlPullParserException( "Missing required value for attribute '" + attribute + "'", parser, null );
            }
        }
        return s;
    } //-- String getRequiredAttributeValue(String, String, XmlPullParser, boolean) 

    /**
     * Method getShortValue
     * 
     * @param s
     * @param strict
     * @param parser
     * @param attribute
     */
    public short getShortValue(String s, String attribute, XmlPullParser parser, boolean strict)
        throws XmlPullParserException
    {
        if ( s != null )
        {
            try
            {
                return Short.valueOf( s ).shortValue();
            }
            catch ( NumberFormatException e )
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unable to parse element '" + attribute + "', must be a short integer", parser, null );
                }
            }
        }
        return 0;
    } //-- short getShortValue(String, String, XmlPullParser, boolean) 

    /**
     * Method getTrimmedValue
     * 
     * @param s
     */
    public String getTrimmedValue(String s)
    {
        if ( s != null )
        {
            s = s.trim();
        }
        return s;
    } //-- String getTrimmedValue(String) 

    /**
     * Method parseBranch
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param parser
     */
    private Branch parseBranch(String tagName, XmlPullParser parser, boolean strict, String encoding)
        throws IOException, XmlPullParserException
    {
        Branch branch = new Branch();
        branch.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "tag" )  )
            {
                if ( parsed.contains( "tag" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "tag" );
                branch.setTag( getTrimmedValue( parser.nextText()) );
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null);
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return branch;
    } //-- Branch parseBranch(String, XmlPullParser, boolean, String) 

    /**
     * Method parseBuild
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param parser
     */
    private Build parseBuild(String tagName, XmlPullParser parser, boolean strict, String encoding)
        throws IOException, XmlPullParserException
    {
        Build build = new Build();
        build.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "nagEmailAddress" )  )
            {
                if ( parsed.contains( "nagEmailAddress" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "nagEmailAddress" );
                build.setNagEmailAddress( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "sourceDirectory" )  )
            {
                if ( parsed.contains( "sourceDirectory" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "sourceDirectory" );
                build.setSourceDirectory( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "unitTestSourceDirectory" )  )
            {
                if ( parsed.contains( "unitTestSourceDirectory" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "unitTestSourceDirectory" );
                build.setUnitTestSourceDirectory( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "aspectSourceDirectory" )  )
            {
                if ( parsed.contains( "aspectSourceDirectory" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "aspectSourceDirectory" );
                build.setAspectSourceDirectory( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "integrationUnitTestSourceDirectory" )  )
            {
                if ( parsed.contains( "integrationUnitTestSourceDirectory" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "integrationUnitTestSourceDirectory" );
                build.setIntegrationUnitTestSourceDirectory( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "sourceModifications" )  )
            {
                if ( parsed.contains( "sourceModifications" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "sourceModifications" );
                java.util.List sourceModifications = new java.util.ArrayList();
                build.setSourceModifications( sourceModifications );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    if ( parser.getName().equals( "sourceModification" ) )
                    {
                        sourceModifications.add( parseSourceModification( "sourceModification", parser, strict, encoding ) );
                    }
                    else
                    {
                        parser.nextText();
                    }
                }
            }
            else if ( parser.getName().equals( "unitTest" )  )
            {
                if ( parsed.contains( "unitTest" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "unitTest" );
                build.setUnitTest( parseUnitTest( "unitTest", parser, strict, encoding ) );
            }
            else if ( parser.getName().equals( "defaultGoal" )  )
            {
                if ( parsed.contains( "defaultGoal" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "defaultGoal" );
                build.setDefaultGoal( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "resources" )  )
            {
                if ( parsed.contains( "resources" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "resources" );
                java.util.List resources = new java.util.ArrayList();
                build.setResources( resources );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    if ( parser.getName().equals( "resource" ) )
                    {
                        resources.add( parseResource( "resource", parser, strict, encoding ) );
                    }
                    else
                    {
                        parser.nextText();
                    }
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null);
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return build;
    } //-- Build parseBuild(String, XmlPullParser, boolean, String) 

    /**
     * Method parseBuildBase
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param parser
     */
    private BuildBase parseBuildBase(String tagName, XmlPullParser parser, boolean strict, String encoding)
        throws IOException, XmlPullParserException
    {
        BuildBase buildBase = new BuildBase();
        buildBase.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "defaultGoal" )  )
            {
                if ( parsed.contains( "defaultGoal" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "defaultGoal" );
                buildBase.setDefaultGoal( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "resources" )  )
            {
                if ( parsed.contains( "resources" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "resources" );
                java.util.List resources = new java.util.ArrayList();
                buildBase.setResources( resources );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    if ( parser.getName().equals( "resource" ) )
                    {
                        resources.add( parseResource( "resource", parser, strict, encoding ) );
                    }
                    else
                    {
                        parser.nextText();
                    }
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null);
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return buildBase;
    } //-- BuildBase parseBuildBase(String, XmlPullParser, boolean, String) 

    /**
     * Method parseContributor
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param parser
     */
    private Contributor parseContributor(String tagName, XmlPullParser parser, boolean strict, String encoding)
        throws IOException, XmlPullParserException
    {
        Contributor contributor = new Contributor();
        contributor.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "name" )  )
            {
                if ( parsed.contains( "name" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "name" );
                contributor.setName( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "email" )  )
            {
                if ( parsed.contains( "email" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "email" );
                contributor.setEmail( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "url" )  )
            {
                if ( parsed.contains( "url" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "url" );
                contributor.setUrl( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "organization" ) || parser.getName().equals( "organisation" )  )
            {
                if ( parsed.contains( "organization" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "organization" );
                contributor.setOrganization( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "organizationUrl" ) || parser.getName().equals( "organisationUrl" )  )
            {
                if ( parsed.contains( "organizationUrl" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "organizationUrl" );
                contributor.setOrganizationUrl( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "roles" )  )
            {
                if ( parsed.contains( "roles" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "roles" );
                java.util.List roles = new java.util.ArrayList();
                contributor.setRoles( roles );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    if ( parser.getName().equals( "role" ) )
                    {
                        roles.add( getTrimmedValue( parser.nextText()) );
                    }
                    else
                    {
                        parser.nextText();
                    }
                }
            }
            else if ( parser.getName().equals( "timezone" )  )
            {
                if ( parsed.contains( "timezone" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "timezone" );
                contributor.setTimezone( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "properties" )  )
            {
                if ( parsed.contains( "properties" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "properties" );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    String key = parser.getName();
                    String value = parser.nextText()
                    .trim()
                    ;
                    contributor.addProperty( key, value );
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null);
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return contributor;
    } //-- Contributor parseContributor(String, XmlPullParser, boolean, String) 

    /**
     * Method parseDependency
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param parser
     */
    private Dependency parseDependency(String tagName, XmlPullParser parser, boolean strict, String encoding)
        throws IOException, XmlPullParserException
    {
        Dependency dependency = new Dependency();
        dependency.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "id" )  )
            {
                if ( parsed.contains( "id" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "id" );
                dependency.setId( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "groupId" )  )
            {
                if ( parsed.contains( "groupId" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "groupId" );
                dependency.setGroupId( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "artifactId" )  )
            {
                if ( parsed.contains( "artifactId" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "artifactId" );
                dependency.setArtifactId( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "version" )  )
            {
                if ( parsed.contains( "version" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "version" );
                dependency.setVersion( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "url" )  )
            {
                if ( parsed.contains( "url" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "url" );
                dependency.setUrl( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "jar" )  )
            {
                if ( parsed.contains( "jar" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "jar" );
                dependency.setJar( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "type" )  )
            {
                if ( parsed.contains( "type" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "type" );
                dependency.setType( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "properties" )  )
            {
                if ( parsed.contains( "properties" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "properties" );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    String key = parser.getName();
                    String value = parser.nextText()
                    .trim()
                    ;
                    dependency.addProperty( key, value );
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null);
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return dependency;
    } //-- Dependency parseDependency(String, XmlPullParser, boolean, String) 

    /**
     * Method parseDeveloper
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param parser
     */
    private Developer parseDeveloper(String tagName, XmlPullParser parser, boolean strict, String encoding)
        throws IOException, XmlPullParserException
    {
        Developer developer = new Developer();
        developer.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "id" )  )
            {
                if ( parsed.contains( "id" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "id" );
                developer.setId( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "name" )  )
            {
                if ( parsed.contains( "name" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "name" );
                developer.setName( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "email" )  )
            {
                if ( parsed.contains( "email" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "email" );
                developer.setEmail( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "url" )  )
            {
                if ( parsed.contains( "url" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "url" );
                developer.setUrl( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "organization" ) || parser.getName().equals( "organisation" )  )
            {
                if ( parsed.contains( "organization" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "organization" );
                developer.setOrganization( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "organizationUrl" ) || parser.getName().equals( "organisationUrl" )  )
            {
                if ( parsed.contains( "organizationUrl" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "organizationUrl" );
                developer.setOrganizationUrl( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "roles" )  )
            {
                if ( parsed.contains( "roles" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "roles" );
                java.util.List roles = new java.util.ArrayList();
                developer.setRoles( roles );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    if ( parser.getName().equals( "role" ) )
                    {
                        roles.add( getTrimmedValue( parser.nextText()) );
                    }
                    else
                    {
                        parser.nextText();
                    }
                }
            }
            else if ( parser.getName().equals( "timezone" )  )
            {
                if ( parsed.contains( "timezone" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "timezone" );
                developer.setTimezone( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "properties" )  )
            {
                if ( parsed.contains( "properties" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "properties" );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    String key = parser.getName();
                    String value = parser.nextText()
                    .trim()
                    ;
                    developer.addProperty( key, value );
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null);
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return developer;
    } //-- Developer parseDeveloper(String, XmlPullParser, boolean, String) 

    /**
     * Method parseFileSet
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param parser
     */
    private FileSet parseFileSet(String tagName, XmlPullParser parser, boolean strict, String encoding)
        throws IOException, XmlPullParserException
    {
        FileSet fileSet = new FileSet();
        fileSet.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "directory" )  )
            {
                if ( parsed.contains( "directory" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "directory" );
                fileSet.setDirectory( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "includes" )  )
            {
                if ( parsed.contains( "includes" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "includes" );
                java.util.List includes = new java.util.ArrayList();
                fileSet.setIncludes( includes );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    if ( parser.getName().equals( "include" ) )
                    {
                        includes.add( getTrimmedValue( parser.nextText()) );
                    }
                    else
                    {
                        parser.nextText();
                    }
                }
            }
            else if ( parser.getName().equals( "excludes" )  )
            {
                if ( parsed.contains( "excludes" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "excludes" );
                java.util.List excludes = new java.util.ArrayList();
                fileSet.setExcludes( excludes );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    if ( parser.getName().equals( "exclude" ) )
                    {
                        excludes.add( getTrimmedValue( parser.nextText()) );
                    }
                    else
                    {
                        parser.nextText();
                    }
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null);
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return fileSet;
    } //-- FileSet parseFileSet(String, XmlPullParser, boolean, String) 

    /**
     * Method parseLicense
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param parser
     */
    private License parseLicense(String tagName, XmlPullParser parser, boolean strict, String encoding)
        throws IOException, XmlPullParserException
    {
        License license = new License();
        license.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "name" )  )
            {
                if ( parsed.contains( "name" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "name" );
                license.setName( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "url" )  )
            {
                if ( parsed.contains( "url" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "url" );
                license.setUrl( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "distribution" )  )
            {
                if ( parsed.contains( "distribution" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "distribution" );
                license.setDistribution( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "comments" )  )
            {
                if ( parsed.contains( "comments" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "comments" );
                license.setComments( getTrimmedValue( parser.nextText()) );
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null);
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return license;
    } //-- License parseLicense(String, XmlPullParser, boolean, String) 

    /**
     * Method parseMailingList
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param parser
     */
    private MailingList parseMailingList(String tagName, XmlPullParser parser, boolean strict, String encoding)
        throws IOException, XmlPullParserException
    {
        MailingList mailingList = new MailingList();
        mailingList.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "name" )  )
            {
                if ( parsed.contains( "name" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "name" );
                mailingList.setName( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "subscribe" )  )
            {
                if ( parsed.contains( "subscribe" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "subscribe" );
                mailingList.setSubscribe( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "unsubscribe" )  )
            {
                if ( parsed.contains( "unsubscribe" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "unsubscribe" );
                mailingList.setUnsubscribe( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "post" )  )
            {
                if ( parsed.contains( "post" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "post" );
                mailingList.setPost( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "archive" )  )
            {
                if ( parsed.contains( "archive" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "archive" );
                mailingList.setArchive( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "otherArchives" )  )
            {
                if ( parsed.contains( "otherArchives" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "otherArchives" );
                java.util.List otherArchives = new java.util.ArrayList();
                mailingList.setOtherArchives( otherArchives );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    if ( parser.getName().equals( "otherArchive" ) )
                    {
                        otherArchives.add( getTrimmedValue( parser.nextText()) );
                    }
                    else
                    {
                        parser.nextText();
                    }
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null);
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return mailingList;
    } //-- MailingList parseMailingList(String, XmlPullParser, boolean, String) 

    /**
     * Method parseModel
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param parser
     */
    private Model parseModel(String tagName, XmlPullParser parser, boolean strict, String encoding)
        throws IOException, XmlPullParserException
    {
        Model model = new Model();
        model.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        int eventType = parser.getEventType();
        boolean foundRoot = false;
        while ( eventType != XmlPullParser.END_DOCUMENT )
        {
            if ( eventType == XmlPullParser.START_TAG )
            {
                if ( parser.getName().equals( tagName ) )
                {
                    foundRoot = true;
                }
                else if ( parser.getName().equals( "extend" )  )
                {
                    if ( parsed.contains( "extend" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "extend" );
                    model.setExtend( getTrimmedValue( parser.nextText()) );
                }
                else if ( parser.getName().equals( "pomVersion" )  )
                {
                    if ( parsed.contains( "pomVersion" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "pomVersion" );
                    model.setPomVersion( getTrimmedValue( parser.nextText()) );
                }
                else if ( parser.getName().equals( "groupId" )  )
                {
                    if ( parsed.contains( "groupId" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "groupId" );
                    model.setGroupId( getTrimmedValue( parser.nextText()) );
                }
                else if ( parser.getName().equals( "artifactId" )  )
                {
                    if ( parsed.contains( "artifactId" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "artifactId" );
                    model.setArtifactId( getTrimmedValue( parser.nextText()) );
                }
                else if ( parser.getName().equals( "id" )  )
                {
                    if ( parsed.contains( "id" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "id" );
                    model.setId( getTrimmedValue( parser.nextText()) );
                }
                else if ( parser.getName().equals( "name" )  )
                {
                    if ( parsed.contains( "name" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "name" );
                    model.setName( getTrimmedValue( parser.nextText()) );
                }
                else if ( parser.getName().equals( "currentVersion" )  )
                {
                    if ( parsed.contains( "currentVersion" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "currentVersion" );
                    model.setCurrentVersion( getTrimmedValue( parser.nextText()) );
                }
                else if ( parser.getName().equals( "shortDescription" )  )
                {
                    if ( parsed.contains( "shortDescription" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "shortDescription" );
                    model.setShortDescription( getTrimmedValue( parser.nextText()) );
                }
                else if ( parser.getName().equals( "description" )  )
                {
                    if ( parsed.contains( "description" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "description" );
                    model.setDescription( getTrimmedValue( parser.nextText()) );
                }
                else if ( parser.getName().equals( "url" )  )
                {
                    if ( parsed.contains( "url" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "url" );
                    model.setUrl( getTrimmedValue( parser.nextText()) );
                }
                else if ( parser.getName().equals( "logo" )  )
                {
                    if ( parsed.contains( "logo" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "logo" );
                    model.setLogo( getTrimmedValue( parser.nextText()) );
                }
                else if ( parser.getName().equals( "issueTrackingUrl" )  )
                {
                    if ( parsed.contains( "issueTrackingUrl" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "issueTrackingUrl" );
                    model.setIssueTrackingUrl( getTrimmedValue( parser.nextText()) );
                }
                else if ( parser.getName().equals( "inceptionYear" )  )
                {
                    if ( parsed.contains( "inceptionYear" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "inceptionYear" );
                    model.setInceptionYear( getTrimmedValue( parser.nextText()) );
                }
                else if ( parser.getName().equals( "gumpRepositoryId" )  )
                {
                    if ( parsed.contains( "gumpRepositoryId" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "gumpRepositoryId" );
                    model.setGumpRepositoryId( getTrimmedValue( parser.nextText()) );
                }
                else if ( parser.getName().equals( "siteAddress" )  )
                {
                    if ( parsed.contains( "siteAddress" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "siteAddress" );
                    model.setSiteAddress( getTrimmedValue( parser.nextText()) );
                }
                else if ( parser.getName().equals( "siteDirectory" )  )
                {
                    if ( parsed.contains( "siteDirectory" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "siteDirectory" );
                    model.setSiteDirectory( getTrimmedValue( parser.nextText()) );
                }
                else if ( parser.getName().equals( "distributionSite" )  )
                {
                    if ( parsed.contains( "distributionSite" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "distributionSite" );
                    model.setDistributionSite( getTrimmedValue( parser.nextText()) );
                }
                else if ( parser.getName().equals( "distributionDirectory" )  )
                {
                    if ( parsed.contains( "distributionDirectory" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "distributionDirectory" );
                    model.setDistributionDirectory( getTrimmedValue( parser.nextText()) );
                }
                else if ( parser.getName().equals( "mailingLists" )  )
                {
                    if ( parsed.contains( "mailingLists" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "mailingLists" );
                    java.util.List mailingLists = new java.util.ArrayList();
                    model.setMailingLists( mailingLists );
                    while ( parser.nextTag() == XmlPullParser.START_TAG )
                    {
                        if ( parser.getName().equals( "mailingList" ) )
                        {
                            mailingLists.add( parseMailingList( "mailingList", parser, strict, encoding ) );
                        }
                        else
                        {
                            parser.nextText();
                        }
                    }
                }
                else if ( parser.getName().equals( "developers" )  )
                {
                    if ( parsed.contains( "developers" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "developers" );
                    java.util.List developers = new java.util.ArrayList();
                    model.setDevelopers( developers );
                    while ( parser.nextTag() == XmlPullParser.START_TAG )
                    {
                        if ( parser.getName().equals( "developer" ) )
                        {
                            developers.add( parseDeveloper( "developer", parser, strict, encoding ) );
                        }
                        else
                        {
                            parser.nextText();
                        }
                    }
                }
                else if ( parser.getName().equals( "contributors" )  )
                {
                    if ( parsed.contains( "contributors" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "contributors" );
                    java.util.List contributors = new java.util.ArrayList();
                    model.setContributors( contributors );
                    while ( parser.nextTag() == XmlPullParser.START_TAG )
                    {
                        if ( parser.getName().equals( "contributor" ) )
                        {
                            contributors.add( parseContributor( "contributor", parser, strict, encoding ) );
                        }
                        else
                        {
                            parser.nextText();
                        }
                    }
                }
                else if ( parser.getName().equals( "licenses" )  )
                {
                    if ( parsed.contains( "licenses" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "licenses" );
                    java.util.List licenses = new java.util.ArrayList();
                    model.setLicenses( licenses );
                    while ( parser.nextTag() == XmlPullParser.START_TAG )
                    {
                        if ( parser.getName().equals( "license" ) )
                        {
                            licenses.add( parseLicense( "license", parser, strict, encoding ) );
                        }
                        else
                        {
                            parser.nextText();
                        }
                    }
                }
                else if ( parser.getName().equals( "versions" )  )
                {
                    if ( parsed.contains( "versions" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "versions" );
                    java.util.List versions = new java.util.ArrayList();
                    model.setVersions( versions );
                    while ( parser.nextTag() == XmlPullParser.START_TAG )
                    {
                        if ( parser.getName().equals( "version" ) )
                        {
                            versions.add( parseVersion( "version", parser, strict, encoding ) );
                        }
                        else
                        {
                            parser.nextText();
                        }
                    }
                }
                else if ( parser.getName().equals( "branches" )  )
                {
                    if ( parsed.contains( "branches" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "branches" );
                    java.util.List branches = new java.util.ArrayList();
                    model.setBranches( branches );
                    while ( parser.nextTag() == XmlPullParser.START_TAG )
                    {
                        if ( parser.getName().equals( "branch" ) )
                        {
                            branches.add( parseBranch( "branch", parser, strict, encoding ) );
                        }
                        else
                        {
                            parser.nextText();
                        }
                    }
                }
                else if ( parser.getName().equals( "packageGroups" )  )
                {
                    if ( parsed.contains( "packageGroups" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "packageGroups" );
                    java.util.List packageGroups = new java.util.ArrayList();
                    model.setPackageGroups( packageGroups );
                    while ( parser.nextTag() == XmlPullParser.START_TAG )
                    {
                        if ( parser.getName().equals( "packageGroup" ) )
                        {
                            packageGroups.add( parsePackageGroup( "packageGroup", parser, strict, encoding ) );
                        }
                        else
                        {
                            parser.nextText();
                        }
                    }
                }
                else if ( parser.getName().equals( "reports" )  )
                {
                    if ( parsed.contains( "reports" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "reports" );
                    java.util.List reports = new java.util.ArrayList();
                    model.setReports( reports );
                    while ( parser.nextTag() == XmlPullParser.START_TAG )
                    {
                        if ( parser.getName().equals( "report" ) )
                        {
                            reports.add( getTrimmedValue( parser.nextText()) );
                        }
                        else
                        {
                            parser.nextText();
                        }
                    }
                }
                else if ( parser.getName().equals( "repository" )  )
                {
                    if ( parsed.contains( "repository" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "repository" );
                    model.setRepository( parseRepository( "repository", parser, strict, encoding ) );
                }
                else if ( parser.getName().equals( "organization" ) || parser.getName().equals( "organisation" )  )
                {
                    if ( parsed.contains( "organization" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "organization" );
                    model.setOrganization( parseOrganization( "organization", parser, strict, encoding ) );
                }
                else if ( parser.getName().equals( "properties" )  )
                {
                    if ( parsed.contains( "properties" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "properties" );
                    while ( parser.nextTag() == XmlPullParser.START_TAG )
                    {
                        String key = parser.getName();
                        String value = parser.nextText()
                        .trim()
                        ;
                        model.addProperty( key, value );
                    }
                }
                else if ( parser.getName().equals( "package" )  )
                {
                    if ( parsed.contains( "package" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "package" );
                    model.setPackageName( getTrimmedValue( parser.nextText()) );
                }
                else if ( parser.getName().equals( "build" )  )
                {
                    if ( parsed.contains( "build" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "build" );
                    model.setBuild( parseBuild( "build", parser, strict, encoding ) );
                }
                else if ( parser.getName().equals( "dependencies" )  )
                {
                    if ( parsed.contains( "dependencies" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                    }
                    parsed.add( "dependencies" );
                    java.util.List dependencies = new java.util.ArrayList();
                    model.setDependencies( dependencies );
                    while ( parser.nextTag() == XmlPullParser.START_TAG )
                    {
                        if ( parser.getName().equals( "dependency" ) )
                        {
                            dependencies.add( parseDependency( "dependency", parser, strict, encoding ) );
                        }
                        else
                        {
                            parser.nextText();
                        }
                    }
                }
                else if ( !foundRoot && strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null);
                }
            }
            eventType = parser.next();
        }
        return model;
    } //-- Model parseModel(String, XmlPullParser, boolean, String) 

    /**
     * Method parseModelBase
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param parser
     */
    private ModelBase parseModelBase(String tagName, XmlPullParser parser, boolean strict, String encoding)
        throws IOException, XmlPullParserException
    {
        ModelBase modelBase = new ModelBase();
        modelBase.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "dependencies" )  )
            {
                if ( parsed.contains( "dependencies" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "dependencies" );
                java.util.List dependencies = new java.util.ArrayList();
                modelBase.setDependencies( dependencies );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    if ( parser.getName().equals( "dependency" ) )
                    {
                        dependencies.add( parseDependency( "dependency", parser, strict, encoding ) );
                    }
                    else
                    {
                        parser.nextText();
                    }
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null);
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return modelBase;
    } //-- ModelBase parseModelBase(String, XmlPullParser, boolean, String) 

    /**
     * Method parseOrganization
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param parser
     */
    private Organization parseOrganization(String tagName, XmlPullParser parser, boolean strict, String encoding)
        throws IOException, XmlPullParserException
    {
        Organization organization = new Organization();
        organization.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "name" )  )
            {
                if ( parsed.contains( "name" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "name" );
                organization.setName( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "url" )  )
            {
                if ( parsed.contains( "url" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "url" );
                organization.setUrl( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "logo" )  )
            {
                if ( parsed.contains( "logo" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "logo" );
                organization.setLogo( getTrimmedValue( parser.nextText()) );
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null);
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return organization;
    } //-- Organization parseOrganization(String, XmlPullParser, boolean, String) 

    /**
     * Method parsePackageGroup
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param parser
     */
    private PackageGroup parsePackageGroup(String tagName, XmlPullParser parser, boolean strict, String encoding)
        throws IOException, XmlPullParserException
    {
        PackageGroup packageGroup = new PackageGroup();
        packageGroup.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "title" )  )
            {
                if ( parsed.contains( "title" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "title" );
                packageGroup.setTitle( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "packages" )  )
            {
                if ( parsed.contains( "packages" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "packages" );
                packageGroup.setPackages( getTrimmedValue( parser.nextText()) );
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null);
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return packageGroup;
    } //-- PackageGroup parsePackageGroup(String, XmlPullParser, boolean, String) 

    /**
     * Method parsePatternSet
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param parser
     */
    private PatternSet parsePatternSet(String tagName, XmlPullParser parser, boolean strict, String encoding)
        throws IOException, XmlPullParserException
    {
        PatternSet patternSet = new PatternSet();
        patternSet.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "includes" )  )
            {
                if ( parsed.contains( "includes" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "includes" );
                java.util.List includes = new java.util.ArrayList();
                patternSet.setIncludes( includes );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    if ( parser.getName().equals( "include" ) )
                    {
                        includes.add( getTrimmedValue( parser.nextText()) );
                    }
                    else
                    {
                        parser.nextText();
                    }
                }
            }
            else if ( parser.getName().equals( "excludes" )  )
            {
                if ( parsed.contains( "excludes" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "excludes" );
                java.util.List excludes = new java.util.ArrayList();
                patternSet.setExcludes( excludes );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    if ( parser.getName().equals( "exclude" ) )
                    {
                        excludes.add( getTrimmedValue( parser.nextText()) );
                    }
                    else
                    {
                        parser.nextText();
                    }
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null);
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return patternSet;
    } //-- PatternSet parsePatternSet(String, XmlPullParser, boolean, String) 

    /**
     * Method parsePluginConfiguration
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param parser
     */
    private PluginConfiguration parsePluginConfiguration(String tagName, XmlPullParser parser, boolean strict, String encoding)
        throws IOException, XmlPullParserException
    {
        PluginConfiguration pluginConfiguration = new PluginConfiguration();
        pluginConfiguration.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( strict )
            {
                throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null);
            }
            else
            {
                // swallow up to end tag since this is not valid
                while ( parser.next() != XmlPullParser.END_TAG ) {}
            }
        }
        return pluginConfiguration;
    } //-- PluginConfiguration parsePluginConfiguration(String, XmlPullParser, boolean, String) 

    /**
     * Method parsePluginContainer
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param parser
     */
    private PluginContainer parsePluginContainer(String tagName, XmlPullParser parser, boolean strict, String encoding)
        throws IOException, XmlPullParserException
    {
        PluginContainer pluginContainer = new PluginContainer();
        pluginContainer.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( strict )
            {
                throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null);
            }
            else
            {
                // swallow up to end tag since this is not valid
                while ( parser.next() != XmlPullParser.END_TAG ) {}
            }
        }
        return pluginContainer;
    } //-- PluginContainer parsePluginContainer(String, XmlPullParser, boolean, String) 

    /**
     * Method parseRepository
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param parser
     */
    private Repository parseRepository(String tagName, XmlPullParser parser, boolean strict, String encoding)
        throws IOException, XmlPullParserException
    {
        Repository repository = new Repository();
        repository.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "connection" )  )
            {
                if ( parsed.contains( "connection" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "connection" );
                repository.setConnection( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "developerConnection" )  )
            {
                if ( parsed.contains( "developerConnection" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "developerConnection" );
                repository.setDeveloperConnection( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "url" )  )
            {
                if ( parsed.contains( "url" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "url" );
                repository.setUrl( getTrimmedValue( parser.nextText()) );
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null);
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return repository;
    } //-- Repository parseRepository(String, XmlPullParser, boolean, String) 

    /**
     * Method parseResource
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param parser
     */
    private Resource parseResource(String tagName, XmlPullParser parser, boolean strict, String encoding)
        throws IOException, XmlPullParserException
    {
        Resource resource = new Resource();
        resource.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "targetPath" )  )
            {
                if ( parsed.contains( "targetPath" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "targetPath" );
                resource.setTargetPath( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "filtering" )  )
            {
                if ( parsed.contains( "filtering" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "filtering" );
                resource.setFiltering( getBooleanValue( getTrimmedValue( parser.nextText()), "filtering", parser ) );
            }
            else if ( parser.getName().equals( "directory" )  )
            {
                if ( parsed.contains( "directory" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "directory" );
                resource.setDirectory( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "includes" )  )
            {
                if ( parsed.contains( "includes" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "includes" );
                java.util.List includes = new java.util.ArrayList();
                resource.setIncludes( includes );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    if ( parser.getName().equals( "include" ) )
                    {
                        includes.add( getTrimmedValue( parser.nextText()) );
                    }
                    else
                    {
                        parser.nextText();
                    }
                }
            }
            else if ( parser.getName().equals( "excludes" )  )
            {
                if ( parsed.contains( "excludes" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "excludes" );
                java.util.List excludes = new java.util.ArrayList();
                resource.setExcludes( excludes );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    if ( parser.getName().equals( "exclude" ) )
                    {
                        excludes.add( getTrimmedValue( parser.nextText()) );
                    }
                    else
                    {
                        parser.nextText();
                    }
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null);
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return resource;
    } //-- Resource parseResource(String, XmlPullParser, boolean, String) 

    /**
     * Method parseSourceModification
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param parser
     */
    private SourceModification parseSourceModification(String tagName, XmlPullParser parser, boolean strict, String encoding)
        throws IOException, XmlPullParserException
    {
        SourceModification sourceModification = new SourceModification();
        sourceModification.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "className" )  )
            {
                if ( parsed.contains( "className" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "className" );
                sourceModification.setClassName( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "property" )  )
            {
                if ( parsed.contains( "property" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "property" );
                sourceModification.setProperty( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "directory" )  )
            {
                if ( parsed.contains( "directory" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "directory" );
                sourceModification.setDirectory( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "includes" )  )
            {
                if ( parsed.contains( "includes" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "includes" );
                java.util.List includes = new java.util.ArrayList();
                sourceModification.setIncludes( includes );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    if ( parser.getName().equals( "include" ) )
                    {
                        includes.add( getTrimmedValue( parser.nextText()) );
                    }
                    else
                    {
                        parser.nextText();
                    }
                }
            }
            else if ( parser.getName().equals( "excludes" )  )
            {
                if ( parsed.contains( "excludes" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "excludes" );
                java.util.List excludes = new java.util.ArrayList();
                sourceModification.setExcludes( excludes );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    if ( parser.getName().equals( "exclude" ) )
                    {
                        excludes.add( getTrimmedValue( parser.nextText()) );
                    }
                    else
                    {
                        parser.nextText();
                    }
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null);
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return sourceModification;
    } //-- SourceModification parseSourceModification(String, XmlPullParser, boolean, String) 

    /**
     * Method parseUnitTest
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param parser
     */
    private UnitTest parseUnitTest(String tagName, XmlPullParser parser, boolean strict, String encoding)
        throws IOException, XmlPullParserException
    {
        UnitTest unitTest = new UnitTest();
        unitTest.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "resources" )  )
            {
                if ( parsed.contains( "resources" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "resources" );
                java.util.List resources = new java.util.ArrayList();
                unitTest.setResources( resources );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    if ( parser.getName().equals( "resource" ) )
                    {
                        resources.add( parseResource( "resource", parser, strict, encoding ) );
                    }
                    else
                    {
                        parser.nextText();
                    }
                }
            }
            else if ( parser.getName().equals( "includes" )  )
            {
                if ( parsed.contains( "includes" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "includes" );
                java.util.List includes = new java.util.ArrayList();
                unitTest.setIncludes( includes );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    if ( parser.getName().equals( "include" ) )
                    {
                        includes.add( getTrimmedValue( parser.nextText()) );
                    }
                    else
                    {
                        parser.nextText();
                    }
                }
            }
            else if ( parser.getName().equals( "excludes" )  )
            {
                if ( parsed.contains( "excludes" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "excludes" );
                java.util.List excludes = new java.util.ArrayList();
                unitTest.setExcludes( excludes );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    if ( parser.getName().equals( "exclude" ) )
                    {
                        excludes.add( getTrimmedValue( parser.nextText()) );
                    }
                    else
                    {
                        parser.nextText();
                    }
                }
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null);
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return unitTest;
    } //-- UnitTest parseUnitTest(String, XmlPullParser, boolean, String) 

    /**
     * Method parseVersion
     * 
     * @param tagName
     * @param encoding
     * @param strict
     * @param parser
     */
    private Version parseVersion(String tagName, XmlPullParser parser, boolean strict, String encoding)
        throws IOException, XmlPullParserException
    {
        Version version = new Version();
        version.setModelEncoding( encoding );
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "name" )  )
            {
                if ( parsed.contains( "name" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "name" );
                version.setName( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "tag" )  )
            {
                if ( parsed.contains( "tag" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "tag" );
                version.setTag( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "id" )  )
            {
                if ( parsed.contains( "id" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null);
                }
                parsed.add( "id" );
                version.setId( getTrimmedValue( parser.nextText()) );
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null);
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return version;
    } //-- Version parseVersion(String, XmlPullParser, boolean, String) 

    /**
     * Method read
     * 
     * @param reader
     * @param strict
     */
    public Model read(Reader reader, boolean strict)
        throws IOException, XmlPullParserException
    {
        XmlPullParser parser = new MXParser();
        
        parser.setInput( reader );
        
        if ( addDefaultEntities ) 
        {
            // ----------------------------------------------------------------------
            // Latin 1 entities
            // ----------------------------------------------------------------------
            
            parser.defineEntityReplacementText( "nbsp", "\u00a0" ); 
            parser.defineEntityReplacementText( "iexcl", "\u00a1" ); 
            parser.defineEntityReplacementText( "cent", "\u00a2" ); 
            parser.defineEntityReplacementText( "pound", "\u00a3" ); 
            parser.defineEntityReplacementText( "curren", "\u00a4" ); 
            parser.defineEntityReplacementText( "yen", "\u00a5" ); 
            parser.defineEntityReplacementText( "brvbar", "\u00a6" ); 
            parser.defineEntityReplacementText( "sect", "\u00a7" ); 
            parser.defineEntityReplacementText( "uml", "\u00a8" ); 
            parser.defineEntityReplacementText( "copy", "\u00a9" ); 
            parser.defineEntityReplacementText( "ordf", "\u00aa" ); 
            parser.defineEntityReplacementText( "laquo", "\u00ab" ); 
            parser.defineEntityReplacementText( "not", "\u00ac" ); 
            parser.defineEntityReplacementText( "shy", "\u00ad" ); 
            parser.defineEntityReplacementText( "reg", "\u00ae" ); 
            parser.defineEntityReplacementText( "macr", "\u00af" ); 
            parser.defineEntityReplacementText( "deg", "\u00b0" ); 
            parser.defineEntityReplacementText( "plusmn", "\u00b1" ); 
            parser.defineEntityReplacementText( "sup2", "\u00b2" ); 
            parser.defineEntityReplacementText( "sup3", "\u00b3" ); 
            parser.defineEntityReplacementText( "acute", "\u00b4" ); 
            parser.defineEntityReplacementText( "micro", "\u00b5" ); 
            parser.defineEntityReplacementText( "para", "\u00b6" ); 
            parser.defineEntityReplacementText( "middot", "\u00b7" ); 
            parser.defineEntityReplacementText( "cedil", "\u00b8" ); 
            parser.defineEntityReplacementText( "sup1", "\u00b9" ); 
            parser.defineEntityReplacementText( "ordm", "\u00ba" ); 
            parser.defineEntityReplacementText( "raquo", "\u00bb" ); 
            parser.defineEntityReplacementText( "frac14", "\u00bc" ); 
            parser.defineEntityReplacementText( "frac12", "\u00bd" ); 
            parser.defineEntityReplacementText( "frac34", "\u00be" ); 
            parser.defineEntityReplacementText( "iquest", "\u00bf" ); 
            parser.defineEntityReplacementText( "Agrave", "\u00c0" ); 
            parser.defineEntityReplacementText( "Aacute", "\u00c1" ); 
            parser.defineEntityReplacementText( "Acirc", "\u00c2" ); 
            parser.defineEntityReplacementText( "Atilde", "\u00c3" ); 
            parser.defineEntityReplacementText( "Auml", "\u00c4" ); 
            parser.defineEntityReplacementText( "Aring", "\u00c5" ); 
            parser.defineEntityReplacementText( "AElig", "\u00c6" ); 
            parser.defineEntityReplacementText( "Ccedil", "\u00c7" ); 
            parser.defineEntityReplacementText( "Egrave", "\u00c8" ); 
            parser.defineEntityReplacementText( "Eacute", "\u00c9" ); 
            parser.defineEntityReplacementText( "Ecirc", "\u00ca" ); 
            parser.defineEntityReplacementText( "Euml", "\u00cb" ); 
            parser.defineEntityReplacementText( "Igrave", "\u00cc" ); 
            parser.defineEntityReplacementText( "Iacute", "\u00cd" ); 
            parser.defineEntityReplacementText( "Icirc", "\u00ce" ); 
            parser.defineEntityReplacementText( "Iuml", "\u00cf" ); 
            parser.defineEntityReplacementText( "ETH", "\u00d0" ); 
            parser.defineEntityReplacementText( "Ntilde", "\u00d1" ); 
            parser.defineEntityReplacementText( "Ograve", "\u00d2" ); 
            parser.defineEntityReplacementText( "Oacute", "\u00d3" ); 
            parser.defineEntityReplacementText( "Ocirc", "\u00d4" ); 
            parser.defineEntityReplacementText( "Otilde", "\u00d5" ); 
            parser.defineEntityReplacementText( "Ouml", "\u00d6" ); 
            parser.defineEntityReplacementText( "times", "\u00d7" ); 
            parser.defineEntityReplacementText( "Oslash", "\u00d8" ); 
            parser.defineEntityReplacementText( "Ugrave", "\u00d9" ); 
            parser.defineEntityReplacementText( "Uacute", "\u00da" ); 
            parser.defineEntityReplacementText( "Ucirc", "\u00db" ); 
            parser.defineEntityReplacementText( "Uuml", "\u00dc" ); 
            parser.defineEntityReplacementText( "Yacute", "\u00dd" ); 
            parser.defineEntityReplacementText( "THORN", "\u00de" ); 
            parser.defineEntityReplacementText( "szlig", "\u00df" ); 
            parser.defineEntityReplacementText( "agrave", "\u00e0" ); 
            parser.defineEntityReplacementText( "aacute", "\u00e1" ); 
            parser.defineEntityReplacementText( "acirc", "\u00e2" ); 
            parser.defineEntityReplacementText( "atilde", "\u00e3" ); 
            parser.defineEntityReplacementText( "auml", "\u00e4" ); 
            parser.defineEntityReplacementText( "aring", "\u00e5" ); 
            parser.defineEntityReplacementText( "aelig", "\u00e6" ); 
            parser.defineEntityReplacementText( "ccedil", "\u00e7" ); 
            parser.defineEntityReplacementText( "egrave", "\u00e8" ); 
            parser.defineEntityReplacementText( "eacute", "\u00e9" ); 
            parser.defineEntityReplacementText( "ecirc", "\u00ea" ); 
            parser.defineEntityReplacementText( "euml", "\u00eb" ); 
            parser.defineEntityReplacementText( "igrave", "\u00ec" ); 
            parser.defineEntityReplacementText( "iacute", "\u00ed" ); 
            parser.defineEntityReplacementText( "icirc", "\u00ee" ); 
            parser.defineEntityReplacementText( "iuml", "\u00ef" ); 
            parser.defineEntityReplacementText( "eth", "\u00f0" ); 
            parser.defineEntityReplacementText( "ntilde", "\u00f1" ); 
            parser.defineEntityReplacementText( "ograve", "\u00f2" ); 
            parser.defineEntityReplacementText( "oacute", "\u00f3" ); 
            parser.defineEntityReplacementText( "ocirc", "\u00f4" ); 
            parser.defineEntityReplacementText( "otilde", "\u00f5" ); 
            parser.defineEntityReplacementText( "ouml", "\u00f6" ); 
            parser.defineEntityReplacementText( "divide", "\u00f7" ); 
            parser.defineEntityReplacementText( "oslash", "\u00f8" ); 
            parser.defineEntityReplacementText( "ugrave", "\u00f9" ); 
            parser.defineEntityReplacementText( "uacute", "\u00fa" ); 
            parser.defineEntityReplacementText( "ucirc", "\u00fb" ); 
            parser.defineEntityReplacementText( "uuml", "\u00fc" ); 
            parser.defineEntityReplacementText( "yacute", "\u00fd" ); 
            parser.defineEntityReplacementText( "thorn", "\u00fe" ); 
            parser.defineEntityReplacementText( "yuml", "\u00ff" ); 
            
            // ----------------------------------------------------------------------
            // Special entities
            // ----------------------------------------------------------------------
            
            parser.defineEntityReplacementText( "OElig", "\u0152" ); 
            parser.defineEntityReplacementText( "oelig", "\u0153" ); 
            parser.defineEntityReplacementText( "Scaron", "\u0160" ); 
            parser.defineEntityReplacementText( "scaron", "\u0161" ); 
            parser.defineEntityReplacementText( "Yuml", "\u0178" ); 
            parser.defineEntityReplacementText( "circ", "\u02c6" ); 
            parser.defineEntityReplacementText( "tilde", "\u02dc" ); 
            parser.defineEntityReplacementText( "ensp", "\u2002" ); 
            parser.defineEntityReplacementText( "emsp", "\u2003" ); 
            parser.defineEntityReplacementText( "thinsp", "\u2009" ); 
            parser.defineEntityReplacementText( "zwnj", "\u200c" ); 
            parser.defineEntityReplacementText( "zwj", "\u200d" ); 
            parser.defineEntityReplacementText( "lrm", "\u200e" ); 
            parser.defineEntityReplacementText( "rlm", "\u200f" ); 
            parser.defineEntityReplacementText( "ndash", "\u2013" ); 
            parser.defineEntityReplacementText( "mdash", "\u2014" ); 
            parser.defineEntityReplacementText( "lsquo", "\u2018" ); 
            parser.defineEntityReplacementText( "rsquo", "\u2019" ); 
            parser.defineEntityReplacementText( "sbquo", "\u201a" ); 
            parser.defineEntityReplacementText( "ldquo", "\u201c" ); 
            parser.defineEntityReplacementText( "rdquo", "\u201d" ); 
            parser.defineEntityReplacementText( "bdquo", "\u201e" ); 
            parser.defineEntityReplacementText( "dagger", "\u2020" ); 
            parser.defineEntityReplacementText( "Dagger", "\u2021" ); 
            parser.defineEntityReplacementText( "permil", "\u2030" ); 
            parser.defineEntityReplacementText( "lsaquo", "\u2039" ); 
            parser.defineEntityReplacementText( "rsaquo", "\u203a" ); 
            parser.defineEntityReplacementText( "euro", "\u20ac" ); 
            
            // ----------------------------------------------------------------------
            // Symbol entities
            // ----------------------------------------------------------------------
            
            parser.defineEntityReplacementText( "fnof", "\u0192" ); 
            parser.defineEntityReplacementText( "Alpha", "\u0391" ); 
            parser.defineEntityReplacementText( "Beta", "\u0392" ); 
            parser.defineEntityReplacementText( "Gamma", "\u0393" ); 
            parser.defineEntityReplacementText( "Delta", "\u0394" ); 
            parser.defineEntityReplacementText( "Epsilon", "\u0395" ); 
            parser.defineEntityReplacementText( "Zeta", "\u0396" ); 
            parser.defineEntityReplacementText( "Eta", "\u0397" ); 
            parser.defineEntityReplacementText( "Theta", "\u0398" ); 
            parser.defineEntityReplacementText( "Iota", "\u0399" ); 
            parser.defineEntityReplacementText( "Kappa", "\u039a" ); 
            parser.defineEntityReplacementText( "Lambda", "\u039b" ); 
            parser.defineEntityReplacementText( "Mu", "\u039c" ); 
            parser.defineEntityReplacementText( "Nu", "\u039d" ); 
            parser.defineEntityReplacementText( "Xi", "\u039e" ); 
            parser.defineEntityReplacementText( "Omicron", "\u039f" ); 
            parser.defineEntityReplacementText( "Pi", "\u03a0" ); 
            parser.defineEntityReplacementText( "Rho", "\u03a1" ); 
            parser.defineEntityReplacementText( "Sigma", "\u03a3" ); 
            parser.defineEntityReplacementText( "Tau", "\u03a4" ); 
            parser.defineEntityReplacementText( "Upsilon", "\u03a5" ); 
            parser.defineEntityReplacementText( "Phi", "\u03a6" ); 
            parser.defineEntityReplacementText( "Chi", "\u03a7" ); 
            parser.defineEntityReplacementText( "Psi", "\u03a8" ); 
            parser.defineEntityReplacementText( "Omega", "\u03a9" ); 
            parser.defineEntityReplacementText( "alpha", "\u03b1" ); 
            parser.defineEntityReplacementText( "beta", "\u03b2" ); 
            parser.defineEntityReplacementText( "gamma", "\u03b3" ); 
            parser.defineEntityReplacementText( "delta", "\u03b4" ); 
            parser.defineEntityReplacementText( "epsilon", "\u03b5" ); 
            parser.defineEntityReplacementText( "zeta", "\u03b6" ); 
            parser.defineEntityReplacementText( "eta", "\u03b7" ); 
            parser.defineEntityReplacementText( "theta", "\u03b8" ); 
            parser.defineEntityReplacementText( "iota", "\u03b9" ); 
            parser.defineEntityReplacementText( "kappa", "\u03ba" ); 
            parser.defineEntityReplacementText( "lambda", "\u03bb" ); 
            parser.defineEntityReplacementText( "mu", "\u03bc" ); 
            parser.defineEntityReplacementText( "nu", "\u03bd" ); 
            parser.defineEntityReplacementText( "xi", "\u03be" ); 
            parser.defineEntityReplacementText( "omicron", "\u03bf" ); 
            parser.defineEntityReplacementText( "pi", "\u03c0" ); 
            parser.defineEntityReplacementText( "rho", "\u03c1" ); 
            parser.defineEntityReplacementText( "sigmaf", "\u03c2" ); 
            parser.defineEntityReplacementText( "sigma", "\u03c3" ); 
            parser.defineEntityReplacementText( "tau", "\u03c4" ); 
            parser.defineEntityReplacementText( "upsilon", "\u03c5" ); 
            parser.defineEntityReplacementText( "phi", "\u03c6" ); 
            parser.defineEntityReplacementText( "chi", "\u03c7" ); 
            parser.defineEntityReplacementText( "psi", "\u03c8" ); 
            parser.defineEntityReplacementText( "omega", "\u03c9" ); 
            parser.defineEntityReplacementText( "thetasym", "\u03d1" ); 
            parser.defineEntityReplacementText( "upsih", "\u03d2" ); 
            parser.defineEntityReplacementText( "piv", "\u03d6" ); 
            parser.defineEntityReplacementText( "bull", "\u2022" ); 
            parser.defineEntityReplacementText( "hellip", "\u2026" ); 
            parser.defineEntityReplacementText( "prime", "\u2032" ); 
            parser.defineEntityReplacementText( "Prime", "\u2033" ); 
            parser.defineEntityReplacementText( "oline", "\u203e" ); 
            parser.defineEntityReplacementText( "frasl", "\u2044" ); 
            parser.defineEntityReplacementText( "weierp", "\u2118" ); 
            parser.defineEntityReplacementText( "image", "\u2111" ); 
            parser.defineEntityReplacementText( "real", "\u211c" ); 
            parser.defineEntityReplacementText( "trade", "\u2122" ); 
            parser.defineEntityReplacementText( "alefsym", "\u2135" ); 
            parser.defineEntityReplacementText( "larr", "\u2190" ); 
            parser.defineEntityReplacementText( "uarr", "\u2191" ); 
            parser.defineEntityReplacementText( "rarr", "\u2192" ); 
            parser.defineEntityReplacementText( "darr", "\u2193" ); 
            parser.defineEntityReplacementText( "harr", "\u2194" ); 
            parser.defineEntityReplacementText( "crarr", "\u21b5" ); 
            parser.defineEntityReplacementText( "lArr", "\u21d0" ); 
            parser.defineEntityReplacementText( "uArr", "\u21d1" ); 
            parser.defineEntityReplacementText( "rArr", "\u21d2" ); 
            parser.defineEntityReplacementText( "dArr", "\u21d3" ); 
            parser.defineEntityReplacementText( "hArr", "\u21d4" ); 
            parser.defineEntityReplacementText( "forall", "\u2200" ); 
            parser.defineEntityReplacementText( "part", "\u2202" ); 
            parser.defineEntityReplacementText( "exist", "\u2203" ); 
            parser.defineEntityReplacementText( "empty", "\u2205" ); 
            parser.defineEntityReplacementText( "nabla", "\u2207" ); 
            parser.defineEntityReplacementText( "isin", "\u2208" ); 
            parser.defineEntityReplacementText( "notin", "\u2209" ); 
            parser.defineEntityReplacementText( "ni", "\u220b" ); 
            parser.defineEntityReplacementText( "prod", "\u220f" ); 
            parser.defineEntityReplacementText( "sum", "\u2211" ); 
            parser.defineEntityReplacementText( "minus", "\u2212" ); 
            parser.defineEntityReplacementText( "lowast", "\u2217" ); 
            parser.defineEntityReplacementText( "radic", "\u221a" ); 
            parser.defineEntityReplacementText( "prop", "\u221d" ); 
            parser.defineEntityReplacementText( "infin", "\u221e" ); 
            parser.defineEntityReplacementText( "ang", "\u2220" ); 
            parser.defineEntityReplacementText( "and", "\u2227" ); 
            parser.defineEntityReplacementText( "or", "\u2228" ); 
            parser.defineEntityReplacementText( "cap", "\u2229" ); 
            parser.defineEntityReplacementText( "cup", "\u222a" ); 
            parser.defineEntityReplacementText( "int", "\u222b" ); 
            parser.defineEntityReplacementText( "there4", "\u2234" ); 
            parser.defineEntityReplacementText( "sim", "\u223c" ); 
            parser.defineEntityReplacementText( "cong", "\u2245" ); 
            parser.defineEntityReplacementText( "asymp", "\u2248" ); 
            parser.defineEntityReplacementText( "ne", "\u2260" ); 
            parser.defineEntityReplacementText( "equiv", "\u2261" ); 
            parser.defineEntityReplacementText( "le", "\u2264" ); 
            parser.defineEntityReplacementText( "ge", "\u2265" ); 
            parser.defineEntityReplacementText( "sub", "\u2282" ); 
            parser.defineEntityReplacementText( "sup", "\u2283" ); 
            parser.defineEntityReplacementText( "nsub", "\u2284" ); 
            parser.defineEntityReplacementText( "sube", "\u2286" ); 
            parser.defineEntityReplacementText( "supe", "\u2287" ); 
            parser.defineEntityReplacementText( "oplus", "\u2295" ); 
            parser.defineEntityReplacementText( "otimes", "\u2297" ); 
            parser.defineEntityReplacementText( "perp", "\u22a5" ); 
            parser.defineEntityReplacementText( "sdot", "\u22c5" ); 
            parser.defineEntityReplacementText( "lceil", "\u2308" ); 
            parser.defineEntityReplacementText( "rceil", "\u2309" ); 
            parser.defineEntityReplacementText( "lfloor", "\u230a" ); 
            parser.defineEntityReplacementText( "rfloor", "\u230b" ); 
            parser.defineEntityReplacementText( "lang", "\u2329" ); 
            parser.defineEntityReplacementText( "rang", "\u232a" ); 
            parser.defineEntityReplacementText( "loz", "\u25ca" ); 
            parser.defineEntityReplacementText( "spades", "\u2660" ); 
            parser.defineEntityReplacementText( "clubs", "\u2663" ); 
            parser.defineEntityReplacementText( "hearts", "\u2665" ); 
            parser.defineEntityReplacementText( "diams", "\u2666" ); 
            
        }
        
        parser.next();
        String encoding = parser.getInputEncoding();
        
        return parseModel( "project", parser, strict, encoding );
    } //-- Model read(Reader, boolean) 

    /**
     * Method read
     * 
     * @param reader
     */
    public Model read(Reader reader)
        throws IOException, XmlPullParserException
    {
        return read( reader, true );
    } //-- Model read(Reader) 

    /**
     * Returns the state of the "add default entities" flag.
     * 
     * @param addDefaultEntities
     */
    public void setAddDefaultEntities(boolean addDefaultEntities)
    {
        this.addDefaultEntities = addDefaultEntities;
    } //-- void setAddDefaultEntities(boolean) 


}
