package me.codeboy.tools.io;

import me.codeboy.tools.annotation.CBNotNull;
import me.codeboy.tools.lang.CBString;
import me.codeboy.tools.util.CBDate;

import java.util.List;

/**
 * 默认日志输出器
 *
 * <p>采用<strong>{}</strong>为占位符</p>
 * <p>打印中包含时间、日志级别、线程Id、线程名、类名</p>
 */
final class CBDummyLogger implements CBLog.CBLogger {
    private static final String BLACK = "\033[0;30m";   // BLACK
    private static final String RED = "\033[0;31m";     // RED
    private static final String GREEN = "\033[0;32m";   // GREEN
    private static final String YELLOW = "\033[0;33m";  // YELLOW
    private static final String BLUE = "\033[0;34m";    // BLUE
    private static final String PURPLE = "\033[0;35m";  // PURPLE
    private static final String CYAN = "\033[0;36m";    // CYAN
    private static final String WHITE = "\033[0;37m";   // WHITE
    private static final String RESET = "\033[0m";     // WHITE
    private static final String DEBUG = "DEBUG";
    private static final String INFO = "INFO";
    private static final String WARN = "WARN";
    private static final String ERROR = "ERROR";

    @Override
    public void debug(String msg) {
        log(msg, DEBUG, CYAN);
    }

    @Override
    public void debug(String format, Object... param) {
        debug(format(format, param));
    }

    @Override
    public void debug(String msg, Throwable throwable) {
        debug(msg);
        throwable.printStackTrace();
    }

    @Override
    public void info(String msg) {
        log(msg, INFO, GREEN);
    }

    @Override
    public void info(String format, Object... param) {
        info(format(format, param));
    }

    @Override
    public void info(String msg, Throwable throwable) {
        info(msg);
        throwable.printStackTrace();
    }

    @Override
    public void warn(String msg) {
        log(msg, WARN, YELLOW);
    }

    @Override
    public void warn(String format, Object... param) {
        warn(format(format, param));
    }

    @Override
    public void warn(String msg, Throwable throwable) {
        warn(msg);
        throwable.printStackTrace();
    }

    @Override
    public void error(String msg) {
        log(msg, ERROR, RED);
    }

    @Override
    public void error(String msg, Object... param) {
        error(format(msg, param));
    }

    @Override
    public void error(String msg, Throwable throwable) {
        error(msg);
        throwable.printStackTrace();
    }

    private void log(String log, String level, String color) {
        System.out.printf("%s %s%5s%s %s%5d%s --- [%16s] %32s : %s\n", CBDate.formatTimeWithMs(), color, level, RESET, PURPLE, Thread.currentThread().getId(), RESET, Thread.currentThread().getName(), getInvokeClass(), log);
    }

    private String format(@CBNotNull String format, Object... arguments) {
        List<String> items = CBString.split(format, "{}");
        if (items.size() == 0) {
            return CBString.EMPTY;
        }
        StringBuilder sb = new StringBuilder(items.get(0));
        int paramLen = arguments.length;
        for (int i = 1; i < items.size(); i++) {
            if (paramLen >= i) {
                sb.append(arguments[i - 1]);
            }
            sb.append(items.get(i));
        }
        return sb.toString();
    }

    private String getInvokeClass() {

        // 查找到CBLog的上一级
        StackTraceElement[] stack = Thread.currentThread().getStackTrace();
        boolean find = false;
        String logClass = CBLog.class.getName();
        for (StackTraceElement element : stack) {
            if (!find && !element.getClassName().equals(logClass)) {
                continue;
            }
            find = true;
            if (element.getClassName().equals(logClass)) {
                continue;
            }
            String invokeClass = element.getClassName();
            if (invokeClass.length() <= 32) {
                return invokeClass;
            }
            String[] items = invokeClass.split("\\.");
            int count = 0;
            int len = items.length;
            int position = len - 1;
            for (int i = position; i >= 0; i--) {
                count += items[i].length();
                if ((count + (i - 1) * 2) <= 32) {
                    continue;
                }

                StringBuilder sb = new StringBuilder();
                for (int j = 0; j <= i; j++) {
                    sb.append(items[j].charAt(0));
                    sb.append(".");
                }
                for (int j = i + 1; j < len; j++) {
                    sb.append(items[j]);
                    if (j != len - 1) {
                        sb.append(".");
                    }
                }
                return sb.toString();
            }
            break;
        }
        return CBString.EMPTY;
    }
}
