package me.codeboy.tools.io;

import me.codeboy.tools.lang.CBString;
import me.codeboy.tools.util.CBBase64;

import java.io.*;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.List;

/**
 * 文件操作
 */
public class CBFile {
    private final static Charset DEFAULT_CHARSET = StandardCharsets.UTF_8; //默认编码

    /**
     * 获取工程目录
     *
     * @return dir
     */
    public static String getProjectPath() {
        return new File("").getAbsolutePath();
    }

    /**
     * 获取文件内容
     *
     * @param filePath 源文件路径
     * @return 文件内容
     * @throws IOException io exception
     */
    public static byte[] getContent(String filePath) throws IOException {
        return getContent(new File(filePath));
    }

    /**
     * 获取文件内容
     *
     * @param file 源文件
     * @return 文件内容
     * @throws IOException io exception
     */
    public static byte[] getContent(File file) throws IOException {
        FileInputStream inputStream = new FileInputStream(file);
        byte[] bytes = new byte[inputStream.available()];
        inputStream.read(bytes);
        inputStream.close();
        return bytes;
    }

    /**
     * 获取文件内容
     *
     * @param is 输入流
     * @return 字节数组
     * @throws IOException io exception
     */
    public static byte[] getContent(InputStream is) throws IOException {
        byte[] buffer = new byte[4096];
        int len;
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        while ((len = is.read(buffer)) != -1) {
            outputStream.write(buffer, 0, len);
        }
        outputStream.close();
        is.close();
        return outputStream.toByteArray();
    }

    /**
     * 按照UTF-8编码获取文件内容
     *
     * @param filePath 源文件路径
     * @return 文件内容
     * @throws IOException io exception
     */
    public static String getContentAsString(String filePath) throws IOException {
        return getContentAsString(filePath, DEFAULT_CHARSET);
    }

    /**
     * 按照指定编码获取文件内容
     *
     * @param filePath 源文件路径
     * @param charset  文件编码
     * @return 文件内容
     * @throws IOException io exception
     */
    public static String getContentAsString(String filePath, Charset charset) throws IOException {
        BufferedReader buff = new BufferedReader(new InputStreamReader(
                Files.newInputStream(Paths.get(filePath)), charset));
        String content = getContentAsString(buff);
        buff.close();
        return content;
    }

    /**
     * 按照UTF-8编码获取文件内容
     *
     * @param file 文件
     * @return 文件内容
     * @throws IOException io exception
     */
    public static String getContentAsString(File file) throws IOException {
        return getContentAsString(Files.newInputStream(file.toPath()));
    }

    /**
     * 按照指定编码获取文件内容
     *
     * @param file    源文件
     * @param charset 文件编码
     * @return 文件内容
     * @throws IOException io exception
     */
    public static String getContentAsString(File file, Charset charset) throws IOException {
        BufferedReader buff = new BufferedReader(new InputStreamReader(Files.newInputStream(file.toPath()), charset));
        String content = getContentAsString(buff);
        buff.close();
        return content;
    }


    /**
     * 按照UTF-8编码获取文件内容
     *
     * @param is 输入流
     * @return 文件内容
     * @throws IOException io exception
     */
    public static String getContentAsString(InputStream is) throws IOException {
        BufferedReader buff = new BufferedReader(new InputStreamReader(is, DEFAULT_CHARSET));
        String content = getContentAsString(buff);
        buff.close();
        return content;
    }

    /**
     * 按照指定编码获取文件内容
     *
     * @param is      输入流
     * @param charset 文件编
     * @return 文件内容
     * @throws IOException io exception
     */
    public static String getContentAsString(InputStream is, Charset charset) throws IOException {
        BufferedReader buff = new BufferedReader(new InputStreamReader(is, charset));
        String content = getContentAsString(buff);
        buff.close();
        return content;
    }

    /**
     * 获取BufferedReader对应字符串
     *
     * @param buff bufferedReader
     * @return 文件内容
     * @throws IOException io exception
     */
    private static String getContentAsString(BufferedReader buff) throws IOException {
        String line;
        StringBuilder content = new StringBuilder();
        while ((line = buff.readLine()) != null) {
            content.append('\n');
            content.append(line);
        }
        if (content.length() > 0) {
            return content.substring(1);
        } else {
            return CBString.EMPTY;
        }
    }

    /**
     * 获取文件内容的base64
     *
     * @param filePath 源文件路径
     * @return 文件内容
     * @throws IOException io exception
     */
    public static String getContentAsBase64(String filePath) throws IOException {
        return CBBase64.encodeToString(getContent(filePath));
    }

    /**
     * 获取文件内容的base64
     *
     * @param file 源文件
     * @return 文件内容
     * @throws IOException io exception
     */
    public static String getContentAsBase64(File file) throws IOException {
        return CBBase64.encodeToString(getContent(file));
    }

    /**
     * 按照UTF-8编码获取文件内容,保存到list中
     *
     * @param filePath 源文件路径
     * @return 文件内容
     * @throws IOException io exception
     */
    public static List<String> getContentAsList(String filePath) throws IOException {
        return getContentAsList(filePath, DEFAULT_CHARSET);
    }

    /**
     * 按照指定编码获取文件内容,保存list中
     *
     * @param filePath 源文件路径
     * @param charset  文件编码
     * @return 文件内容
     * @throws IOException io exception
     */
    public static List<String> getContentAsList(String filePath, Charset charset) throws IOException {
        return getContentAsList(new File(filePath), charset);
    }

    /**
     * 按照UTF-8编码获取文件内容,保存到list中
     *
     * @param file 源文件
     * @return 文件内容
     * @throws IOException io exception
     */
    public static List<String> getContentAsList(File file) throws IOException {
        return getContentAsList(file, DEFAULT_CHARSET);
    }

    /**
     * 按照指定编码获取文件内容,保存list中
     *
     * @param file    源文件
     * @param charset 文件编码
     * @return 文件内容
     * @throws IOException io exception
     */
    public static List<String> getContentAsList(File file, Charset charset) throws IOException {
        return Arrays.asList(getContentAsString(Files.newInputStream(file.toPath()), charset).split("\\n"));
    }

    /**
     * 获取对象
     *
     * @param filePath 文件路径
     * @param <T>      对象
     * @return 文件内容
     * @throws IOException            io exception
     * @throws ClassNotFoundException cnf exception
     */
    public static <T> T getContentAsObject(String filePath) throws IOException, ClassNotFoundException {
        return getContentAsObject(new File(filePath));
    }

    /**
     * 获取对象
     *
     * @param file 源文件
     * @param <T>  对象
     * @return 文件内容
     * @throws IOException            io exception
     * @throws ClassNotFoundException cnf exception
     */
    public static <T> T getContentAsObject(File file) throws IOException, ClassNotFoundException {
        ObjectInputStream objectInputStream = new ObjectInputStream(Files.newInputStream(file.toPath()));
        return (T) objectInputStream.readObject();
    }

    /**
     * 将java对象写入文件
     *
     * @param object   对象
     * @param filePath 文件路径
     * @throws IOException io exception
     */
    public static void saveContent(Object object, String filePath) throws IOException {
        saveContent(object, new File(filePath));
    }

    /**
     * 将java对象写入文件
     *
     * @param object 对象
     * @param file   文件
     * @throws IOException io exception
     */
    public static void saveContent(Object object, File file) throws IOException {
        if (object == null) {
            return;
        }
        ObjectOutputStream objectOutputStream = new ObjectOutputStream(Files.newOutputStream(file.toPath()));
        objectOutputStream.writeObject(object);
        objectOutputStream.close();
    }


    /**
     * 将指定内容写入文件
     *
     * @param bytes    字节流
     * @param filePath 文件路径
     * @throws IOException io exception
     */
    public static void saveContent(byte[] bytes, String filePath) throws IOException {
        saveContent(bytes, new File(filePath));
    }

    /**
     * 将指定内容写入文件
     *
     * @param bytes 字节流
     * @param file  文件
     * @throws IOException io exception
     */
    public static void saveContent(byte[] bytes, File file) throws IOException {
        FileOutputStream fos = new FileOutputStream(file);
        fos.write(bytes);
        fos.flush();
        fos.close();
    }

    /**
     * 将输入流写入文件
     *
     * @param is       输入流
     * @param filePath 文件路径
     * @throws IOException io exception
     */
    public static void saveContent(InputStream is, String filePath) throws IOException {
        saveContent(is, new File(filePath));
    }

    /**
     * 将输入流写入文件
     *
     * @param is   输入流
     * @param file 文件
     * @throws IOException io exception
     */
    public static void saveContent(InputStream is, File file) throws IOException {
        Files.copy(is, file.toPath());
    }

    /**
     * 按照UTF-8编码保存到相应内容到指定路径的文件
     *
     * @param content  文件内容
     * @param filePath 文件路径
     * @throws IOException io exception
     */
    public static void saveContent(String content, String filePath) throws IOException {
        saveContent(content, filePath, DEFAULT_CHARSET);
    }

    /**
     * 按照UTF-8编码保存到相应内容到指定文件
     *
     * @param content 文件内容
     * @param file    文件
     * @throws IOException io exception
     */
    public static void saveContent(String content, File file) throws IOException {
        saveContent(content, file, DEFAULT_CHARSET, false);
    }

    /**
     * 按照指定编码保存相应内容到指定路径的文件
     *
     * @param content  文件内容
     * @param filePath 文件路径
     * @param charset  编码
     * @throws IOException io exception
     */
    public static void saveContent(String content, String filePath, Charset charset) throws IOException {
        saveContent(content, filePath, charset, false);
    }

    /**
     * 按照指定编码保存相应内容到指定路径的文件
     *
     * @param content 文件内容
     * @param file    文件
     * @param charset 编码
     * @throws IOException io exception
     */
    public static void saveContent(String content, File file, Charset charset) throws IOException {
        saveContent(content, file, charset, false);
    }

    /**
     * 按照UTF-8编码将相应内容保存到指定路径的文件
     *
     * @param content  文件内容
     * @param filePath 文件路径
     * @param append   是否追加
     * @throws IOException io exception
     */
    public static void saveContent(String content, String filePath, boolean append) throws IOException {
        saveContent(content, new File(filePath), DEFAULT_CHARSET, append);
    }

    /**
     * 按照UTF-8编码将相应内容保存到指定文件
     *
     * @param content 文件内容
     * @param file    文件
     * @param append  是否追加
     * @throws IOException io exception
     */
    public static void saveContent(String content, File file, boolean append) throws IOException {
        saveContent(content, file, DEFAULT_CHARSET, append);
    }


    /**
     * 按照指定编码保存相应内容到指定文件
     *
     * @param content  文件内容
     * @param filePath 文件路径
     * @param charset  编码
     * @param append   是否追加
     * @throws IOException io exception
     */
    public static void saveContent(String content, String filePath, Charset charset, boolean append) throws IOException {
        saveContent(content, new File(filePath), charset, append);
    }

    /**
     * 按照指定编码保存相应内容到指定文件
     *
     * @param content 文件内容
     * @param file    文件
     * @param charset 编码
     * @param append  是否追加
     * @throws IOException io exception
     */
    public static void saveContent(String content, File file, Charset charset, boolean append) throws IOException {
        BufferedWriter buff = new BufferedWriter(new OutputStreamWriter(
                new FileOutputStream(file, append), charset));
        buff.write(content);
        buff.flush();
        buff.close();
    }

    /**
     * 保存网络文件到指定路径
     *
     * @param url      url地址
     * @param filePath 文件路径
     * @throws IOException io exception
     */
    public static void download(URL url, String filePath) throws IOException {
        download(url, new File(filePath));
    }

    /**
     * 保存网络文件到指定文件
     *
     * @param url  url地址
     * @param file 文件
     * @throws IOException io exception
     */
    public static void download(URL url, File file) throws IOException {
        InputStream is = url.openStream();
        saveContent(is, file);
        is.close();
    }
}