package me.codeboy.tools.net.core;

import me.codeboy.tools.annotation.CBNotNull;
import me.codeboy.tools.io.CBLog;
import me.codeboy.tools.net.exception.CBNetworkException;
import me.codeboy.tools.net.util.CBParam;
import me.codeboy.tools.lang.CBString;

import java.io.File;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * 请求数据
 */
public class CBBody {
    private String data = CBString.EMPTY;
    private Map<String, String> multipartFields;
    private Map<String, ? extends File> multipartFiles;
    private Charset charset = StandardCharsets.UTF_8;
    private CBContentType contentType = CBContentType.FORM;

    /**
     * 设置请求数据
     *
     * @param data 表单数据
     * @return 连接
     */
    public static CBBody formRaw(String data) {
        return formRaw(StandardCharsets.UTF_8, data);
    }

    /**
     * 设置请求数据
     *
     * @param charset 字符编码
     * @param data    表单数据
     * @return 连接
     */
    public static CBBody formRaw(Charset charset, String data) {
        CBBody body = new CBBody();
        body.charset = charset;
        body.contentType = CBContentType.FORM;
        body.data = data;
        return body;
    }


    /**
     * 设置请求数据
     *
     * @param data 表单数据
     * @return 连接
     */
    public static CBBody form(String data) {
        return form(StandardCharsets.UTF_8, data);
    }

    /**
     * 设置请求数据
     *
     * @param charset 字符编码
     * @param data    表单数据, 如a=1&amp;b=2
     * @return 连接
     */
    public static CBBody form(Charset charset, @CBNotNull String data) {
        String[] items = data.split("&");
        Map<String, String> params = new LinkedHashMap<>();
        for (String item : items) {
            int equalPos = item.indexOf("=");
            if (equalPos == -1) {
                CBLog.warn("body param doesn't correct!!");
                continue;
            }
            params.put(item.substring(0, equalPos), item.substring(equalPos + 1));
        }
        return form(charset, params);
    }

    /**
     * 设置请求数据
     *
     * @param data 表单数据
     * @return 连接
     */
    public static CBBody form(String... data) {
        return form(StandardCharsets.UTF_8, data);
    }

    /**
     * 设置请求数据
     *
     * @param data 表单数据
     * @return 连接
     */
    public static CBBody form(Map<String, String> data) {
        return form(StandardCharsets.UTF_8, data);
    }

    /**
     * 设置请求数据
     *
     * @param charset 字符编码
     * @param data    表单数据，"a","1","b","2"
     * @return 连接
     */
    public static CBBody form(Charset charset, String... data) {
        CBBody body = new CBBody();
        body.charset = charset;
        body.contentType = CBContentType.FORM;
        if (data.length % 2 == 0) {
            body.data = CBParam.combine(body.charset, data);
        }
        return body;
    }

    /**
     * 设置请求数据
     *
     * @param charset 字符编码
     * @param data    表单数据
     * @return 连接
     */
    public static CBBody form(Charset charset, Map<String, String> data) {
        CBBody body = new CBBody();
        body.charset = charset;
        body.contentType = CBContentType.FORM;
        if (data != null) {
            body.data = CBParam.combine(body.charset, data);
        }
        return body;
    }

    /**
     * 设置请求数据
     *
     * @param fields 表单数据
     * @param files  文件
     * @return 连接
     */
    public static CBBody multipartForm(Map<String, String> fields, Map<String, ? extends File> files) {
        CBBody body = new CBBody();
        body.contentType = CBContentType.MULTIPART_FORM;
        body.multipartFields = fields;
        body.multipartFiles = files;
        return body;
    }

    /**
     * 设置请求json数据
     *
     * @param data json data
     * @return 连接
     */
    public static CBBody json(String data) {
        CBBody body = new CBBody();
        body.contentType = CBContentType.JSON;
        if (body.data != null) {
            body.data = data;
        }
        return body;
    }

    @Override
    public String toString() {
        return data;
    }

    /**
     * 非multipartForm下的数据
     *
     * @return 拼接后的字符串
     */
    protected String body() {
        if (contentType == CBContentType.MULTIPART_FORM) {
            throw new CBNetworkException("get call body method when the content-type is multipart/form-data ");
        }
        return data;
    }

    @CBNotNull
    protected Map<String, String> multipartFields() {
        if (multipartFields == null) {
            multipartFields = new LinkedHashMap<>();
        }
        return multipartFields;
    }

    @CBNotNull
    protected Map<String, ? extends File> multipartFiles() {
        if (multipartFiles == null) {
            multipartFiles = new LinkedHashMap<>();
        }
        return multipartFiles;
    }

    protected CBContentType contentType() {
        return contentType;
    }

    protected String boundary() {
        return contentType.boundary();
    }
}
