package me.codeboy.tools.net.core;

import me.codeboy.tools.annotation.CBNotNull;
import me.codeboy.tools.io.CBLog;
import me.codeboy.tools.lang.CBString;
import me.codeboy.tools.net.exception.CBNetworkException;
import me.codeboy.tools.net.util.CBCookieUtil;

import java.io.*;
import java.lang.reflect.Field;
import java.net.*;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.*;

import static me.codeboy.tools.net.core.CBHeader.*;

/**
 * 网络操作，可以获取网页源代码，保存网络文件
 * 支持form、multipartForm、json模式请求
 */
public abstract class CBConnection {

    public final static String HEADER_MOCK_PC_UA = "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/101.0.4951.41 Safari/537.36 Edg/101.0.1210.32"; //pc user agent

    private final static int DEFAULT_TIMEOUT = 30000; //30s
    private String url = null; //请求地址
    private int timeout = DEFAULT_TIMEOUT; //超时时间,默认30s
    private CBMethod method = CBMethod.GET; //请求方式,默认get
    private CBBody body = null; //请求数据
    private Map<String, String> headers = new HashMap<>(); //头部
    private Charset charset = StandardCharsets.UTF_8; //编码
    private final static CookieManager cookieManager = new CookieManager(null, CookiePolicy.ACCEPT_ALL); // cookie
    private static final String LINE_FEED = "\r\n";
    private boolean executed = false;
    private boolean allowCustomOrigin = false;
    private boolean useMemoryCookie = false; // 默认cookie记录在内存，但不会使用
    private final Map<String, String> cookies = new LinkedHashMap<>(); //存放cookie

    /**
     * 连接地址
     *
     * @param url url
     * @return 连接
     */
    public CBConnection head(String url) {
        return get(url, charset);
    }

    /**
     * options请求
     *
     * @param url url
     * @return 连接
     */
    public CBConnection options(String url) {
        return method(CBMethod.OPTIONS, url);
    }

    /**
     * put请求
     *
     * @param url url
     * @return 连接
     */
    public CBConnection put(String url) {
        return method(CBMethod.PUT, url);
    }

    /**
     * delete请求
     *
     * @param url url
     * @return 连接
     */
    public CBConnection delete(String url) {
        return method(CBMethod.DELETE, url);
    }

    /**
     * get请求
     *
     * @param url url
     * @return 连接
     */
    public CBConnection get(String url) {
        return get(url, charset);
    }

    /**
     * get请求
     *
     * @param url     url
     * @param charset 编码
     * @return 连接
     */
    public CBConnection get(String url, Charset charset) {
        return connection(CBMethod.GET, url, charset);
    }

    /**
     * post请求
     *
     * @param url url
     * @return 连接
     */
    public CBConnection post(String url) {
        return post(url, charset);
    }

    /**
     * 连接地址
     *
     * @param url     url
     * @param charset 编码
     * @return 连接
     */
    public CBConnection post(String url, Charset charset) {
        return connection(CBMethod.POST, url, charset);
    }


    /**
     * 按照指定请求方法
     *
     * @param method 方法
     * @param url    url
     * @return 连接
     */
    public CBConnection method(CBMethod method, String url) {
        return connection(method, url, charset);
    }

    /**
     * 连接地址
     *
     * @param method  请求方法
     * @param url     url
     * @param charset 编码
     * @return 连接
     */
    protected CBConnection connection(CBMethod method, String url, Charset charset) {
        this.method = method;
        this.url = url;
        this.charset = charset;
        return this;
    }

    /**
     * 设置请求头部中的cookie
     *
     * @param cookie cookie
     * @return 连接
     */
    public CBConnection cookie(String cookie) {
        cookie(CBCookieUtil.parse(cookie));
        return this;
    }

    /**
     * 设置请求头部中的cookie
     *
     * @param key   cookie key
     * @param value cookie value
     * @return 连接
     */
    public CBConnection cookie(@CBNotNull String key, @CBNotNull String value) {
        cookies.put(key, value);
        return this;
    }

    /**
     * 设置请求头部中的cookie
     *
     * @param cookie cookie
     * @return 连接
     */
    public CBConnection cookie(Map<String, String> cookie) {
        if (cookie != null) {
            this.cookies.putAll(cookie);
        }
        return this;
    }

    /**
     * 设置请求连接与读取的超时时间
     *
     * @param timeout 时间,单位毫秒,默认30s
     * @return 连接
     */
    public CBConnection timeout(int timeout) {
        this.timeout = timeout;
        return this;
    }

    /**
     * 设置请求头部中的origin
     *
     * @param origin origin
     * @return 连接
     */
    public CBConnection origin(String origin) {
        allowCustomOrigin();
        headers.put(ORIGIN, origin);
        return this;
    }

    /**
     * 设置请求头部中的referer
     *
     * @param referrer 来源
     * @return 连接
     */
    public CBConnection referrer(String referrer) {
        headers.put(REFERRER, referrer);
        return this;
    }

    /**
     * 允许修改请求头部中的origin
     *
     * @return 连接
     */
    public CBConnection allowCustomOrigin() {
        if (allowCustomOrigin) {
            return this;
        }
        try {
            // 去除header限制
            Field field = Class.forName("sun.net.www.protocol.http.HttpURLConnection").getDeclaredField("restrictedHeaderSet");
            field.setAccessible(true);
            Set<String> restrictedHeaderSet = (Set<String>) field.get(null);
            restrictedHeaderSet.remove("origin");
            allowCustomOrigin = true;
        } catch (Exception e) {
            CBLog.warn("open origin switch error", e);
        }
        return this;
    }

    /**
     * 设置请求头部中的user agent
     *
     * @param userAgent ua
     * @return 连接
     */
    public CBConnection userAgent(String userAgent) {
        headers.put(USER_AGENT, userAgent);
        return this;
    }

    /**
     * 添加请求头部,此方法将会清空之前所有已经设置的头部信息
     *
     * @param headers 请求头部
     * @return 连接
     * @see #header(String, String)
     */
    public CBConnection headers(Map<String, String> headers) {
        if (headers != null) {
            this.headers = headers;
        } else {
            this.headers.clear();
        }
        return this;
    }

    /**
     * 添加请求头部,仅仅添加一个,不会清空之前已经设置的头部
     *
     * @param key   请求头部名字
     * @param value 请求头部值
     * @return 连接
     * @see #headers(Map)
     */
    public CBConnection header(String key, String value) {
        if (key != null && value != null) {
            headers.put(key.toLowerCase(), value);
        }
        return this;
    }

    /**
     * 设置请求body数据
     *
     * @param body 原始数据
     * @return 连接
     */
    public CBConnection body(CBBody body) {
        this.body = body;
        return this;
    }

    /**
     * 清空所有cookie
     *
     * @return 操作结果
     */
    public static boolean clearCookie() {
        return cookieManager.getCookieStore().removeAll();
    }

    /**
     * 使用内存cookie，本次请求中会携带内存中的cookie信息
     *
     * @return 连接
     */
    public CBConnection withMemCookie() {
        this.useMemoryCookie = true;
        return this;
    }

    /**
     * 获取网页的内容
     *
     * @return 网页/目标源码
     */
    public CBResponse execute() {
        if (executed) {
            throw new CBNetworkException("the request has executed");
        }
        executed = true;
        CBResponse response = new CBResponse(this);
        if (CBString.isEmptyOrNull(this.url)) {
            response.setMsg("url is missing");
            return response;
        }
        try {
            URL url = new URL(this.url);
            HttpURLConnection conn = (HttpURLConnection) url.openConnection();
            //设置请求方法
            conn.setRequestMethod(method.value);

            if (useMemoryCookie) {
                //设置cookie
                List<HttpCookie> httpCookies = cookieManager.getCookieStore().get(URI.create(this.url));
                String path = url.getPath();
                for (HttpCookie httpCookie : httpCookies) {
                    if (path.startsWith(httpCookie.getPath())) {
                        String key = httpCookie.getName();
                        if (cookies.containsKey(key)) {
                            CBLog.warn("The same cookie is found, the priority set by the user is higher");
                            continue;
                        }
                        cookies.put(key, httpCookie.getValue());
                    }
                }
            }

            // 补充userAgent
            if (!headers.containsKey(USER_AGENT)) {
                headers.put(USER_AGENT, HEADER_MOCK_PC_UA);
            }
            // 补充origin
            if (!headers.containsKey(ORIGIN)) {
                int port = url.getPort();
                String portPart = "";
                if (port != -1) {
                    portPart = ":" + url.getPort();
                }
                headers.put(ORIGIN, url.getProtocol() + "//" + url.getHost() + portPart);
            }

            // 补充accept
            if (!headers.containsKey(ACCEPT)) {
                headers.put(ACCEPT, "*/*");
            }

            // 设置cookie
            if (!headers.containsKey(COOKIE)) {
                StringBuilder allCookies = new StringBuilder();
                for (Map.Entry<String, String> singleCookie : cookies.entrySet()) {
                    String key = singleCookie.getKey();
                    String value = singleCookie.getValue();
                    if (key == null || value == null) {
                        continue;
                    }
                    allCookies.append(key);
                    allCookies.append("=");
                    allCookies.append(value);
                    allCookies.append("; ");
                }
                if (allCookies.length() > 0) {
                    headers.put(COOKIE, allCookies.toString().trim());
                }
            }

            //设置头部
            for (Map.Entry<String, String> item : headers.entrySet()) {
                conn.setRequestProperty(item.getKey().toLowerCase(), item.getValue());
            }

            //设置超时时间
            conn.setConnectTimeout(timeout);
            conn.setReadTimeout(timeout);

            conn.setInstanceFollowRedirects(false);
            if ((method == CBMethod.POST || method == CBMethod.PUT || method == CBMethod.DELETE) & body != null) {
                conn.setDoOutput(true);
                String headerContentType = body.contentType().string(charset.name());
                conn.setRequestProperty(CONTENT_TYPE, headerContentType);
                headers.put(CONTENT_TYPE, headerContentType);
                OutputStream outputStream = conn.getOutputStream();
                switch (body.contentType()) {
                    case FORM:
                    case JSON:
                        outputStream.write(body.body().getBytes(charset));
                        break;
                    case MULTIPART_FORM:
                        Map<String, String> multipartFields = body.multipartFields();
                        Map<String, ? extends File> multipartFiles = body.multipartFiles();
                        String boundary = body.boundary();
                        PrintWriter writer = new PrintWriter(new OutputStreamWriter(outputStream, charset), true);
                        multipartFields.forEach((name, value) -> addMultipartField(writer, name, value, boundary));
                        multipartFiles.forEach((fileKey, file) -> {
                            try {
                                String contentType = "";
                                if (file instanceof CBFormFile) {
                                    contentType = ((CBFormFile) file).getContentType();
                                } else {
                                    contentType = URLConnection.guessContentTypeFromName(file.getName());
                                }
                                addMultipartFile(writer, outputStream, fileKey, file, contentType, boundary);
                            } catch (IOException e) {
                                CBLog.warn("write multiform data error", e);
                            }
                        });

                        writer.flush();
                        writer.append("--").append(boundary).append("--").append(LINE_FEED);
                        writer.close();
                        break;
                }
                outputStream.flush();
                outputStream.close();
            }
            InputStream inputStream = conn.getInputStream();
            ByteArrayOutputStream outStream = new ByteArrayOutputStream();
            //创建一个Buffer字符串
            byte[] buffer = new byte[4096];
            //每次读取的字符串长度，如果为-1，代表全部读取完毕
            int len = 0;
            //使用一个输入流从buffer里把数据读取出来
            while ((len = inputStream.read(buffer)) != -1) {
                outStream.write(buffer, 0, len);
            }
            //关闭输入流
            inputStream.close();
            response.setStatusCode(conn.getResponseCode());
            response.setBody(outStream.toByteArray());
            Map<String, String> headers = new HashMap<>();
            Map<String, List<String>> responseHeaders = conn.getHeaderFields();
            cookieManager.put(URI.create(this.url), responseHeaders);
            for (String key : responseHeaders.keySet()) {
                if (key == null) continue;
                List<String> values = responseHeaders.get(key);
                headers.put(key.toLowerCase(), CBString.join(values, ";"));
            }
            response.setHeaders(headers);
            // guess charset
            String responseContentType = headers.get(CONTENT_TYPE);
            if (!CBString.isEmptyOrNull(responseContentType)) {
                int index = responseContentType.indexOf("charset=");
                if (index == -1) {
                    response.setCharset(charset);
                } else {
                    String responseCharset = responseContentType.substring(index + 8);
                    int index2 = responseCharset.indexOf(";");
                    if (index2 != -1) {
                        responseCharset = responseCharset.substring(0, index2);
                    }
                    try {
                        response.setCharset(Charset.forName(responseCharset));
                    } catch (IllegalArgumentException e) {
                        response.setCharset(charset);
                    }
                }
            } else {
                response.setCharset(charset);
            }
        } catch (Exception e) {
            CBLog.warn("network error", e);
            response.setMsg(e.getMessage());
        }
        return response;
    }

    /**
     * 添加form字段到请求
     *
     * @param writer write
     * @param name   key
     * @param value  值
     */
    private void addMultipartField(@CBNotNull PrintWriter writer, String name, String value, String boundary) {
        writer.append("--").append(boundary).append(LINE_FEED);
        writer.append("Content-Disposition: form-data; name=\"").append(name).append("\"").append(LINE_FEED);
        writer.append("Content-Type: text/plain; charset=").append(String.valueOf(charset)).append(LINE_FEED);
        writer.append(LINE_FEED);
        writer.append(value).append(LINE_FEED);
        writer.flush();
    }

    /**
     * 添加文件到请求
     *
     * @param writer       writer
     * @param outputStream 输出流
     * @param fileKey      文件对应的key值
     * @param file         文件
     * @param contentType  文件类型
     * @param boundary     分隔符
     */
    private void addMultipartFile(@CBNotNull PrintWriter writer, @CBNotNull OutputStream outputStream, String fileKey, File file, String contentType, String boundary) throws IOException {
        writer.append("--").append(boundary).append(LINE_FEED);
        writer.append("Content-Disposition: form-data; name=\"").append(fileKey).append("\"; filename=\"").append(file.getName()).append("\"").append(LINE_FEED);
        writer.append("Content-Type: ").append(contentType).append(LINE_FEED);
        writer.append("Content-Transfer-Encoding: binary").append(LINE_FEED);
        writer.append(LINE_FEED);
        writer.flush();

        FileInputStream inputStream = new FileInputStream(file);
        byte[] buffer = new byte[4096];
        int bytesRead = -1;
        while ((bytesRead = inputStream.read(buffer)) != -1) {
            outputStream.write(buffer, 0, bytesRead);
        }
        outputStream.flush();
        inputStream.close();
        writer.append(LINE_FEED);
        writer.flush();
    }

    /**
     * 保存指定位置的文件，使用该功能，不要主动调用execute
     *
     * @param file 保存文件
     * @return 保存结果
     * @throws IOException IO异常
     */
    public boolean save(File file) throws IOException {
        CBResponse response = execute();
        if (!response.success()) {
            return false;
        }
        OutputStream os = Files.newOutputStream(file.toPath());
        byte[] body = response.body();
        os.write(body, 0, body.length);
        os.flush();
        os.close();
        return true;
    }

    protected String getCurlCommand() {
        StringBuilder command = new StringBuilder("curl");
        // method
        if (method != CBMethod.GET) {
            command.append(" -X ");
            command.append(method.value);
        }

        //header
        for (Map.Entry<String, String> item : headers.entrySet()) {
            String key = item.getKey().toLowerCase();
            String value = item.getValue();
            if (CONTENT_TYPE.equals(key) && body.contentType() == CBContentType.MULTIPART_FORM) {
                continue;
            }
            command.append(" -H '");
            command.append(key);
            command.append(": ");
            command.append(value);
            command.append("'");
        }

        //data
        switch (body.contentType()) {
            case FORM:
                command.append(" --data-binary '");
                command.append(body.body());
                command.append("'");
                break;
            case JSON:
                command.append(" --data-raw '");
                command.append(body.body());
                command.append("'");
                break;
            case MULTIPART_FORM:
                for (Map.Entry<String, ? extends File> entry : body.multipartFiles().entrySet()) {
                    command.append(" -F '");
                    command.append(entry.getKey());
                    command.append("=@");
                    command.append(entry.getValue());
                    command.append("'");
                }

                for (Map.Entry<String, String> entry : body.multipartFields().entrySet()) {
                    command.append(" -F '");
                    command.append(entry.getKey());
                    command.append("=");
                    command.append(entry.getValue());
                    command.append("'");
                }
                break;
        }
        command.append(" ");
        command.append(url);

        return command.toString();
    }
}