package me.codeboy.tools.task;

import me.codeboy.tools.annotation.CBNotNull;
import me.codeboy.tools.task.base.CBTask;
import me.codeboy.tools.task.base.CBTaskListener;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * 单线程池
 */
public class CBTaskPool {
    private ScheduledExecutorService executorService;
    private final static CBTaskPool defaultTaskPool = new CBTaskPool();

    public synchronized static CBTaskPool instance() {
        if (defaultTaskPool.executorService == null) {
            defaultTaskPool.executorService = Executors.newSingleThreadScheduledExecutor();
        }
        return defaultTaskPool;
    }

    public static CBTaskPool newInstance() {
        CBTaskPool singleTaskPool = new CBTaskPool();
        singleTaskPool.executorService = Executors.newSingleThreadScheduledExecutor();
        return singleTaskPool;
    }

    /**
     * 提交任务
     *
     * @param <T>  任务结束时返回的数据类型
     * @param task 任务
     */
    public <T> void submit(CBTask<T> task) {
        submit(task, null);
    }

    /**
     * 提交任务
     *
     * @param <T>      任务结束时返回的数据类型
     * @param task     任务
     * @param listener 任务监听器
     */
    public <T> void submit(CBTask<T> task, CBTaskListener<T> listener) {
        postDelayed(task, listener, 0);
    }

    /**
     * 定时执行任务
     *
     * @param task       任务
     * @param execMillis 执行任务时间
     * @param <T>        任务结束时返回的数据类型
     */
    public <T> void postAtTime(@CBNotNull CBTask<T> task, long execMillis) {
        postAtTime(task, null, execMillis);
    }

    /**
     * 定时执行任务
     *
     * @param task       任务
     * @param listener   任务监听器
     * @param execMillis 执行任务时间
     * @param <T>        任务结束时返回的数据类型
     */
    public <T> void postAtTime(@CBNotNull CBTask<T> task, CBTaskListener<T> listener, long execMillis) {
        postDelayed(task, listener, execMillis - System.currentTimeMillis());
    }


    /**
     * 延迟执行任务
     *
     * @param task      任务
     * @param delayTime 延迟任务时间
     * @param <T>       任务结束时返回的数据类型
     */
    public <T> void postDelayed(@CBNotNull CBTask<T> task, long delayTime) {
        postDelayed(task, null, delayTime);
    }

    /**
     * 延迟执行任务
     *
     * @param task      任务
     * @param listener  任务监听器
     * @param delayTime 延迟任务时间
     * @param <T>       任务结束时返回的数据类型
     */
    public <T> void postDelayed(@CBNotNull CBTask<T> task, CBTaskListener<T> listener, long delayTime) {
        executorService.schedule(() -> {
            processTask(task, listener);
        }, delayTime, TimeUnit.MILLISECONDS);
    }

    /**
     * 固定时间间隔执行任务
     *
     * @param task         任务
     * @param listener     任务监听器
     * @param initialDelay 首次延迟任务时间
     * @param fixedTime    固定时间
     * @param <T>          任务结束时返回的数据类型
     */
    public <T> void scheduleAtFixedRate(@CBNotNull CBTask<T> task, CBTaskListener<T> listener, long initialDelay, long fixedTime) {
        executorService.scheduleAtFixedRate(() -> {
            processTask(task, listener);
        }, initialDelay, fixedTime, TimeUnit.MILLISECONDS);
    }

    private <T> void processTask(@CBNotNull CBTask<T> task, CBTaskListener<T> listener) {
        if (listener != null) {
            listener.onStart(this, task);
            try {
                T result = task.execute();
                listener.onFinish(this, task, result);
            } catch (Exception e) {
                listener.onError(this, task, e);
            }
        } else {
            task.execute();
        }
    }

    /**
     * 是否已经关闭
     *
     * @return 关闭状态
     */
    public boolean isShutDown() {
        return executorService.isShutdown();
    }

    /**
     * 关闭任务池
     *
     * @return 关闭成功
     */
    public boolean shutdown() {
        // 默认线程池不能关闭
        if (this == defaultTaskPool) {
            return false;
        }
        executorService.shutdown();
        return true;
    }
}
