package me.codeboy.tools.util;

import me.codeboy.tools.annotation.CBNotNull;
import me.codeboy.tools.annotation.CBNullable;
import me.codeboy.tools.lang.CBString;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 正则表达式工具
 */
public class CBRegex {

    /**
     * 匹配所有
     *
     * @param origin 字符串
     * @param regex  正则表达式
     * @return 所有匹配的串，没有返回空集合
     */
    @CBNotNull
    public static List<String> matchAll(String origin, String regex) {
        return matchAll(origin, regex, 0);
    }

    /**
     * 匹配所有
     *
     * @param origin 字符串
     * @param regex  正则表达式
     * @param group  第几个
     * @return 所有匹配的串，没有返回空集合
     */
    @CBNotNull
    public static List<String> matchAll(String origin, String regex, int group) {
        List<String> result = new ArrayList<>();
        if (CBString.isEmptyOrNull(origin) || CBString.isEmptyOrNull(regex)) {
            return result;
        }
        Pattern pattern = Pattern.compile(regex);
        Matcher matcher = pattern.matcher(origin);
        while (matcher.find()) {
            result.add(matcher.group(group));
        }
        return result;
    }

    /**
     * 字符串是否能够符合正则表达式
     * <p>
     * 整个字符串匹配，仅包含部分也处于失败场景
     * </p>
     *
     * @param origin 字符串
     * @param regex  正则表达式
     * @return 匹配结果
     */
    public static boolean match(String origin, String regex) {
        if (CBString.isEmptyOrNull(origin) || CBString.isEmptyOrNull(regex)) {
            return false;
        }
        Pattern pattern = Pattern.compile(regex);
        return pattern.matcher(origin).matches();
    }

    /**
     * 匹配第一个符合规则的子串
     *
     * @param origin 字符串
     * @param regex  正则表达式
     * @return 所有匹配的串，没有返回null
     */
    @CBNullable
    public static String matchFirst(String origin, String regex) {
        return matchFirst(origin, regex, 0);
    }

    /**
     * 匹配第一个符合规则的子串
     *
     * @param origin 字符串
     * @param regex  正则表达式
     * @param group  第一个
     * @return 所有匹配的串，没有返回null
     */
    @CBNullable
    public static String matchFirst(String origin, String regex, int group) {
        if (CBString.isEmptyOrNull(origin) || CBString.isEmptyOrNull(regex)) {
            return null;
        }
        Pattern pattern = Pattern.compile(regex);
        Matcher matcher = pattern.matcher(origin);
        if (matcher.find()) {
            return matcher.group(group);
        }
        return null;
    }
}
