/*
 * Decompiled with CFR 0.152.
 */
package me.desair.tus.server.upload.disk;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import me.desair.tus.server.exception.InvalidUploadOffsetException;
import me.desair.tus.server.exception.TusException;
import me.desair.tus.server.exception.UploadNotFoundException;
import me.desair.tus.server.upload.UploadId;
import me.desair.tus.server.upload.UploadIdFactory;
import me.desair.tus.server.upload.UploadInfo;
import me.desair.tus.server.upload.UploadLockingService;
import me.desair.tus.server.upload.UploadStorageService;
import me.desair.tus.server.upload.UploadType;
import me.desair.tus.server.upload.concatenation.UploadConcatenationService;
import me.desair.tus.server.upload.concatenation.VirtualConcatenationService;
import me.desair.tus.server.upload.disk.AbstractDiskBasedService;
import me.desair.tus.server.upload.disk.ExpiredUploadFilter;
import me.desair.tus.server.util.Utils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DiskStorageService
extends AbstractDiskBasedService
implements UploadStorageService {
    private static final Logger log = LoggerFactory.getLogger(DiskStorageService.class);
    private static final String UPLOAD_SUB_DIRECTORY = "uploads";
    private static final String INFO_FILE = "info";
    private static final String DATA_FILE = "data";
    private Long maxUploadSize = null;
    private Long uploadExpirationPeriod = null;
    private UploadIdFactory idFactory;
    private UploadConcatenationService uploadConcatenationService;

    public DiskStorageService(String storagePath) {
        super(storagePath + File.separator + UPLOAD_SUB_DIRECTORY);
        this.setUploadConcatenationService(new VirtualConcatenationService(this));
    }

    public DiskStorageService(UploadIdFactory idFactory, String storagePath) {
        this(storagePath);
        Validate.notNull((Object)idFactory, (String)"The IdFactory cannot be null", (Object[])new Object[0]);
        this.idFactory = idFactory;
    }

    @Override
    public void setIdFactory(UploadIdFactory idFactory) {
        Validate.notNull((Object)idFactory, (String)"The IdFactory cannot be null", (Object[])new Object[0]);
        this.idFactory = idFactory;
    }

    @Override
    public void setMaxUploadSize(Long maxUploadSize) {
        this.maxUploadSize = maxUploadSize != null && maxUploadSize > 0L ? maxUploadSize : 0L;
    }

    @Override
    public long getMaxUploadSize() {
        return this.maxUploadSize == null ? 0L : this.maxUploadSize;
    }

    @Override
    public UploadInfo getUploadInfo(String uploadUrl, String ownerKey) throws IOException {
        UploadInfo uploadInfo = this.getUploadInfo(this.idFactory.readUploadId(uploadUrl));
        if (uploadInfo == null || !Objects.equals(uploadInfo.getOwnerKey(), ownerKey)) {
            return null;
        }
        return uploadInfo;
    }

    @Override
    public UploadInfo getUploadInfo(UploadId id) throws IOException {
        try {
            Path infoPath = this.getInfoPath(id);
            return Utils.readSerializable(infoPath, UploadInfo.class);
        }
        catch (UploadNotFoundException e) {
            return null;
        }
    }

    @Override
    public String getUploadUri() {
        return this.idFactory.getUploadUri();
    }

    @Override
    public UploadInfo create(UploadInfo info, String ownerKey) throws IOException {
        UploadId id = this.createNewId();
        this.createUploadDirectory(id);
        try {
            Path bytesPath = this.getBytesPath(id);
            Files.createFile(bytesPath, new FileAttribute[0]);
            info.setId(id);
            info.setOffset(0L);
            info.setOwnerKey(ownerKey);
            this.update(info);
            return info;
        }
        catch (UploadNotFoundException e) {
            log.error("Unable to create UploadInfo because of an upload not found exception", (Throwable)e);
            return null;
        }
    }

    @Override
    public void update(UploadInfo uploadInfo) throws IOException, UploadNotFoundException {
        Path infoPath = this.getInfoPath(uploadInfo.getId());
        Utils.writeSerializable(uploadInfo, infoPath);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public UploadInfo append(UploadInfo info, InputStream inputStream) throws IOException, TusException {
        if (info != null) {
            Path bytesPath = this.getBytesPath(info.getId());
            long max = this.getMaxUploadSize() > 0L ? this.getMaxUploadSize() : Long.MAX_VALUE;
            long transferred = 0L;
            Long offset = info.getOffset();
            long newOffset = offset;
            try (ReadableByteChannel uploadedBytes = Channels.newChannel(inputStream);
                 FileChannel file = FileChannel.open(bytesPath, StandardOpenOption.WRITE);){
                try {
                    file.lock();
                    if (!offset.equals(file.size())) {
                        throw new InvalidUploadOffsetException("The upload offset does not correspond to the written bytes. You can only append to the end of an upload");
                    }
                    transferred = file.transferFrom(uploadedBytes, offset, max - offset);
                    file.force(true);
                    newOffset = offset + transferred;
                }
                catch (Exception ex) {
                    newOffset = this.writeAsMuchAsPossible(file);
                    throw ex;
                }
            }
            finally {
                info.setOffset(newOffset);
                this.update(info);
            }
        }
        return info;
    }

    @Override
    public void removeLastNumberOfBytes(UploadInfo info, long byteCount) throws UploadNotFoundException, IOException {
        if (info != null && byteCount > 0L) {
            Path bytesPath = this.getBytesPath(info.getId());
            try (FileChannel file = FileChannel.open(bytesPath, StandardOpenOption.WRITE);){
                file.lock();
                file.truncate(file.size() - byteCount);
                file.force(true);
                info.setOffset(file.size());
                this.update(info);
            }
        }
    }

    @Override
    public void terminateUpload(UploadInfo info) throws UploadNotFoundException, IOException {
        if (info != null) {
            Path uploadPath = this.getPathInStorageDirectory(info.getId());
            FileUtils.deleteDirectory((File)uploadPath.toFile());
        }
    }

    @Override
    public Long getUploadExpirationPeriod() {
        return this.uploadExpirationPeriod;
    }

    @Override
    public void setUploadExpirationPeriod(Long uploadExpirationPeriod) {
        this.uploadExpirationPeriod = uploadExpirationPeriod;
    }

    @Override
    public void setUploadConcatenationService(UploadConcatenationService concatenationService) {
        Validate.notNull((Object)concatenationService);
        this.uploadConcatenationService = concatenationService;
    }

    @Override
    public UploadConcatenationService getUploadConcatenationService() {
        return this.uploadConcatenationService;
    }

    @Override
    public InputStream getUploadedBytes(String uploadUri, String ownerKey) throws IOException, UploadNotFoundException {
        UploadId id = this.idFactory.readUploadId(uploadUri);
        UploadInfo uploadInfo = this.getUploadInfo(id);
        if (uploadInfo == null || !Objects.equals(uploadInfo.getOwnerKey(), ownerKey)) {
            throw new UploadNotFoundException("The upload with id " + id + " could not be found for owner " + ownerKey);
        }
        return this.getUploadedBytes(id);
    }

    @Override
    public InputStream getUploadedBytes(UploadId id) throws IOException, UploadNotFoundException {
        InputStream inputStream = null;
        UploadInfo uploadInfo = this.getUploadInfo(id);
        if (UploadType.CONCATENATED.equals((Object)uploadInfo.getUploadType()) && this.uploadConcatenationService != null) {
            inputStream = this.uploadConcatenationService.getConcatenatedBytes(uploadInfo);
        } else {
            Path bytesPath = this.getBytesPath(id);
            if (bytesPath != null) {
                inputStream = Channels.newInputStream(FileChannel.open(bytesPath, StandardOpenOption.READ));
            }
        }
        return inputStream;
    }

    @Override
    public void copyUploadTo(UploadInfo info, OutputStream outputStream) throws UploadNotFoundException, IOException {
        List<UploadInfo> uploads = this.getUploads(info);
        try (WritableByteChannel outputChannel = Channels.newChannel(outputStream);){
            for (UploadInfo upload : uploads) {
                if (upload == null) {
                    log.warn("We cannot copy the bytes of an upload that does not exist");
                    continue;
                }
                if (upload.isUploadInProgress()) {
                    log.warn("We cannot copy the bytes of upload {} because it is still in progress", (Object)upload.getId());
                    continue;
                }
                Path bytesPath = this.getBytesPath(upload.getId());
                FileChannel file = FileChannel.open(bytesPath, StandardOpenOption.READ);
                try {
                    file.transferTo(0L, upload.getLength(), outputChannel);
                }
                finally {
                    if (file == null) continue;
                    file.close();
                }
            }
        }
    }

    @Override
    public void cleanupExpiredUploads(UploadLockingService uploadLockingService) throws IOException {
        try (DirectoryStream<Path> expiredUploadsStream = Files.newDirectoryStream(this.getStoragePath(), new ExpiredUploadFilter(this, uploadLockingService));){
            for (Path path : expiredUploadsStream) {
                FileUtils.deleteDirectory((File)path.toFile());
            }
        }
    }

    private List<UploadInfo> getUploads(UploadInfo info) throws IOException, UploadNotFoundException {
        List<UploadInfo> uploads;
        if (info != null && UploadType.CONCATENATED.equals((Object)info.getUploadType()) && this.uploadConcatenationService != null) {
            this.uploadConcatenationService.merge(info);
            uploads = this.uploadConcatenationService.getPartialUploads(info);
        } else {
            uploads = Collections.singletonList(info);
        }
        return uploads;
    }

    private Path getBytesPath(UploadId id) throws UploadNotFoundException {
        return this.getPathInUploadDir(id, DATA_FILE);
    }

    private Path getInfoPath(UploadId id) throws UploadNotFoundException {
        return this.getPathInUploadDir(id, INFO_FILE);
    }

    private Path createUploadDirectory(UploadId id) throws IOException {
        return Files.createDirectories(this.getPathInStorageDirectory(id), new FileAttribute[0]);
    }

    private Path getPathInUploadDir(UploadId id, String fileName) throws UploadNotFoundException {
        Path uploadDir = this.getPathInStorageDirectory(id);
        if (uploadDir != null && Files.exists(uploadDir, new LinkOption[0])) {
            return uploadDir.resolve(fileName);
        }
        throw new UploadNotFoundException("The upload for id " + id + " was not found.");
    }

    private synchronized UploadId createNewId() throws IOException {
        UploadId id;
        while (this.getUploadInfo(id = this.idFactory.createId()) != null) {
        }
        return id;
    }

    private long writeAsMuchAsPossible(FileChannel file) throws IOException {
        long offset = 0L;
        if (file != null) {
            file.force(true);
            offset = file.size();
        }
        return offset;
    }
}

