package eleme.openapi.sdk.media.common.http;

import eleme.openapi.sdk.media.MediaConfiguration;
import eleme.openapi.sdk.media.Result;
import eleme.openapi.sdk.media.trace.Code;
import eleme.openapi.sdk.media.trace.Reporter;
import eleme.openapi.sdk.media.upload.*;
import eleme.openapi.sdk.utils.JacksonUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;

import java.io.IOException;
import java.util.Map;

import static eleme.openapi.sdk.media.trace.Reporter.sharedInstance;

/**
 * @author jinli Jan 26, 2015
 */
public abstract class HttpClientBase {
    private static HttpClient httpClient;
    public static  boolean    keepAlive = true;

    protected HttpClient getHttpClient() {
        // double check, synchronized
        if (httpClient == null) {
            synchronized (this) {
                if (httpClient == null) {
                    httpClient = HttpClientBuilder.create()                                     // create http builder
                            .setMaxConnTotal(MediaConfiguration.MAX_HTTP_CONNECTIONS)           // max http connections
                            .setMaxConnPerRoute(MediaConfiguration.MAX_HTTP_CONNECTIONS)        // max http connections per route
                            .disableAutomaticRetries()                                          // disable retry
                            .build();                                                           // build http client
                }
            }
        }
        return httpClient;
    }

    protected <T> Result<T> executeWithUploadTrace(final HttpRequestBase request, BaseUploadRequest baseUploadRequest, final Class<T> clazz, Reporter.OP op, long uploadSize) {
        sharedInstance().start();
        sharedInstance().enter(op);
        String uploadId = sharedInstance().getSessionId();
        request.setHeader("X-Media-Session-Id", uploadId);
        sharedInstance().addKV("uploadStartTime", String.valueOf(System.currentTimeMillis()));
        Result<T> result = execute(request, clazz);
        String code;
        if (result.getT() != null) {
            code = Code.LOCAL_ERROR.value;
        } else {
            code = Code.REMOTE_PREFIX.value + result.getHttpStatus();
        }

        String message = result.getT() != null ? result.getT().getMessage() : result.getMessage();
        message = message != null ? message.replaceAll(";", "") : null;
        sharedInstance().release(op);


        if (baseUploadRequest != null) {
            if (baseUploadRequest.getUploadPolicy() != null) { //数据都在token里面,无法取到,暂时不取
                sharedInstance().addKV("namespace", baseUploadRequest.getUploadPolicy().getNamespace());
                sharedInstance().addKV("policyName", baseUploadRequest.getUploadPolicy().getName());
                sharedInstance().addKV("policyDir", baseUploadRequest.getUploadPolicy().getDir());
            }

            if (baseUploadRequest instanceof MultipartCancelRequest) {
                MultipartCancelRequest req = (MultipartCancelRequest) baseUploadRequest;
                sharedInstance().addKV("multipartId", req.getId());
                sharedInstance().addKV("multipartUploadId", req.getUploadId());
                sharedInstance().addKV("dir", req.getDir());
                sharedInstance().addKV("name", req.getName());
            }


            if (baseUploadRequest instanceof MultipartCompleteRequest) {
                MultipartCompleteRequest req = (MultipartCompleteRequest) baseUploadRequest;
                sharedInstance().addKV("multipartId", req.getId());
                sharedInstance().addKV("multipartUploadId", req.getUploadId());
                sharedInstance().addKV("dir", req.getDir());
                sharedInstance().addKV("name", req.getName());
            }

            if (baseUploadRequest instanceof MultipartInitRequest) {
                MultipartInitRequest req = (MultipartInitRequest) baseUploadRequest;
                sharedInstance().addKV("dir", req.getDir());
                sharedInstance().addKV("name", req.getName());
            }

            if (baseUploadRequest instanceof MultipartUploadRequest) {
                MultipartUploadRequest req = (MultipartUploadRequest) baseUploadRequest;
                sharedInstance().addKV("dir", req.getDir());
                sharedInstance().addKV("multipartId", req.getId());
                sharedInstance().addKV("multipartUploadId", req.getUploadId());
                sharedInstance().addKV("name", req.getName());
            }

            if (baseUploadRequest instanceof UploadRequest) {
                UploadRequest req = (UploadRequest) baseUploadRequest;
                sharedInstance().addKV("dir", req.getDir());
                sharedInstance().addKV("name", req.getName());
            }
        }


        sharedInstance().finish(result.getRequestId(), uploadId, message, code, uploadSize);


        return result;
    }

    protected <T> Result<T> execute(final HttpRequestBase request, final Class<T> clazz) {
        HttpClient httpClient = getHttpClient();

        if (!keepAlive)
            request.setHeader("Connection", "close");


        ResponseHandler<Result<T>> handler = new ResponseHandler<Result<T>>() {
            @SuppressWarnings("unchecked")
            public Result<T> handleResponse(final HttpResponse response) throws ClientProtocolException, IOException {
                int status = response.getStatusLine().getStatusCode();
                HttpEntity entity = response.getEntity();
                String body = entity != null ? EntityUtils.toString(entity) : null;
                // OK
                if (status >= 200 && status < 300) {
                    if (clazz != null && clazz != Void.class) {
                        return Result.create(status, JacksonUtils.json2pojo(body, clazz), getRequestIdQuietly(body));
                    } else {
                        return Result.create(status, null, getRequestIdQuietly(body));
                    }

                }
                // ERROR
                else {
                    Map<String, Object> info = JacksonUtils.json2map(body);
                    String requestId = getValue(info, "requestId");
                    String code = getValue(info, "code");
                    String msg = getValue(info, "message");
                    if (msg == null) {
                        msg = getValue(info, "msg");
                    }

                    return Result.create(status, requestId, code, msg);
                }
            }

        };

        try {
            return httpClient.execute(request, handler);
        } catch (Exception e) {
            Result<T> result = Result.create(0, null, "ClientError", e.getMessage());
            result.setT(e);
            return result;
        } finally {
            try {
                request.releaseConnection();
            } catch (Exception e) {
                // ignore
            }
        }
    }


    private String getRequestIdQuietly(String json) {
        try {
            return String.valueOf(JacksonUtils.json2map(json).get("requestId"));
        } catch (Throwable t) {
            return null;
        }
    }

    private String getValue(Map<String, Object> kvs, String key) {
        return kvs != null && kvs.containsKey(key) ? String.valueOf(kvs.get(key)) : null;
    }
}
