package eleme.openapi.sdk.api.service;

import eleme.openapi.sdk.api.annotation.Service;
import eleme.openapi.sdk.api.base.BaseNopService;
import eleme.openapi.sdk.api.exception.ServiceException;
import eleme.openapi.sdk.oauth.response.Token;
import eleme.openapi.sdk.config.Config;
import eleme.openapi.sdk.api.entity.order.*;
import eleme.openapi.sdk.api.enumeration.order.*;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.*;

/**
 * 订单服务
 */
@Service("eleme.order")
public class OrderService extends BaseNopService {
    public OrderService(Config config,Token token) {
        super(config, token, OrderService.class);
    }

    /**
     * 获取订单配送轨迹
     *
     * @param orderId 订单Id
     * @return 骑手路线信息
     * @throws ServiceException 服务异常
     */
    public DeliveryRouteInfo getDeliveryRoutes(String orderId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        return call("eleme.order.delivery.getDeliveryRoutes", params);
    }

    /**
     * 确认订单(推荐)
     *
     * @param orderId 订单Id
     * @throws ServiceException 服务异常
     */
    public void confirmOrderLite(String orderId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        call("eleme.order.confirmOrderLite", params);
    }

    /**
     * 同意退单/同意取消单(推荐)
     *
     * @param orderId 订单Id
     * @throws ServiceException 服务异常
     */
    public void agreeRefundLite(String orderId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        call("eleme.order.agreeRefundLite", params);
    }

    /**
     * 不同意退单/不同意取消单(推荐)
     *
     * @param orderId 订单Id
     * @param reason 商家不同意退单原因
     * @throws ServiceException 服务异常
     */
    public void disagreeRefundLite(String orderId, String reason) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        params.put("reason", reason);
        call("eleme.order.disagreeRefundLite", params);
    }

    /**
     * 获取订单配送记录
     *
     * @param orderId 订单Id
     * @return 配送记录列表
     * @throws ServiceException 服务异常
     */
    public List<ODeliveryRecord> getDeliveryStateRecord(String orderId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        return call("eleme.order.getDeliveryStateRecord", params);
    }

    /**
     * 批量获取订单最新配送记录
     *
     * @param orderIds 订单Id列表
     * @return 订单Id和该订单配送记录
     * @throws ServiceException 服务异常
     */
    public Map<String,ODeliveryRecord> batchGetDeliveryStates(List<String> orderIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderIds", orderIds);
        return call("eleme.order.batchGetDeliveryStates", params);
    }

    /**
     * 配送异常或者物流拒单后选择自行配送(推荐)
     *
     * @param orderId 订单Id
     * @throws ServiceException 服务异常
     */
    public void deliveryBySelfLite(String orderId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        call("eleme.order.deliveryBySelfLite", params);
    }

    /**
     * 配送异常或者物流拒单后选择不再配送(推荐)
     *
     * @param orderId 订单Id
     * @throws ServiceException 服务异常
     */
    public void noMoreDeliveryLite(String orderId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        call("eleme.order.noMoreDeliveryLite", params);
    }

    /**
     * 回复催单
     *
     * @param remindId 催单Id
     * @param type 回复类别
     * @param content 回复内容,如果type为custom,content必填,回复内容不能超过30个字符
     * @throws ServiceException 服务异常
     */
    public void replyReminder(String remindId, ReplyReminderType type, String content) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("remindId", remindId);
        params.put("type", type);
        params.put("content", content);
        call("eleme.order.replyReminder", params);
    }

    /**
     * 获取指定订单菜品活动价格.
     *
     * @param orderId 订单Id
     * @return 菜品价格信息
     * @throws ServiceException 服务异常
     */
    public Map<String,OCommodity> getCommodities(String orderId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        return call("eleme.order.getCommodities", params);
    }

    /**
     * 获取订单退款信息
     *
     * @param orderId 订单Id
     * @return 订单退款信息
     * @throws ServiceException 服务异常
     */
    public ORefundOrder getRefundOrder(String orderId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        return call("eleme.order.getRefundOrder", params);
    }

    /**
     * 批量获取订单退款信息
     *
     * @param orderIds 订单Id列表
     * @return 订单Id和该订单的退款信息
     * @throws ServiceException 服务异常
     */
    public Map<String,ORefundOrder> mgetRefundOrders(List<String> orderIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderIds", orderIds);
        return call("eleme.order.mgetRefundOrders", params);
    }

    /**
     * 取消呼叫配送
     *
     * @param orderId 订单Id
     * @throws ServiceException 服务异常
     */
    public void cancelDelivery(String orderId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        call("eleme.order.cancelDelivery", params);
    }

    /**
     * 呼叫配送
     *
     * @param orderId 订单Id
     * @param fee 小费,1-8之间的整数
     * @throws ServiceException 服务异常
     */
    public void callDelivery(String orderId, Integer fee) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        params.put("fee", fee);
        call("eleme.order.callDelivery", params);
    }

    /**
     * 获取店铺未回复的催单
     *
     * @param shopId 店铺Id
     * @return 催单集合
     * @throws ServiceException 服务异常
     */
    public List<OReminder> getUnreplyReminders(long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.order.getUnreplyReminders", params);
    }

    /**
     * 查询店铺未处理订单
     *
     * @param shopId 店铺Id
     * @return 订单Id集合
     * @throws ServiceException 服务异常
     */
    public List<String> getUnprocessOrders(long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.order.getUnprocessOrders", params);
    }

    /**
     * 查询店铺未处理的取消单
     *
     * @param shopId 店铺Id
     * @return 订单Id集合
     * @throws ServiceException 服务异常
     */
    public List<String> getCancelOrders(long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.order.getCancelOrders", params);
    }

    /**
     * 查询店铺未处理的退单
     *
     * @param shopId 店铺Id
     * @return 订单Id集合
     * @throws ServiceException 服务异常
     */
    public List<String> getRefundOrders(long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.order.getRefundOrders", params);
    }

    /**
     * 批量查询订单是否支持索赔
     *
     * @param orderIds 索赔订单Id的列表
     * @return 订单ID和该订单的索赔详情
     * @throws ServiceException 服务异常
     */
    public Map<String,CompensationOrder> querySupportedCompensationOrders(List<String> orderIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderIds", orderIds);
        return call("eleme.order.querySupportedCompensationOrders", params);
    }

    /**
     * 批量申请索赔
     *
     * @param requests 索赔请求的列表
     * @return 订单ID和该订单ID的申请索赔结果
     * @throws ServiceException 服务异常
     */
    public Map<String,Boolean> batchApplyCompensations(List<CompensationRequest> requests) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("requests", requests);
        return call("eleme.order.batchApplyCompensations", params);
    }

    /**
     * 批量查询索赔结果
     *
     * @param orderIds 索赔订单Id的列表
     * @return 订单ID和该订单的索赔信息，索赔信息可能为null
     * @throws ServiceException 服务异常
     */
    public Map<String,CompensationInfo> queryCompensationOrders(List<String> orderIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderIds", orderIds);
        return call("eleme.order.queryCompensationOrders", params);
    }

    /**
     * 众包订单询价，获取配送费
     *
     * @param orderId 订单Id
     * @return 配送费
     * @throws ServiceException 服务异常
     */
    public double getDeliveryFeeForCrowd(String orderId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        return call("eleme.order.getDeliveryFeeForCrowd", params);
    }

    /**
     * 评价骑手
     *
     * @param orderId 订单Id
     * @param evaluationInfo 评价信息
     * @throws ServiceException 服务异常
     */
    public void evaluateRider(String orderId, EvaluationInfo evaluationInfo) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        params.put("evaluationInfo", evaluationInfo);
        call("eleme.order.evaluateRider", params);
    }

    /**
     * 批量获取订单加小费信息
     *
     * @param orderIds 订单Id的列表
     * @return 订单Id和小费信息
     * @throws ServiceException 服务异常
     */
    public Map<String,FeeInfo> mgetDeliveryTipInfos(List<String> orderIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderIds", orderIds);
        return call("eleme.order.mgetDeliveryTipInfos", params);
    }

    /**
     * 订单加小费
     *
     * @param orderId 订单Id
     * @param tip 小费金额
     * @throws ServiceException 服务异常
     */
    public void addDeliveryTipByOrderId(String orderId, Integer tip) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        params.put("tip", tip);
        call("eleme.order.addDeliveryTipByOrderId", params);
    }

    /**
     * 主动发起退单
     *
     * @param orderId 订单Id
     * @param type 取消原因
     * @param remark 备注说明
     * @throws ServiceException 服务异常
     */
    public void applyRefund(String orderId, OInvalidateType type, String remark) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        params.put("type", type);
        params.put("remark", remark);
        call("eleme.order.applyRefund", params);
    }

    /**
     * 商家主动发起退单（企业到店买单订单）
     *
     * @param orderId 订单Id
     * @param type 取消原因
     * @param remark 备注说明
     * @throws ServiceException 服务异常
     */
    public void shopPayApplyRefund(String orderId, OInvalidateType type, String remark) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        params.put("type", type);
        params.put("remark", remark);
        call("eleme.order.shopPayApplyRefund", params);
    }

    /**
     * 商家确认已完成出餐
     *
     * @param orderId 订单Id
     * @throws ServiceException 服务异常
     */
    public void setOrderPrepared(String orderId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        call("eleme.order.setOrderPrepared", params);
    }

    /**
     * 查询已出餐列表
     *
     * @param orderIds 查询已出餐订单Id的列表
     * @return 出餐订单Id和出餐时间
     * @throws ServiceException 服务异常
     */
    public Map<String,String> getPreparedTimesByOrderIds(List<String> orderIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderIds", orderIds);
        return call("eleme.order.getPreparedTimesByOrderIds", params);
    }

    /**
     * 查询顾客联系方式
     *
     * @param orderIds 订单Id的列表
     * @return 订单Id和该订单的顾客信息
     * @throws ServiceException 服务异常
     */
    public Map<String,OUserSimpleInfo> mgetUserSimpleInfoByOrderIds(List<String> orderIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderIds", orderIds);
        return call("eleme.order.mgetUserSimpleInfoByOrderIds", params);
    }

    /**
     * 商家部分退款
     *
     * @param orderId 订单Id
     * @param refundOrderMessage 退款详情
     * @throws ServiceException 服务异常
     */
    public void refundPart(String orderId, ORefundOrderMessage refundOrderMessage) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        params.put("refundOrderMessage", refundOrderMessage);
        call("eleme.order.refundPart", params);
    }

    /**
     * 设置订单开票地址
     *
     * @param orderId 订单Id
     * @param invoiceUrl 开票地址
     * @throws ServiceException 服务异常
     */
    public void setInvoiceUrl(String orderId, String invoiceUrl) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        params.put("invoiceUrl", invoiceUrl);
        call("eleme.order.setInvoiceUrl", params);
    }

    /**
     * 自配送商家同步运单的位置信息
     *
     * @param shopId 店铺Id
     * @param orderId 订单Id
     * @param locationInfo 位置信息,仅接受火星坐标系
     * @throws ServiceException 服务异常
     */
    public void selfDeliveryLocationSync(long shopId, String orderId, OLocationInfo locationInfo) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("orderId", orderId);
        params.put("locationInfo", locationInfo);
        call("eleme.order.selfDeliveryLocationSync", params);
    }

    /**
     * 订单预计出餐时间
     *
     * @param orderId 订单Id
     * @param predictTime 预计订单出餐时间
     * @throws ServiceException 服务异常
     */
    public void orderPredictFinishTime(String orderId, String predictTime) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        params.put("predictTime", predictTime);
        call("eleme.order.orderPredictFinishTime", params);
    }

    /**
     * 菜品预计出餐时间
     *
     * @param shopId 店铺Id
     * @param commodityInfo 菜品信息
     * @throws ServiceException 服务异常
     */
    public void commodityPredictFinishTime(long shopId, CommodityInfo commodityInfo) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("commodityInfo", commodityInfo);
        call("eleme.order.commodityPredictFinishTime", params);
    }

    /**
     * 菜品实际出餐时间
     *
     * @param shopId 店铺Id
     * @param commodityInfo 菜品信息
     * @throws ServiceException 服务异常
     */
    public void commodityActualFinishTime(long shopId, CommodityInfo commodityInfo) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("commodityInfo", commodityInfo);
        call("eleme.order.commodityActualFinishTime", params);
    }

    /**
     * 匿名订单查询可用虚拟小号
     *
     * @param orderId 订单Id
     * @return 虚拟小号
     * @throws ServiceException 服务异常
     */
    public String queryCallAvailable(String orderId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        return call("eleme.order.queryCallAvailable", params);
    }

    /**
     * 批量获取订单退款信息V2
     *
     * @param orderIds 订单Id列表
     * @return 订单Id和该订单的退款信息
     * @throws ServiceException 服务异常
     */
    public Map<String,ORefundOrder> batchGetRefundOrders(List<String> orderIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderIds", orderIds);
        return call("eleme.order.batchGetRefundOrders", params);
    }

    /**
     * 获取订单退款金额分摊明细信息
     *
     * @param orderId 订单Id
     * @return 订单退款金额分摊明细信息
     * @throws ServiceException 服务异常
     */
    public ORefundAmountInfo queryRefundAmountInfo(String orderId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        return call("eleme.order.queryRefundAmountInfo", params);
    }

    /**
     * 查询店铺订单出餐时长和推荐出餐时长
     *
     * @param shopId 店铺Id
     * @return 商家订单出餐时长配置
     * @throws ServiceException 服务异常
     */
    public OrderCookingTimeConfig queryCookingTimeConfig(String shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.order.queryCookingTimeConfig", params);
    }

    /**
     * 设置店铺高峰期出餐时长
     *
     * @param shopId 店铺Id
     * @param configs 高峰期出餐时长配置
     * @throws ServiceException 服务异常
     */
    public void setPeakPeriodCookingTime(String shopId, List<PeakCookTimeConfig> configs) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("configs", configs);
        call("eleme.order.setPeakPeriodCookingTime", params);
    }

    /**
     * 设置店铺非高峰期出餐时长
     *
     * @param shopId 店铺Id
     * @param config 非高峰期出餐配置
     * @throws ServiceException 服务异常
     */
    public void setNonPeakPeriodCookingTime(String shopId, CookingTimeConfig config) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("config", config);
        call("eleme.order.setNonPeakPeriodCookingTime", params);
    }

    /**
     * 商户设置延长出餐
     *
     * @param request 延长出餐时长
     * @throws ServiceException 服务异常
     */
    public void setDelayCookingTimeSetting(SetDelayCookingSettingRequest request) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("request", request);
        call("eleme.order.setDelayCookingTimeSetting", params);
    }

    /**
     * 批量获取订单地址
     *
     * @param orderIds 查询订单对应地址信息
     * @return 订单ID 和 地址
     * @throws ServiceException 服务异常
     */
    public Map<String,String> batchGetOrderAddress(List<String> orderIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderIds", orderIds);
        return call("eleme.order.batchGetOrderAddress", params);
    }

    /**
     * 查询订单退款商品明细
     *
     * @param orderId 订单Id
     * @return 订单退款商品明细
     * @throws ServiceException 服务异常
     */
    public RefundFoodInfos queryRefundFoodInfos(String orderId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        return call("eleme.order.queryRefundFoodInfos", params);
    }

    /**
     * 查询自配送店铺送达准时率信息
     *
     * @param shopId 店铺Id
     * @return 自配送店铺送达准时率信息
     * @throws ServiceException 服务异常
     */
    public SelfDeliveryShopOnTimeRate querySelfDeliveryOnTimeRate(String shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.order.querySelfDeliveryOnTimeRate", params);
    }

    /**
     * 查询全部订单
     *
     * @param shopId 店铺Id
     * @param pageNo 页码。取值范围:大于零的整数最大限制为100
     * @param pageSize 每页获取条数。最小值1，最大值50。
     * @param date 日期,默认当天,格式:yyyy-MM-dd
     * @return 订单分页数据
     * @throws ServiceException 服务异常
     */
    public OrderList getAllOrders(long shopId, int pageNo, int pageSize, String date) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("pageNo", pageNo);
        params.put("pageSize", pageSize);
        params.put("date", date);
        return call("eleme.order.getAllOrders", params);
    }

    /**
     * 批量获取订单
     *
     * @param orderIds 订单Id的列表
     * @return 订单列表
     * @throws ServiceException 服务异常
     */
    public Map<String,OOrder> mgetOrders(List<String> orderIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderIds", orderIds);
        return call("eleme.order.mgetOrders", params);
    }

    /**
     * 自配送订单送达凭证上传
     *
     * @param content 凭证内容(如果是image 内容为文件内容base64编码值，如果是 text，即文本内容，限 100 字以内)
     * @param proofType 凭证类型
     * @param orderId 订单号
     * @throws ServiceException 服务异常
     */
    public void selfDeliveryFinishProofSync(String content, String proofType, String orderId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("content", content);
        params.put("proofType", proofType);
        params.put("orderId", orderId);
        call("eleme.order.selfDeliveryFinishProofSync", params);
    }

    /**
     * 自配送商家同步运单的状态信息
     *
     * @param shopId 店铺Id
     * @param stateInfo 运单状态信息
     * @throws ServiceException 服务异常
     */
    public void selfDeliveryStateSync(long shopId, OSelfDeliveryStateInfo stateInfo) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("stateInfo", stateInfo);
        call("eleme.order.selfDeliveryStateSync", params);
    }

    /**
     * 获取订单
     *
     * @param orderId 订单Id
     * @return 订单
     * @throws ServiceException 服务异常
     */
    public OOrder getOrder(String orderId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        return call("eleme.order.getOrder", params);
    }

    /**
     * 设置订单自取码
     *
     * @param pickupQrCode 取餐二维码
     * @param pickupCode 取餐号
     * @param orderId 订单Id
     * @throws ServiceException 服务异常
     */
    public void setOrderPickupCode(String pickupQrCode, String pickupCode, String orderId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("pickupQrCode", pickupQrCode);
        params.put("pickupCode", pickupCode);
        params.put("orderId", orderId);
        call("eleme.order.setOrderPickupCode", params);
    }

    /**
     * 设置订单取餐完成
     *
     * @param orderId 订单号
     * @throws ServiceException 服务异常
     */
    public void setOrderPickupComplete(String orderId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        call("eleme.order.setOrderPickupComplete", params);
    }

    /**
     * 取消订单(推荐)
     *
     * @param orderId 订单Id
     * @param type 取消原因
     * @param remark 备注说明
     * @throws ServiceException 服务异常
     */
    public void cancelOrderLite(String orderId, OInvalidateType type, String remark) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        params.put("type", type);
        params.put("remark", remark);
        call("eleme.order.cancelOrderLite", params);
    }
}
