/*
 Syntax is distributed under the Revised, or 3-clause BSD license
 ===============================================================================
 Copyright (c) 1985, 2012, 2016, Jaime Garza
 All rights reserved.
 
 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions are met:
     * Redistributions of source code must retain the above copyright
       notice, this list of conditions and the following disclaimer.
     * Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.
     * Neither the name of Jaime Garza nor the
       names of its contributors may be used to endorse or promote products
       derived from this software without specific prior written permission.
 
 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
 DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ===============================================================================
*/
package me.jaimegarza.syntax.generator;

import java.io.IOException;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import me.jaimegarza.syntax.exception.ParsingException;
import me.jaimegarza.syntax.model.parser.Associativity;
import me.jaimegarza.syntax.model.parser.ErrorToken;
import me.jaimegarza.syntax.model.parser.NonTerminal;
import me.jaimegarza.syntax.model.parser.Rule;
import me.jaimegarza.syntax.model.parser.RuleItem;
import me.jaimegarza.syntax.model.parser.Symbol;
import me.jaimegarza.syntax.model.parser.Terminal;
import me.jaimegarza.syntax.model.parser.Type;
import me.jaimegarza.syntax.model.graph.Construct;
import me.jaimegarza.syntax.model.graph.Nfa;
import me.jaimegarza.syntax.model.graph.Dfa;
import me.jaimegarza.syntax.model.graph.NfaNode;
import me.jaimegarza.syntax.model.graph.DfaNode;
import me.jaimegarza.syntax.model.graph.symbol.CharacterClass;
import me.jaimegarza.syntax.env.Environment;
import me.jaimegarza.syntax.util.NfaUtil;

/**
 * Parser for a grammar.<p>
 *
 * Phases:
 * 
 * <ol>
 *   <li><b>Code Parser</b> (This Phase)
 *   <li>Structural Analysis
 *   <li>Table Generation
 *   <li>Writing Code
 * </ol>
 * @author jaimegarza@gmail.com
 *
 */
@SuppressWarnings("unused")
public class SyntaxCodeParser extends AbstractCodeParser {

  private String currentTypeName = null;
  private CharacterClass currentCharacterClass = null;
  private LexerMode lexerMode = LexerMode.NORMAL;
  private Nfa currentNfa = null;
  private Terminal currentTerminal = null;
  private Dfa currentDfa = null;
  private String currentRegex;
  
  private enum LexerMode {
    NORMAL, REGEX, CHARACTER_CLASS
  }
  
  public SyntaxCodeParser(Environment env) {
    super(env);
  }
  
  public static class StackElement {
      public int stateNumber;
      public int value;
      public boolean mustClose;
      public String id;
      public char c;
      public List<String> ids = new LinkedList<String>();
      public Symbol symbol;
      public CharacterClass cc;
      public Construct construct;
  
      StackElement(int state, int value, boolean mustClose, String id, char c) {
        super();
        this.stateNumber = state;
        this.value = value;
        this.mustClose = mustClose;
        this.id = id;
        this.c = c;
      }  
  
      StackElement(Construct construct) {
        super();
        this.construct = construct;
      }  
  
      StackElement(CharacterClass cc, String id) {
        super();
        this.cc = cc;
        this.id = id;
      }  
  
      StackElement() {
        super();
      }  
  
      @Override
      public String toString() {
        StringBuilder sb = new StringBuilder();
        boolean comma = false;
        if (stateNumber != -1) {
          sb.append("state: ").append(stateNumber);
          comma = true;
        }  
        if (comma)
          sb.append(", ");
        sb.append("value: ").append(value).append(", mustClose: ").append(mustClose);
        if (c != '\0')
          sb.append(", c:\'").append(""+c).append("\'");;
        if (id != null) {
          sb.append(", id:").append(id);
        }  
        if (symbol != null) {
          sb.append(", symbol:").append(symbol);
        }  
        if (cc != null) {
          sb.append(", cc:").append(cc);
        }  
        if (construct != null) {
          sb.append(", construct:").append(construct);
        }  
        
        return sb.toString();
      }  
  }


  // Code Generator
  private static final int STACK_DEPTH = 5000;
  StackElement stack[] = new StackElement[STACK_DEPTH];
  int stackTop;

  boolean generateCode(int rule) {
    switch(rule){

      // 3. SectionMarker ->  MARK
      case 3:
        generateLexerFooter();
        break;
      // 4. Actions -> 
      case 4:
        {
                                                         generateCodeGeneratorFooter();
                                                         finalActions = false;
                                                       }
        break;
      // 5. Actions ->  MARK
      case 5:
        {
                                                         generateCodeGeneratorFooter();
                                                         finalActions = true;
                                                       }
        break;
      // 9. Declaration ->  START TOKEN
      case 9:
        if (!declareStart(stack[stackTop].id)) return false;
        break;
      // 11. Declaration ->  TYPE TYPENAME
      case 11:
        currentTypeName = stack[stackTop].id;
        break;
      // 12. Declaration ->  TYPE TYPENAME $code-fragment-1 NonTerminals
      case 12:
        currentTypeName = null;
        break;
      // 13. Declaration ->  DECLARE TYPENAME
      case 13:
        declareOneType(stack[stackTop].id);
        break;
      // 15. Declaration ->  UNION
      case 15:
        if (!generateStructure()) return false;
        break;
      // 16. Declaration ->  Precedence Definition
      case 16:
        currentType = null;
        break;
      // 17. Declaration ->  {
      case 17:
        if (!generateDeclaration()) return false;
        break;
      // 19. Declaration ->  GROUP TOKEN : TOKEN GroupTokens
      case 19:
        if(!groupTokens(stack[stackTop].ids, stack[stackTop-3].id, stack[stackTop-1].id)) return false;
        break;
      // 20. Precedence ->  TERM
      case 20:
        {
                                                         ruleAssociativity = Associativity.NONE;
                                                         isErrorToken = false;
                                                       }
        break;
      // 21. Precedence ->  LEFT
      case 21:
        {
                                                         rulePrecedence++;
                                                         ruleAssociativity = Associativity.LEFT;
                                                         isErrorToken = false;
                                                       }
        break;
      // 22. Precedence ->  RIGHT
      case 22:
        {
                                                         rulePrecedence++;
                                                         ruleAssociativity = Associativity.RIGHT;
                                                         isErrorToken = false;
                                                       }
        break;
      // 23. Precedence ->  BINARY
      case 23:
        {
                                                         rulePrecedence++;
                                                         ruleAssociativity = Associativity.BINARY;
                                                         isErrorToken = false;
                                                       }
        break;
      // 24. Precedence ->  ERRDEF
      case 24:
        {
                                                         ruleAssociativity = Associativity.NONE;
                                                         isErrorToken = true;
                                                       }
        break;
      // 25. NonTerminals ->  NonTerminals , TOKEN
      case 25:
        if (!declareOneNonTerminal(currentTypeName, stack[stackTop].id)) return false;
        break;
      // 26. NonTerminals ->  TOKEN
      case 26:
        if (!declareOneNonTerminal(currentTypeName, stack[stackTop].id)) return false;
        break;
      // 29. Name ->  TOKEN : TOKEN
      case 29:
        if (!nameOneNonTerminal(stack[stackTop-2].id, stack[stackTop].id)) return false;
        break;
      // 30. GroupTokens ->  GroupTokens , GroupToken
      case 30:
        stack[stackTop-2].ids.add(stack[stackTop].id);
        break;
      // 31. GroupTokens ->  GroupToken
      case 31:
        {
                                                         stack[stackTop].ids = new LinkedList<String>();
                                                         stack[stackTop].ids.add(stack[stackTop].id);
                                                       }
        break;
      // 32. GroupToken ->  TOKEN
      case 32:
        {
                                                         if (declareOneTerminal(stack[stackTop].id, false, Associativity.NONE, rulePrecedence, null, -1, null) == null) return false;
                                                         stack[stackTop].id = stack[stackTop].id;
                                                       }
        break;
      // 34. Type ->  TYPENAME
      case 34:
        {
                                                         currentType = new Type(stack[stackTop].id);
                                                         if (runtimeData.getTypes().contains(currentType)) {
                                                           currentType = runtimeData.getTypes().get(runtimeData.getTypes().indexOf(currentType));
                                                         } else {
                                                           runtimeData.getTypes().add(currentType);
                                                         }
                                                       }
        break;
      // 35. Type -> 
      case 35:
        currentType = null;
        break;
      // 39. TokenDef ->  TOKEN Number TokenName
      case 39:
        {
                                                         int number = -1;
                                                         if (stack[stackTop-1].value != -1) {
                                                           number = stack[stackTop-1].value;
                                                         } else if (stack[stackTop-2].value != -1) {
                                                           number = stack[stackTop-2].value;
                                                         }
                                                         currentTerminal = declareOneTerminal(stack[stackTop-2].id, isErrorToken, ruleAssociativity, rulePrecedence, currentType, number, stack[stackTop].id);
                                                         if (currentTerminal == null) return false;
                                                         stack[stackTop-2].symbol = currentTerminal;
                                                       }
        break;
      // 41. Number -> 
      case 41:
        stack[stackTop+1] = new StackElement(0, -1, false, "", '\0');
        break;
      // 42. TokenName ->  : TOKEN
      case 42:
        stack[stackTop-1].id = stack[stackTop].id;
        break;
      // 43. TokenName -> 
      case 43:
        stack[stackTop+1] = new StackElement(0, 0, false, "", '\0');
        break;
      // 44. LexicAction ->  Equals TOKEN
      case 44:
        stack[stackTop-1] = null;
        break;
      // 45. LexicAction ->  RStart RExp REnd
      case 45:
        generateDefaultRegexCode(currentDfa.getStart().getId(), currentTerminal);
        break;
      // 47. LexicAction -> 
      case 47:
        stack[stackTop+1] = new StackElement(0, -1, false, "", '\0');
        break;
      // 48. REquals ->  =
      case 48:
        generateRegexCode("default", currentDfa.getStart().getId(), currentTerminal);
        break;
      // 49. REquals ->  [ TOKEN ] =
      case 49:
        generateRegexCode(stack[stackTop-2].id, currentDfa.getStart().getId(), currentTerminal);
        break;
      // 50. Equals ->  =
      case 50:
        generateLexerCode("default", currentTerminal);
        break;
      // 51. Equals ->  [ TOKEN ] =
      case 51:
        generateLexerCode(stack[stackTop-2].id, currentTerminal);
        break;
      // 52. RStart ->  / TOKEN
      case 52:
        currentNfa = new Nfa();
        break;
      // 54. RExp ->  RegExp
      case 54:
        {
                                                         currentDfa = NfaUtil.finalize(currentNfa, currentRegex, stack[stackTop].construct);
                                                         runtimeData.addRegularExpression(currentDfa);
                                                       }
        break;
      // 55. RegExp ->  RegExp PIPE Concatenation
      case 55:
        stack[stackTop-2].construct = NfaUtil.alternate(currentNfa, stack[stackTop-2].construct, stack[stackTop].construct);
        break;
      // 56. RegExp ->  Concatenation
      case 56:
        stack[stackTop].construct = stack[stackTop].construct;
        break;
      // 57. Concatenation ->  Concatenation UnaryRegex
      case 57:
        stack[stackTop-1].construct = NfaUtil.concatenate(currentNfa, stack[stackTop-1].construct, stack[stackTop].construct);
        break;
      // 58. Concatenation ->  UnaryRegex
      case 58:
        stack[stackTop].construct = stack[stackTop].construct;
        break;
      // 59. UnaryRegex ->  BasicElement STAR
      case 59:
        stack[stackTop-1].construct = NfaUtil.zeroOrMany(currentNfa, stack[stackTop-1].construct);
        break;
      // 60. UnaryRegex ->  BasicElement PLUS
      case 60:
        stack[stackTop-1].construct = NfaUtil.oneOrMany(currentNfa, stack[stackTop-1].construct);
        break;
      // 61. UnaryRegex ->  BasicElement HUH
      case 61:
        stack[stackTop-1].construct = NfaUtil.optional(currentNfa, stack[stackTop-1].construct);
        break;
      // 62. UnaryRegex ->  BasicElement
      case 62:
        stack[stackTop].construct = stack[stackTop].construct;
        break;
      // 63. BasicElement ->  ( RegExp )
      case 63:
        stack[stackTop-2].construct = stack[stackTop-1].construct;
        break;
      // 64. BasicElement ->  CHAR
      case 64:
        stack[stackTop].construct = NfaUtil.character(currentNfa, stack[stackTop].c);
        break;
      // 65. BasicElement ->  [ CHARACTER_CLASS ]
      case 65:
        stack[stackTop-2].construct = NfaUtil.characterClass(currentNfa, stack[stackTop-1].cc);
        break;
      // 66. BasicElement ->  ANY
      case 66:
        stack[stackTop].construct = NfaUtil.any(currentNfa);
        break;
      // 67. Productions ->  Productions TOKEN : Rules ;
      case 67:
        if (!setLeftHandOfLastRule(stack[stackTop-3].id)) return false;
        break;
      // 68. Productions ->  TOKEN : Rules ;
      case 68:
        if (!setLeftHandOfLastRule(stack[stackTop-3].id)) return false;
        break;
      // 69. Rules ->  Rules | GrammarRule
      case 69:
        {
                                                         newRule();
                                                         bActionDone = false;
                                                       }
        break;
      // 70. Rules ->  GrammarRule
      case 70:
        {
                                                         newRule();
                                                         currentRuleIndex = runtimeData.getRules().size() - 1;
                                                         bActionDone = false;
                                                       }
        break;
      // 72. GrammarRule -> 
      case 72:
        {
                                                         bActionDone = false;
                                                       }
        break;
      // 75. Symbol ->  TOKEN
      case 75:
        {
                                                         if (stack[stackTop].id.length() == 0) {
                                                           break;
                                                         }
                                                         if (!declareOneItem(stack[stackTop].id, stack[stackTop].value, stack[stackTop].mustClose)) return false;
                                                       }
        break;
      // 76. Symbol ->  PREC TOKEN
      case 76:
        if(!computeAssociativityAndPrecedence(stack[stackTop].id)) return false;
        break;
      // 77. Symbol ->  =
      case 77:
        {
                                                         int i = runtimeData.currentRuleItems != null ? runtimeData.currentRuleItems.size() : 0;
                                                         if (!ruleAction(runtimeData.getRules().size(), i, currentNonTerminalName)) {
                                                           return false;
                                                         }
                                                         bActionDone = true;
                                                       }
        break;
    }
    return true; // OK
  }

  private static final int RECOVERS=0;

  // Contains tokens in compact mode, and column in matrix
  int recoverTable[] = {0};

  private static int TOKENS=38;
  private static int tokens[] = {
    0, // $ ($)
    32768, // MARK ('%%')
    32769, // START ('%start')
    32770, // TOKEN (token)
    32771, // TYPE ('%type')
    32772, // UNION ('%union')
    32773, // TYPENAME (type definition)
    32774, // TERM ('%token')
    32775, // LEFT ('%left')
    32776, // RIGHT ('%right')
    32777, // BINARY ('%binary')
    32778, // ERRDEF ('%error')
    32779, // NUM (number)
    32780, // PREC ('%prec')
    32781, // NAME ('%name')
    32782, // ERROR (error)
    32783, // LEXER (%lexer)
    32784, // DECLARE (%declare)
    32785, // GROUP (%group)
    32786, // PIPE ('|')
    32787, // STAR ('*')
    32788, // PLUS ('+')
    32789, // HUH ('?')
    32790, // ANY ('.')
    32791, // CHAR (character)
    32792, // CHARACTER_CLASS (a valid character class)
    32793, // LEXCODE (lexical code)
    59, // ; (semicolon)
    58, // : (colon)
    61, // = (equals sign)
    91, // [ (token mode marker)
    93, // ] (closing token mode marker)
    44, // , (comma)
    124, // | (rule separator ('|'))
    40, // ( (opening parenthesis)
    41, // ) (closing parenthesis)
    47, // / (regular expression marker ('/'))
    123 // { ({)
  };

  // Token definitions
  private static final int EOS=0;
  private static final int MARK=32768;
  private static final int START=32769;
  private static final int TOKEN=32770;
  private static final int TYPE=32771;
  private static final int UNION=32772;
  private static final int TYPENAME=32773;
  private static final int TERM=32774;
  private static final int LEFT=32775;
  private static final int RIGHT=32776;
  private static final int BINARY=32777;
  private static final int ERRDEF=32778;
  private static final int NUM=32779;
  private static final int PREC=32780;
  private static final int NAME=32781;
  private static final int ERROR=32782;
  private static final int LEXER=32783;
  private static final int DECLARE=32784;
  private static final int GROUP=32785;
  private static final int PIPE=32786;
  private static final int STAR=32787;
  private static final int PLUS=32788;
  private static final int HUH=32789;
  private static final int ANY=32790;
  private static final int CHAR=32791;
  private static final int CHARACTER_CLASS=32792;
  private static final int LEXCODE=32793;

  private class TokenDef {
    int token;
    String name;
    String fullName;
    boolean reserved;

    TokenDef(String name, String fullName, int token, boolean reserved) {
      this.name = name;
      this.fullName = fullName;
      this.token = token;
      this.reserved = reserved;
    }
  }

  private TokenDef tokenDefs[] = {
    new TokenDef("EOS", "$", 0, true),
    new TokenDef("MARK", "'%%'", 32768, true),
    new TokenDef("START", "'%start'", 32769, true),
    new TokenDef("TOKEN", "token", 32770, true),
    new TokenDef("TYPE", "'%type'", 32771, true),
    new TokenDef("UNION", "'%union'", 32772, true),
    new TokenDef("TYPENAME", "type definition", 32773, true),
    new TokenDef("TERM", "'%token'", 32774, true),
    new TokenDef("LEFT", "'%left'", 32775, true),
    new TokenDef("RIGHT", "'%right'", 32776, true),
    new TokenDef("BINARY", "'%binary'", 32777, true),
    new TokenDef("ERRDEF", "'%error'", 32778, true),
    new TokenDef("NUM", "number", 32779, true),
    new TokenDef("PREC", "'%prec'", 32780, true),
    new TokenDef("NAME", "'%name'", 32781, true),
    new TokenDef("ERROR", "error", 32782, true),
    new TokenDef("LEXER", "%lexer", 32783, true),
    new TokenDef("DECLARE", "%declare", 32784, true),
    new TokenDef("GROUP", "%group", 32785, true),
    new TokenDef("PIPE", "'|'", 32786, true),
    new TokenDef("STAR", "'*'", 32787, true),
    new TokenDef("PLUS", "'+'", 32788, true),
    new TokenDef("HUH", "'?'", 32789, true),
    new TokenDef("ANY", "'.'", 32790, true),
    new TokenDef("CHAR", "character", 32791, true),
    new TokenDef("CHARACTER_CLASS", "a valid character class", 32792, true),
    new TokenDef("LEXCODE", "lexical code", 32793, true),
    new TokenDef(";", "semicolon", 59, false),
    new TokenDef(":", "colon", 58, false),
    new TokenDef("=", "equals sign", 61, false),
    new TokenDef("[", "token mode marker", 91, false),
    new TokenDef("]", "closing token mode marker", 93, false),
    new TokenDef(",", "comma", 44, false),
    new TokenDef("|", "rule separator ('|')", 124, false),
    new TokenDef("(", "opening parenthesis", 40, false),
    new TokenDef(")", "closing parenthesis", 41, false),
    new TokenDef("/", "regular expression marker ('/')", 47, false),
    new TokenDef("{", "{", 123, false)
  };

  private final int EDGES=0;

  private final int edgeTable[] = {
  };

  private final int VERTICES=0;

  private final int vertexTable[] = {
  };

  private static final int FINAL=127;
  private static final int SYMBS=72;

  private static final int ACCEPT=Integer.MAX_VALUE;

  // Parsing Table
  private int parsingTable[][] = {
          //       $     %% %start  token  %type %union type d %token  %left %right %binar %error number  %prec  %name  error %lexer %decla %group      |      *      +      ?      . charac a vali lexica semico  colon equals  token closin  comma rule s openin closin regula      { number a toke name o TokenD  basic concat  unary regula  group  token declar a decl code s token, parser lexica a toke  group  token preced a defi an equ an equ type d  token      /      / rule d rule d syntax end of produc $code-   RExp 
     /*  0*/ {     0,     5,     7,     0,     9,    12,     0,    17,    18,    19,    20,    21,     0,     0,    11,     0,    15,    10,    16,     0,     0,     0,     0,     0,     0,     0,     8,     6,     0,     0,     0,     0,     0,     0,     0,     0,     0,    14,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     2,     4,     0,     0,     0,     0,     0,     0,     0,    13,     0,     0,     0,     0,     0,     0,     0,     0,     0,     1,     3,     0,     0,     0},
     /*  1*/ {ACCEPT,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*  2*/ {     0,     5,     7,     0,     9,    12,     0,    17,    18,    19,    20,    21,     0,     0,    11,     0,    15,    10,    16,     0,     0,     0,     0,     0,     0,     0,     8,     6,     0,     0,     0,     0,     0,     0,     0,     0,     0,    14,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    23,     0,     0,     0,     0,     0,     0,     0,    13,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    22,     0,     0,     0},
     /*  3*/ {     0,     0,     0,    25,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    24,     0,     0},
     /*  4*/ {     0,    -7,    -7,    -7,    -7,    -7,    -7,    -7,    -7,    -7,    -7,    -7,     0,     0,    -7,     0,    -7,    -7,    -7,     0,     0,     0,     0,     0,     0,     0,    -7,    -7,    -7,    -7,    -7,    -7,     0,     0,     0,     0,     0,    -7,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*  5*/ {     0,     0,     0,    -3,     0,     0,     0,     0,     0,     0,     0,     0,     0,    -3,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    -3,    -3,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*  6*/ {     0,    -8,    -8,    -8,    -8,    -8,    -8,    -8,    -8,    -8,    -8,    -8,     0,     0,    -8,     0,    -8,    -8,    -8,     0,     0,     0,     0,     0,     0,     0,    -8,    -8,    -8,    -8,    -8,    -8,     0,     0,     0,     0,     0,    -8,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*  7*/ {     0,     0,     0,    26,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*  8*/ {     0,     0,     0,    27,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*  9*/ {     0,     0,     0,     0,     0,     0,    28,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 10*/ {     0,     0,     0,     0,     0,     0,    29,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 11*/ {     0,     0,     0,    32,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    31,     0,    30,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 12*/ {     0,   -15,   -15,   -15,   -15,   -15,   -15,   -15,   -15,   -15,   -15,   -15,     0,     0,   -15,     0,   -15,   -15,   -15,     0,     0,     0,     0,     0,     0,     0,   -15,   -15,   -15,   -15,   -15,   -15,     0,     0,     0,     0,     0,   -15,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 13*/ {     0,     0,     0,   -35,     0,     0,    35,     0,     0,     0,     0,     0,   -35,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -35,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    33,     0,     0,    34,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 14*/ {     0,   -17,   -17,   -17,   -17,   -17,   -17,   -17,   -17,   -17,   -17,   -17,     0,     0,   -17,     0,   -17,   -17,   -17,     0,     0,     0,     0,     0,     0,     0,   -17,   -17,   -17,   -17,   -17,   -17,     0,     0,     0,     0,     0,   -17,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 15*/ {     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    37,    38,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    36,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 16*/ {     0,     0,     0,    39,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 17*/ {     0,     0,     0,   -20,     0,     0,   -20,     0,     0,     0,     0,     0,   -20,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -20,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 18*/ {     0,     0,     0,   -21,     0,     0,   -21,     0,     0,     0,     0,     0,   -21,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -21,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 19*/ {     0,     0,     0,   -22,     0,     0,   -22,     0,     0,     0,     0,     0,   -22,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -22,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 20*/ {     0,     0,     0,   -23,     0,     0,   -23,     0,     0,     0,     0,     0,   -23,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -23,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 21*/ {     0,     0,     0,   -24,     0,     0,   -24,     0,     0,     0,     0,     0,   -24,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -24,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 22*/ {     0,     0,     0,    25,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    40,     0,     0},
     /* 23*/ {     0,    -6,    -6,    -6,    -6,    -6,    -6,    -6,    -6,    -6,    -6,    -6,     0,     0,    -6,     0,    -6,    -6,    -6,     0,     0,     0,     0,     0,     0,     0,    -6,    -6,    -6,    -6,    -6,    -6,     0,     0,     0,     0,     0,    -6,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 24*/ {    -4,    43,     0,    42,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    41,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 25*/ {     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    44,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 26*/ {     0,    -9,    -9,    -9,    -9,    -9,    -9,    -9,    -9,    -9,    -9,    -9,     0,     0,    -9,     0,    -9,    -9,    -9,     0,     0,     0,     0,     0,     0,     0,    -9,    -9,    -9,    -9,    -9,    -9,     0,     0,     0,     0,     0,    -9,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 27*/ {     0,   -10,   -10,   -10,   -10,   -10,   -10,   -10,   -10,   -10,   -10,   -10,     0,     0,   -10,     0,   -10,   -10,   -10,     0,     0,     0,     0,     0,     0,     0,   -10,   -10,   -10,   -10,   -10,   -10,     0,     0,     0,     0,     0,   -10,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 28*/ {     0,     0,     0,   -11,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    45,     0},
     /* 29*/ {     0,   -13,   -13,   -13,   -13,   -13,   -13,   -13,   -13,   -13,   -13,   -13,     0,     0,   -13,     0,   -13,   -13,   -13,     0,     0,     0,     0,     0,     0,     0,   -13,   -13,   -13,   -13,   -13,   -13,     0,     0,     0,     0,     0,   -13,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 30*/ {     0,   -14,   -14,   -14,   -14,   -14,   -14,   -14,   -14,   -14,   -14,   -14,     0,     0,   -14,     0,   -14,   -14,   -14,     0,     0,     0,     0,     0,     0,     0,   -14,   -14,   -14,   -14,   -14,   -14,    46,     0,     0,     0,     0,   -14,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 31*/ {     0,   -28,   -28,   -28,   -28,   -28,   -28,   -28,   -28,   -28,   -28,   -28,     0,     0,   -28,     0,   -28,   -28,   -28,     0,     0,     0,     0,     0,     0,     0,   -28,   -28,   -28,   -28,   -28,   -28,   -28,     0,     0,     0,     0,   -28,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 32*/ {     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    47,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 33*/ {     0,   -16,   -16,   -16,   -16,   -16,   -16,   -16,   -16,   -16,   -16,   -16,     0,     0,   -16,     0,   -16,   -16,   -16,     0,     0,     0,     0,     0,     0,     0,   -16,   -16,   -16,   -16,   -16,   -16,     0,     0,     0,     0,     0,   -16,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 34*/ {     0,     0,     0,    51,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    49,     0,    50,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    48,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 35*/ {     0,     0,     0,   -34,     0,     0,     0,     0,     0,     0,     0,     0,   -34,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -34,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 36*/ {     0,     0,     0,    52,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 37*/ {     0,     0,     0,   -50,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 38*/ {     0,     0,     0,    53,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 39*/ {     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    54,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 40*/ {    -4,    43,     0,    42,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    55,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 41*/ {    -2,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 42*/ {     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    56,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 43*/ {    -5,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 44*/ {     0,     0,     0,    61,     0,     0,     0,     0,     0,     0,     0,     0,     0,    62,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -72,     0,    63,     0,     0,     0,   -72,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    60,    58,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    57,    59,     0,     0,     0,     0,     0},
     /* 45*/ {     0,     0,     0,    65,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    64,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 46*/ {     0,     0,     0,    32,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    66,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 47*/ {     0,     0,     0,    67,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 48*/ {     0,   -33,   -33,   -33,   -33,   -33,   -33,   -33,   -33,   -33,   -33,   -33,     0,     0,   -33,     0,   -33,   -33,   -33,     0,     0,     0,     0,     0,     0,     0,   -33,   -33,   -33,   -33,   -33,   -33,    68,     0,     0,     0,     0,   -33,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 49*/ {     0,   -37,   -37,   -37,   -37,   -37,   -37,   -37,   -37,   -37,   -37,   -37,     0,     0,   -37,     0,   -37,   -37,   -37,     0,     0,     0,     0,     0,     0,     0,   -37,   -37,   -37,   -37,   -37,   -37,   -37,     0,     0,     0,     0,   -37,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 50*/ {     0,   -47,   -47,   -47,   -47,   -47,   -47,   -47,   -47,   -47,   -47,   -47,     0,     0,   -47,     0,   -47,   -47,   -47,     0,     0,     0,     0,     0,     0,     0,   -47,   -47,   -47,    37,    38,   -47,   -47,     0,     0,     0,    72,   -47,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    69,     0,     0,     0,     0,     0,     0,    70,     0,     0,    71,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 51*/ {     0,   -41,   -41,   -41,   -41,   -41,   -41,   -41,   -41,   -41,   -41,   -41,    74,     0,   -41,     0,   -41,   -41,   -41,     0,     0,     0,     0,     0,     0,     0,   -41,   -41,   -41,   -41,   -41,   -41,   -41,     0,     0,     0,   -41,   -41,    73,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 52*/ {     0,   -18,   -18,   -18,   -18,   -18,   -18,   -18,   -18,   -18,   -18,   -18,     0,     0,   -18,     0,   -18,   -18,   -18,     0,     0,     0,     0,     0,     0,     0,   -18,   -18,   -18,   -18,   -18,   -18,     0,     0,     0,     0,     0,   -18,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 53*/ {     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    75,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 54*/ {     0,     0,     0,    76,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 55*/ {    -1,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 56*/ {     0,     0,     0,    61,     0,     0,     0,     0,     0,     0,     0,     0,     0,    62,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -72,     0,    63,     0,     0,     0,   -72,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    60,    58,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    77,    59,     0,     0,     0,     0,     0},
     /* 57*/ {     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    78,     0,     0,     0,     0,     0,    79,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 58*/ {     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -70,     0,     0,     0,     0,     0,   -70,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 59*/ {     0,     0,     0,    61,     0,     0,     0,     0,     0,     0,     0,     0,     0,    62,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -71,     0,    63,     0,     0,     0,   -71,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    80,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 60*/ {     0,     0,     0,   -74,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -74,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -74,     0,   -74,     0,     0,     0,   -74,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 61*/ {     0,     0,     0,   -75,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -75,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -75,     0,   -75,     0,     0,     0,   -75,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 62*/ {     0,     0,     0,    81,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 63*/ {     0,     0,     0,   -77,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -77,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -77,     0,   -77,     0,     0,     0,   -77,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 64*/ {     0,   -12,   -12,   -12,   -12,   -12,   -12,   -12,   -12,   -12,   -12,   -12,     0,     0,   -12,     0,   -12,   -12,   -12,     0,     0,     0,     0,     0,     0,     0,   -12,   -12,   -12,   -12,   -12,   -12,    82,     0,     0,     0,     0,   -12,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 65*/ {     0,   -26,   -26,   -26,   -26,   -26,   -26,   -26,   -26,   -26,   -26,   -26,     0,     0,   -26,     0,   -26,   -26,   -26,     0,     0,     0,     0,     0,     0,     0,   -26,   -26,   -26,   -26,   -26,   -26,   -26,     0,     0,     0,     0,   -26,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 66*/ {     0,   -27,   -27,   -27,   -27,   -27,   -27,   -27,   -27,   -27,   -27,   -27,     0,     0,   -27,     0,   -27,   -27,   -27,     0,     0,     0,     0,     0,     0,     0,   -27,   -27,   -27,   -27,   -27,   -27,   -27,     0,     0,     0,     0,   -27,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 67*/ {     0,   -29,   -29,   -29,   -29,   -29,   -29,   -29,   -29,   -29,   -29,   -29,     0,     0,   -29,     0,   -29,   -29,   -29,     0,     0,     0,     0,     0,     0,     0,   -29,   -29,   -29,   -29,   -29,   -29,   -29,     0,     0,     0,     0,   -29,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 68*/ {     0,     0,     0,    51,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    83,     0,    50,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 69*/ {     0,   -38,   -38,   -38,   -38,   -38,   -38,   -38,   -38,   -38,   -38,   -38,     0,     0,   -38,     0,   -38,   -38,   -38,     0,     0,     0,     0,     0,     0,     0,   -38,   -38,   -38,   -38,   -38,   -38,   -38,     0,     0,     0,     0,   -38,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 70*/ {     0,     0,     0,    84,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 71*/ {     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    93,    91,     0,     0,     0,     0,     0,    92,     0,     0,     0,    90,     0,     0,     0,     0,     0,     0,     0,    89,    87,    88,    86,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    85},
     /* 72*/ {     0,     0,     0,    94,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 73*/ {     0,   -43,   -43,   -43,   -43,   -43,   -43,   -43,   -43,   -43,   -43,   -43,     0,     0,   -43,     0,   -43,   -43,   -43,     0,     0,     0,     0,     0,     0,     0,   -43,   -43,    96,   -43,   -43,   -43,   -43,     0,     0,     0,   -43,   -43,     0,     0,    95,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 74*/ {     0,   -40,   -40,   -40,   -40,   -40,   -40,   -40,   -40,   -40,   -40,   -40,     0,     0,   -40,     0,   -40,   -40,   -40,     0,     0,     0,     0,     0,     0,     0,   -40,   -40,   -40,   -40,   -40,   -40,   -40,     0,     0,     0,   -40,   -40,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 75*/ {     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    97,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 76*/ {     0,     0,     0,   100,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    98,    99,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 77*/ {     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   101,     0,     0,     0,     0,     0,    79,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 78*/ {   -68,   -68,     0,   -68,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 79*/ {     0,     0,     0,    61,     0,     0,     0,     0,     0,     0,     0,     0,     0,    62,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -72,     0,    63,     0,     0,     0,   -72,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    60,   102,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    59,     0,     0,     0,     0,     0},
     /* 80*/ {     0,     0,     0,   -73,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -73,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -73,     0,   -73,     0,     0,     0,   -73,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 81*/ {     0,     0,     0,   -76,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -76,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -76,     0,   -76,     0,     0,     0,   -76,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 82*/ {     0,     0,     0,   103,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 83*/ {     0,   -36,   -36,   -36,   -36,   -36,   -36,   -36,   -36,   -36,   -36,   -36,     0,     0,   -36,     0,   -36,   -36,   -36,     0,     0,     0,     0,     0,     0,     0,   -36,   -36,   -36,   -36,   -36,   -36,   -36,     0,     0,     0,     0,   -36,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 84*/ {     0,   -44,   -44,   -44,   -44,   -44,   -44,   -44,   -44,   -44,   -44,   -44,     0,     0,   -44,     0,   -44,   -44,   -44,     0,     0,     0,     0,     0,     0,     0,   -44,   -44,   -44,   -44,   -44,   -44,   -44,     0,     0,     0,     0,   -44,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 85*/ {     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   105,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   104,     0,     0,     0,     0,     0,     0,     0},
     /* 86*/ {     0,     0,     0,   -54,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   106,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -54,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 87*/ {     0,     0,     0,   -56,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -56,     0,     0,     0,    93,    91,     0,     0,     0,     0,     0,    92,     0,     0,     0,    90,   -56,   -56,     0,     0,     0,     0,     0,    89,     0,   107,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 88*/ {     0,     0,     0,   -58,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -58,     0,     0,     0,   -58,   -58,   -58,     0,     0,     0,     0,   -58,   -58,     0,     0,   -58,   -58,   -58,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 89*/ {     0,     0,     0,   -62,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -62,   108,   109,   110,   -62,   -62,   -62,     0,     0,     0,     0,   -62,   -62,     0,     0,   -62,   -62,   -62,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 90*/ {     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    93,    91,     0,     0,     0,     0,     0,    92,     0,     0,     0,    90,     0,     0,     0,     0,     0,     0,     0,    89,    87,    88,   111,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 91*/ {     0,     0,     0,   -64,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -64,   -64,   -64,   -64,   -64,   -64,   -64,     0,     0,     0,     0,   -64,   -64,     0,     0,   -64,   -64,   -64,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 92*/ {     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   112,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 93*/ {     0,     0,     0,   -66,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -66,   -66,   -66,   -66,   -66,   -66,   -66,     0,     0,     0,     0,   -66,   -66,     0,     0,   -66,   -66,   -66,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 94*/ {     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -52,   -52,   -52,     0,     0,     0,     0,   -52,   -52,     0,     0,   -52,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 95*/ {     0,   -39,   -39,   -39,   -39,   -39,   -39,   -39,   -39,   -39,   -39,   -39,     0,     0,   -39,     0,   -39,   -39,   -39,     0,     0,     0,     0,     0,     0,     0,   -39,   -39,   -39,   -39,   -39,   -39,   -39,     0,     0,     0,   -39,   -39,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 96*/ {     0,     0,     0,   113,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 97*/ {     0,     0,     0,   -51,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 98*/ {     0,   -19,   -19,   -19,   -19,   -19,   -19,   -19,   -19,   -19,   -19,   -19,     0,     0,   -19,     0,   -19,   -19,   -19,     0,     0,     0,     0,     0,     0,     0,   -19,   -19,   -19,   -19,   -19,   -19,   114,     0,     0,     0,     0,   -19,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /* 99*/ {     0,   -31,   -31,   -31,   -31,   -31,   -31,   -31,   -31,   -31,   -31,   -31,     0,     0,   -31,     0,   -31,   -31,   -31,     0,     0,     0,     0,     0,     0,     0,   -31,   -31,   -31,   -31,   -31,   -31,   -31,     0,     0,     0,     0,   -31,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*100*/ {     0,   -32,   -32,   -32,   -32,   -32,   -32,   -32,   -32,   -32,   -32,   -32,     0,     0,   -32,     0,   -32,   -32,   -32,     0,     0,     0,     0,     0,     0,     0,   -32,   -32,   -32,   -32,   -32,   -32,   -32,     0,     0,     0,     0,   -32,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*101*/ {   -67,   -67,     0,   -67,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*102*/ {     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -69,     0,     0,     0,     0,     0,   -69,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*103*/ {     0,   -25,   -25,   -25,   -25,   -25,   -25,   -25,   -25,   -25,   -25,   -25,     0,     0,   -25,     0,   -25,   -25,   -25,     0,     0,     0,     0,     0,     0,     0,   -25,   -25,   -25,   -25,   -25,   -25,   -25,     0,     0,     0,     0,   -25,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*104*/ {     0,   -45,   -45,   -45,   -45,   -45,   -45,   -45,   -45,   -45,   -45,   -45,     0,     0,   -45,     0,   -45,   -45,   -45,     0,     0,     0,     0,     0,     0,     0,   -45,   -45,   -45,   116,   117,   -45,   -45,     0,     0,     0,     0,   -45,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   115,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*105*/ {     0,     0,     0,   118,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*106*/ {     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,    93,    91,     0,     0,     0,     0,     0,    92,     0,     0,     0,    90,     0,     0,     0,     0,     0,     0,     0,    89,   119,    88,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*107*/ {     0,     0,     0,   -57,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -57,     0,     0,     0,   -57,   -57,   -57,     0,     0,     0,     0,   -57,   -57,     0,     0,   -57,   -57,   -57,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*108*/ {     0,     0,     0,   -59,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -59,     0,     0,     0,   -59,   -59,   -59,     0,     0,     0,     0,   -59,   -59,     0,     0,   -59,   -59,   -59,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*109*/ {     0,     0,     0,   -60,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -60,     0,     0,     0,   -60,   -60,   -60,     0,     0,     0,     0,   -60,   -60,     0,     0,   -60,   -60,   -60,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*110*/ {     0,     0,     0,   -61,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -61,     0,     0,     0,   -61,   -61,   -61,     0,     0,     0,     0,   -61,   -61,     0,     0,   -61,   -61,   -61,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*111*/ {     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   106,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   120,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*112*/ {     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   121,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*113*/ {     0,   -42,   -42,   -42,   -42,   -42,   -42,   -42,   -42,   -42,   -42,   -42,     0,     0,   -42,     0,   -42,   -42,   -42,     0,     0,     0,     0,     0,     0,     0,   -42,   -42,   -42,   -42,   -42,   -42,   -42,     0,     0,     0,   -42,   -42,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*114*/ {     0,     0,     0,   100,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   122,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*115*/ {     0,     0,     0,   123,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*116*/ {     0,     0,     0,   -48,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*117*/ {     0,     0,     0,   124,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*118*/ {     0,   -53,   -53,   -53,   -53,   -53,   -53,   -53,   -53,   -53,   -53,   -53,     0,     0,   -53,     0,   -53,   -53,   -53,     0,     0,     0,     0,     0,     0,     0,   -53,   -53,   -53,   -53,   -53,   -53,   -53,     0,     0,     0,     0,   -53,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*119*/ {     0,     0,     0,   -55,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -55,     0,     0,     0,    93,    91,     0,     0,     0,     0,     0,    92,     0,     0,     0,    90,   -55,   -55,     0,     0,     0,     0,     0,    89,     0,   107,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*120*/ {     0,     0,     0,   -63,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -63,   -63,   -63,   -63,   -63,   -63,   -63,     0,     0,     0,     0,   -63,   -63,     0,     0,   -63,   -63,   -63,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*121*/ {     0,     0,     0,   -65,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   -65,   -65,   -65,   -65,   -65,   -65,   -65,     0,     0,     0,     0,   -65,   -65,     0,     0,   -65,   -65,   -65,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*122*/ {     0,   -30,   -30,   -30,   -30,   -30,   -30,   -30,   -30,   -30,   -30,   -30,     0,     0,   -30,     0,   -30,   -30,   -30,     0,     0,     0,     0,     0,     0,     0,   -30,   -30,   -30,   -30,   -30,   -30,   -30,     0,     0,     0,     0,   -30,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*123*/ {     0,   -46,   -46,   -46,   -46,   -46,   -46,   -46,   -46,   -46,   -46,   -46,     0,     0,   -46,     0,   -46,   -46,   -46,     0,     0,     0,     0,     0,     0,     0,   -46,   -46,   -46,   -46,   -46,   -46,   -46,     0,     0,     0,     0,   -46,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*124*/ {     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   125,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*125*/ {     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,   126,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0},
     /*126*/ {     0,     0,     0,   -49,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0}
  };
  
// Parsing Errors
  private int parsingError[] = {
     /*   0 */ 0, // Expecting declarations
     /*   1 */ 1, // No more elements expected
     /*   2 */ 2, // Expecting a declaration
     /*   3 */ 3, // token expected
     /*   4 */ 4, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, { may be missing
     /*   5 */ 5, // One of token, '%prec', colon, equals sign may be missing
     /*   6 */ 4, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, { may be missing
     /*   7 */ 3, // token expected
     /*   8 */ 3, // token expected
     /*   9 */ 6, // type definition expected
     /*  10 */ 6, // type definition expected
     /*  11 */ 3, // token expected
     /*  12 */ 4, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, { may be missing
     /*  13 */ 6, // type definition expected
     /*  14 */ 4, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, { may be missing
     /*  15 */ 7, // Expecting an equals sign (starting of a token action)
     /*  16 */ 3, // token expected
     /*  17 */ 8, // One of token, type definition, number, colon may be missing
     /*  18 */ 8, // One of token, type definition, number, colon may be missing
     /*  19 */ 8, // One of token, type definition, number, colon may be missing
     /*  20 */ 8, // One of token, type definition, number, colon may be missing
     /*  21 */ 8, // One of token, type definition, number, colon may be missing
     /*  22 */ 3, // token expected
     /*  23 */ 4, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, { may be missing
     /*  24 */ 9, // Expecting code section
     /*  25 */ 10, // colon expected
     /*  26 */ 4, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, { may be missing
     /*  27 */ 4, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, { may be missing
     /*  28 */ 11, // Expecting $code-fragment-1
     /*  29 */ 4, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, { may be missing
     /*  30 */ 12, // comma expected
     /*  31 */ 13, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, comma, { may be missing
     /*  32 */ 10, // colon expected
     /*  33 */ 4, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, { may be missing
     /*  34 */ 3, // token expected
     /*  35 */ 14, // One of token, number, colon may be missing
     /*  36 */ 3, // token expected
     /*  37 */ 3, // token expected
     /*  38 */ 3, // token expected
     /*  39 */ 10, // colon expected
     /*  40 */ 9, // Expecting code section
     /*  41 */ 15, // $ expected
     /*  42 */ 10, // colon expected
     /*  43 */ 15, // $ expected
     /*  44 */ 16, // token, '%prec' or equals sign expected
     /*  45 */ 3, // token expected
     /*  46 */ 3, // token expected
     /*  47 */ 3, // token expected
     /*  48 */ 12, // comma expected
     /*  49 */ 13, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, comma, { may be missing
     /*  50 */ 17, // Expecting lexical token definition
     /*  51 */ 18, // number expected
     /*  52 */ 4, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, { may be missing
     /*  53 */ 19, // closing token mode marker expected
     /*  54 */ 3, // token expected
     /*  55 */ 15, // $ expected
     /*  56 */ 16, // token, '%prec' or equals sign expected
     /*  57 */ 20, // semicolon or rule separator ('|') expected
     /*  58 */ 21, // One of semicolon, rule separator ('|') may be missing
     /*  59 */ 22, // Expecting token, precedence declaration or '='
     /*  60 */ 23, // One of token, '%prec', semicolon, equals sign, rule separator ('|') may be missing
     /*  61 */ 23, // One of token, '%prec', semicolon, equals sign, rule separator ('|') may be missing
     /*  62 */ 3, // token expected
     /*  63 */ 23, // One of token, '%prec', semicolon, equals sign, rule separator ('|') may be missing
     /*  64 */ 12, // comma expected
     /*  65 */ 13, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, comma, { may be missing
     /*  66 */ 13, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, comma, { may be missing
     /*  67 */ 13, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, comma, { may be missing
     /*  68 */ 3, // token expected
     /*  69 */ 13, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, comma, { may be missing
     /*  70 */ 3, // token expected
     /*  71 */ 24, // '.', character, token mode marker or opening parenthesis expected
     /*  72 */ 3, // token expected
     /*  73 */ 10, // colon expected
     /*  74 */ 25, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, comma, regular expression marker ('/'), { may be missing
     /*  75 */ 26, // equals sign expected
     /*  76 */ 3, // token expected
     /*  77 */ 20, // semicolon or rule separator ('|') expected
     /*  78 */ 27, // One of $, '%%', token may be missing
     /*  79 */ 22, // Expecting token, precedence declaration or '='
     /*  80 */ 23, // One of token, '%prec', semicolon, equals sign, rule separator ('|') may be missing
     /*  81 */ 23, // One of token, '%prec', semicolon, equals sign, rule separator ('|') may be missing
     /*  82 */ 3, // token expected
     /*  83 */ 13, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, comma, { may be missing
     /*  84 */ 13, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, comma, { may be missing
     /*  85 */ 28, // regular expression marker ('/') expected
     /*  86 */ 29, // '|' expected
     /*  87 */ 30, // Expecting basic element
     /*  88 */ 31, // One of token, '|', '.', character, a valid character class, token mode marker, closing token mode marker, opening parenthesis, closing parenthesis, regular expression marker ('/') may be missing
     /*  89 */ 32, // '*', '+' or '?' expected
     /*  90 */ 30, // Expecting basic element
     /*  91 */ 33, // One of token, '|', '*', '+', '?', '.', character, a valid character class, token mode marker, closing token mode marker, opening parenthesis, closing parenthesis, regular expression marker ('/') may be missing
     /*  92 */ 34, // a valid character class expected
     /*  93 */ 33, // One of token, '|', '*', '+', '?', '.', character, a valid character class, token mode marker, closing token mode marker, opening parenthesis, closing parenthesis, regular expression marker ('/') may be missing
     /*  94 */ 35, // One of '.', character, a valid character class, token mode marker, closing token mode marker, opening parenthesis may be missing
     /*  95 */ 25, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, comma, regular expression marker ('/'), { may be missing
     /*  96 */ 3, // token expected
     /*  97 */ 3, // token expected
     /*  98 */ 12, // comma expected
     /*  99 */ 13, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, comma, { may be missing
     /* 100 */ 13, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, comma, { may be missing
     /* 101 */ 27, // One of $, '%%', token may be missing
     /* 102 */ 21, // One of semicolon, rule separator ('|') may be missing
     /* 103 */ 13, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, comma, { may be missing
     /* 104 */ 7, // Expecting an equals sign (starting of a token action)
     /* 105 */ 3, // token expected
     /* 106 */ 30, // Expecting basic element
     /* 107 */ 31, // One of token, '|', '.', character, a valid character class, token mode marker, closing token mode marker, opening parenthesis, closing parenthesis, regular expression marker ('/') may be missing
     /* 108 */ 31, // One of token, '|', '.', character, a valid character class, token mode marker, closing token mode marker, opening parenthesis, closing parenthesis, regular expression marker ('/') may be missing
     /* 109 */ 31, // One of token, '|', '.', character, a valid character class, token mode marker, closing token mode marker, opening parenthesis, closing parenthesis, regular expression marker ('/') may be missing
     /* 110 */ 31, // One of token, '|', '.', character, a valid character class, token mode marker, closing token mode marker, opening parenthesis, closing parenthesis, regular expression marker ('/') may be missing
     /* 111 */ 36, // '|' or closing parenthesis expected
     /* 112 */ 19, // closing token mode marker expected
     /* 113 */ 25, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, comma, regular expression marker ('/'), { may be missing
     /* 114 */ 3, // token expected
     /* 115 */ 3, // token expected
     /* 116 */ 3, // token expected
     /* 117 */ 3, // token expected
     /* 118 */ 13, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, comma, { may be missing
     /* 119 */ 30, // Expecting basic element
     /* 120 */ 33, // One of token, '|', '*', '+', '?', '.', character, a valid character class, token mode marker, closing token mode marker, opening parenthesis, closing parenthesis, regular expression marker ('/') may be missing
     /* 121 */ 33, // One of token, '|', '*', '+', '?', '.', character, a valid character class, token mode marker, closing token mode marker, opening parenthesis, closing parenthesis, regular expression marker ('/') may be missing
     /* 122 */ 13, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, comma, { may be missing
     /* 123 */ 13, // One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, comma, { may be missing
     /* 124 */ 19, // closing token mode marker expected
     /* 125 */ 26, // equals sign expected
     /* 126 */ 3  // token expected
  };

  // Error Messages
  private String errorTable[] = {
     /* 0 */ "Expecting declarations",
     /* 1 */ "No more elements expected",
     /* 2 */ "Expecting a declaration",
     /* 3 */ "token expected",
     /* 4 */ "One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, { may be missing",
     /* 5 */ "One of token, '%prec', colon, equals sign may be missing",
     /* 6 */ "type definition expected",
     /* 7 */ "Expecting an equals sign (starting of a token action)",
     /* 8 */ "One of token, type definition, number, colon may be missing",
     /* 9 */ "Expecting code section",
     /* 10 */ "colon expected",
     /* 11 */ "Expecting $code-fragment-1",
     /* 12 */ "comma expected",
     /* 13 */ "One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, comma, { may be missing",
     /* 14 */ "One of token, number, colon may be missing",
     /* 15 */ "$ expected",
     /* 16 */ "token, '%prec' or equals sign expected",
     /* 17 */ "Expecting lexical token definition",
     /* 18 */ "number expected",
     /* 19 */ "closing token mode marker expected",
     /* 20 */ "semicolon or rule separator ('|') expected",
     /* 21 */ "One of semicolon, rule separator ('|') may be missing",
     /* 22 */ "Expecting token, precedence declaration or '='",
     /* 23 */ "One of token, '%prec', semicolon, equals sign, rule separator ('|') may be missing",
     /* 24 */ "'.', character, token mode marker or opening parenthesis expected",
     /* 25 */ "One of '%%', '%start', token, '%type', '%union', type definition, '%token', '%left', '%right', '%binary', '%error', '%name', %lexer, %declare, %group, lexical code, semicolon, colon, equals sign, token mode marker, closing token mode marker, comma, regular expression marker ('/'), { may be missing",
     /* 26 */ "equals sign expected",
     /* 27 */ "One of $, '%%', token may be missing",
     /* 28 */ "regular expression marker ('/') expected",
     /* 29 */ "'|' expected",
     /* 30 */ "Expecting basic element",
     /* 31 */ "One of token, '|', '.', character, a valid character class, token mode marker, closing token mode marker, opening parenthesis, closing parenthesis, regular expression marker ('/') may be missing",
     /* 32 */ "'*', '+' or '?' expected",
     /* 33 */ "One of token, '|', '*', '+', '?', '.', character, a valid character class, token mode marker, closing token mode marker, opening parenthesis, closing parenthesis, regular expression marker ('/') may be missing",
     /* 34 */ "a valid character class expected",
     /* 35 */ "One of '.', character, a valid character class, token mode marker, closing token mode marker, opening parenthesis may be missing",
     /* 36 */ "'|' or closing parenthesis expected"
  };

  // symbols and reductions table
  private class Grammar {
    int symbol;
    int reductions;

    Grammar(int symbol, int reductions) {
      this.symbol = symbol;
      this.reductions = reductions;
    }
  }

  private Grammar grammarTable[]={
    /*Rule   0 */ new Grammar(    72,      1),
    /*Rule   1 */ new Grammar(    67,      4),
    /*Rule   2 */ new Grammar(    67,      3),
    /*Rule   3 */ new Grammar(    68,      1),
    /*Rule   4 */ new Grammar(    50,      0),
    /*Rule   5 */ new Grammar(    50,      1),
    /*Rule   6 */ new Grammar(    48,      2),
    /*Rule   7 */ new Grammar(    48,      1),
    /*Rule   8 */ new Grammar(    49,      1),
    /*Rule   9 */ new Grammar(    49,      2),
    /*Rule  10 */ new Grammar(    49,      2),
    /*Rule  11 */ new Grammar(    70,      0),
    /*Rule  12 */ new Grammar(    49,      4),
    /*Rule  13 */ new Grammar(    49,      2),
    /*Rule  14 */ new Grammar(    49,      2),
    /*Rule  15 */ new Grammar(    49,      1),
    /*Rule  16 */ new Grammar(    49,      2),
    /*Rule  17 */ new Grammar(    49,      1),
    /*Rule  18 */ new Grammar(    49,      3),
    /*Rule  19 */ new Grammar(    49,      5),
    /*Rule  20 */ new Grammar(    57,      1),
    /*Rule  21 */ new Grammar(    57,      1),
    /*Rule  22 */ new Grammar(    57,      1),
    /*Rule  23 */ new Grammar(    57,      1),
    /*Rule  24 */ new Grammar(    57,      1),
    /*Rule  25 */ new Grammar(    55,      3),
    /*Rule  26 */ new Grammar(    55,      1),
    /*Rule  27 */ new Grammar(    56,      3),
    /*Rule  28 */ new Grammar(    56,      1),
    /*Rule  29 */ new Grammar(    54,      3),
    /*Rule  30 */ new Grammar(    46,      3),
    /*Rule  31 */ new Grammar(    46,      1),
    /*Rule  32 */ new Grammar(    47,      1),
    /*Rule  33 */ new Grammar(    58,      2),
    /*Rule  34 */ new Grammar(    61,      1),
    /*Rule  35 */ new Grammar(    61,      0),
    /*Rule  36 */ new Grammar(    62,      3),
    /*Rule  37 */ new Grammar(    62,      1),
    /*Rule  38 */ new Grammar(    39,      2),
    /*Rule  39 */ new Grammar(    41,      3),
    /*Rule  40 */ new Grammar(    38,      1),
    /*Rule  41 */ new Grammar(    38,      0),
    /*Rule  42 */ new Grammar(    40,      2),
    /*Rule  43 */ new Grammar(    40,      0),
    /*Rule  44 */ new Grammar(    53,      2),
    /*Rule  45 */ new Grammar(    53,      3),
    /*Rule  46 */ new Grammar(    53,      5),
    /*Rule  47 */ new Grammar(    53,      0),
    /*Rule  48 */ new Grammar(    59,      1),
    /*Rule  49 */ new Grammar(    59,      4),
    /*Rule  50 */ new Grammar(    60,      1),
    /*Rule  51 */ new Grammar(    60,      4),
    /*Rule  52 */ new Grammar(    63,      2),
    /*Rule  53 */ new Grammar(    64,      2),
    /*Rule  54 */ new Grammar(    71,      1),
    /*Rule  55 */ new Grammar(    45,      3),
    /*Rule  56 */ new Grammar(    45,      1),
    /*Rule  57 */ new Grammar(    43,      2),
    /*Rule  58 */ new Grammar(    43,      1),
    /*Rule  59 */ new Grammar(    44,      2),
    /*Rule  60 */ new Grammar(    44,      2),
    /*Rule  61 */ new Grammar(    44,      2),
    /*Rule  62 */ new Grammar(    44,      1),
    /*Rule  63 */ new Grammar(    42,      3),
    /*Rule  64 */ new Grammar(    42,      1),
    /*Rule  65 */ new Grammar(    42,      3),
    /*Rule  66 */ new Grammar(    42,      1),
    /*Rule  67 */ new Grammar(    69,      5),
    /*Rule  68 */ new Grammar(    69,      4),
    /*Rule  69 */ new Grammar(    65,      3),
    /*Rule  70 */ new Grammar(    65,      1),
    /*Rule  71 */ new Grammar(    52,      1),
    /*Rule  72 */ new Grammar(    52,      0),
    /*Rule  73 */ new Grammar(    66,      2),
    /*Rule  74 */ new Grammar(    66,      1),
    /*Rule  75 */ new Grammar(    51,      1),
    /*Rule  76 */ new Grammar(    51,      2),
    /*Rule  77 */ new Grammar(    51,      1)
  };

  private final int NON_TERMINALS=35;
  private final int nonTerminals[] = {
    38,// Number
    39,// Token
    40,// TokenName
    41,// TokenDef
    42,// BasicElement
    43,// Concatenation
    44,// UnaryRegex
    45,// RegExp
    46,// GroupTokens
    47,// GroupToken
    48,// Declarations
    49,// Declaration
    50,// Actions
    51,// Symbol
    52,// GrammarRule
    53,// LexicAction
    54,// Name
    55,// NonTerminals
    56,// TokenNames
    57,// Precedence
    58,// Definition
    59,// REquals
    60,// Equals
    61,// Type
    62,// Tokens
    63,// RStart
    64,// REnd
    65,// Rules
    66,// Rule
    67,// Descriptor
    68,// SectionMarker
    69,// Productions
    70,// $code-fragment-1
    71,// RExp
    
    72 // $start
  };

  /*
   *
   * Begin of Skeleton
   *
   */

  /* ****************************************************************
    Java Skeleton Parser for matrix tables

    This is not a sample program, but rather the parser skeleton
    to be included in the generated code.
    Modify at your own risk.

    Copyright (c), 1985-2016 Jaime Garza
  ***************************************************************** */

  private static final int ERROR_FAIL = 0;
  private static final int ERROR_RE_ATTEMPT = 1;

  private int    stateStack[] = new int[STACK_DEPTH];
  int            state;
  StackElement   lexicalValue;
  int            lexicalToken;
  int            errorCount;
  int            errorFlag;
  boolean        verbose = false;

  /**
   * Change the verbose flag
   * @param verbose if verbose is desired
   */
  public void setVerbose(boolean verbose) {
    this.verbose = verbose;
  }

  /**
   * @return the verbose flag
   * 
   */
  public boolean isVerbose() {
    return this.verbose;
  }
  
  /*
   * ==========================================================
   *                  Regular Expressions
   * ==========================================================
   */
  
  private int edgeIndex = 0;
  
  /**
   * checks one transition
   */
  private boolean matchesWholeTransition() {
    int transitionSize = edgeTable[edgeIndex ++];
    boolean negate = false;
    if (transitionSize < 0) {
      negate = true;
      transitionSize = -transitionSize;
    }

    boolean matchesTransition = false;
    if (transitionSize == 0) { // ANY match
      matchesTransition = currentChar != '\0';
    } else {
      // all elements of one transition
      for (int j = 0; j < transitionSize; j++) {
        int rangeStart = edgeTable[edgeIndex ++];
        int rangeEnd = edgeTable[edgeIndex ++];
        if (currentChar >= rangeStart && currentChar <= rangeEnd) {
          matchesTransition = true;
          // no break since the new vertex is at the end using edgeIndex
        }
      }
    }
    if (negate) {
      matchesTransition = !matchesTransition;
    }
    return currentChar == '\0' ? false : matchesTransition;
  }
  
  /**
   * tries to match a regular expression
   */
  private boolean matchesRegex(int vertex) {
    boolean accept = false;
    boolean goOn = true;
    
    recognized = "";
    
    do {
      accept = false;
      edgeIndex = vertexTable[vertex];
      if (edgeIndex < 0) {
        accept = true;
        edgeIndex = -edgeIndex;
      }
      int numTransitions = edgeTable[edgeIndex ++];
      boolean matchedOneTransition = false;
      for (int i = 0; i < numTransitions; i++) {
        // each transition
        int newVertex = edgeTable[edgeIndex ++];
        boolean matchesTransition = matchesWholeTransition();
        if (matchesTransition) {
          recognized += currentChar;
          currentChar = getNextChar(false);
          vertex = newVertex;
          matchedOneTransition = true;
          break; // found a matching transition. new vertex
        }
      }
      if (!matchedOneTransition) {
        if (accept) {
          return true;
        } else {
          // backtrack characters
          for (int i = recognized.length() -1; i >= 0; i--) {
            ungetChar(currentChar);
            currentChar = recognized.charAt(i);
          }
          goOn = false;
        }
      }
    } while (goOn);
    
    return false;
 }

  /**
   * This routine maps a state and a token to a new state on the action table
   * @param state is the current state
   * @param symbol is the given symbol to find (if not found, defa will be used
   * @return the next state
   */
  private int parserAction(int state, int symbol) {
    int index = getTokenIndex(symbol);
    return parsingTable[state][index];
  }

  /**
   * This routine maps a origin state to a destination state
   * using the symbol position
   * @param state is the current state
   * @param non terminal that causes the transition
   * @return the next state
   */
  private int parserGoto(int state, int symbol) {
    int index = symbol;
    return parsingTable[state][index];
  }

  /**
   * This routine prints the contents of the parsing stack
   */
  private void parserPrintStack() {
    int i;

    System.out.println("States: [");
    for(i=stackTop; i>=0; i--) {
      System.out.print("  " + stateStack[i]);
      if (i == stackTop) {
        System.out.println("<--Top Of Stack (" + stackTop + ")");
      }
      else {
        System.out.println();
      }
    }
    System.out.println("]");
    System.out.println("Values: [");
    for(i=stackTop;i >=0; i--) {
      System.out.print("  " + (stack[i] != null ? stack[i].toString() : "(null)"));
      if (i == stackTop) {
        System.out.println("<--Top Of Stack (" + stackTop + ")");
      }
      else {
        System.out.println();
      }
    }
    System.out.println("]");
  }

  /**
   * Does a shift operation.  Puts a new state on the top of the stack
   * @param sym is the symbol causing the shift
   * @param state is the current state
   * @return 1 if OK
   */
  private int parserShift(int sym, int state) {
    if(stackTop >= STACK_DEPTH-1) {
      return 0;
    }

    stateStack[++stackTop] = state;
    stack[stackTop] = lexicalValue;
    this.state = state;
    if (isVerbose()) {
      System.out.println("Shift to " + state + " with " + sym);
      parserPrintStack();
    }
    return 1;
  }

  /**
   * Recognizes a rule an removes all its elements from the stack
   * @param sym is the symbol causing the shift
   * @param rule is the number of rule being used
   * @return 1 if OK
   */
  private int parserReduce(int sym, int rule) {
    if (isVerbose()) {
      System.out.println("Reduce on rule " + rule + " with symbol " + sym);
    }
    if(generateCode(rule) == false) {
      return 0;
    }
    stackTop -= grammarTable[rule].reductions;
    stateStack[stackTop+1] =
        parserGoto(stateStack[stackTop], grammarTable[rule].symbol);
    state = stateStack[++stackTop];
    if (isVerbose()) {
        parserPrintStack();
    }
    return 1;
  }

  /**
   * @return the error message for a state
   */
  private String getErrorMessage() {
    int msgIndex = parsingError[state];
    if (msgIndex >= 0) {
      return errorTable[msgIndex];
    } else {
      return "Syntax error on state " + state + " with token " + getTokenName(lexicalToken);
    }
  }

  /**
   * Recover from a syntax error removing stack states/symbols, and removing
   * input tokens.  The array StxRecover contains the tokens that bound
   * the error
   * @return 1 if OK
   */
  private int parserRecover() {
    int i, acc;

    switch(errorFlag) {
      case 0: // 1st error
        if(parserError(state, lexicalToken, stackTop, getErrorMessage()) == 0) {
          return 0;
        }
        errorCount++;
        // continues and goes into 1 and 2.  No break on purpose

      case 1:
      case 2: // three attempts are made before dropping the current token
        errorFlag = 3; // Remove token

        while(stackTop > 0) {
          // Look if the state on the stack's top has a transition with one of
          // the recovering elements in StxRecoverTable
          for (i=0; i<RECOVERS; i++) {
            if((acc = parserAction(state, recoverTable[i])) > 0) {
              // valid shift
              return parserShift(recoverTable[i], acc);
            }
          }
          if (isVerbose()) {
            System.out.println("Recuperate removing state " + state + " and going to state " +
                            stack[stackTop-1]);
          }
          state = stateStack[--stackTop];
        }
        stackTop = 0;
        return 0;

      case 3: // I need to drop the current token
        if (isVerbose()) {
          System.out.println("Recuperate removing symbol " + lexicalToken);
        }
        if(lexicalToken == 0) { // end of file
          return 0;
        }
        lexicalToken = parserElement(false);
        return 1;
    }
    // should never reach
    System.err.println("ASSERTION FAILED ON PARSER");
    Exception e = new Exception();
    e.printStackTrace(System.err);
    return 0;
  }

  /**
   * Main parser routine, uses Shift, Reduce and Recover
   * @return 1 if OK
   */
  public int parse() {
    int action;

    stackTop = 0;
    stateStack[0] = 0;
    stack[0] = null;
    lexicalToken = parserElement(true);
    state = 0;
    errorFlag = 0;
    errorCount = 0;

    if (isVerbose()) {
      System.out.println("Starting to parse");
      parserPrintStack();
    }

    while(2 != 1) { // forever with break and return below
      action = parserAction(state, lexicalToken);
      if(action == ACCEPT) {
        if (isVerbose()) {
          System.out.println("Program Accepted");
        }
        return 1;
      }

      if(action > 0) {
        if(parserShift(lexicalToken, action) == 0) {
          return 0;
        }
        lexicalToken = parserElement(false);
        if(errorFlag > 0) {
           errorFlag--; // properly recovering from error
        }
      } else if(action < 0) {
        if(parserReduce(lexicalToken, -action) == 0) {
          if(errorFlag == -1) {
            if(parserRecover() == 0) {
              return 0;
            }
          } else {
            return 0;
          }
        }
      } else if(action == 0) {
        if(parserRecover() == 0) {
          return 0;
        }
      }
    }
  }

  /**
   * @return the current lexical value
   */
  public StackElement getResult() {
    return stack[stackTop];
  }

  /**
   * @param token is the number of the token
   * @return the name of a token, given the token number
   */
  public String getTokenName(int token) {
    for (int i = 0; i < tokenDefs.length; i++) {
      if (tokenDefs[i].token == token) {
        return tokenDefs[i].name;
        }
    }
    if (token < 256) {
      return "\'" + (char) token + "\'";
    } else {
      return "UNKNOWN TOKEN";
    }
  }

  /**
   * @param token is the number of the token
   * @return the name of a token, given the token number
   */
  public int getTokenIndex(int token) {
    for (int i = 0; i < tokenDefs.length; i++) {
      if (tokenDefs[i].token == token) {
        return i;
      }
    }
    return -1;
  }

  /**
   * Perform a round of tokenization and dump the results
   */
  public void dumpTokens() {
    lexicalToken = parserElement(true);
    lexicalValue = null;
    while (lexicalToken != 0) {
      System.out.println("Token: " + getTokenName(lexicalToken) + "(" + lexicalToken + "):" + (lexicalValue == null? "null": lexicalValue.toString()));
      lexicalValue = null;
      lexicalToken = parserElement(false);
    }
  }

  int findReservedWord(String word) {
    for (int i = 0; i < tokenDefs.length; i++) {
      if (tokenDefs[i].reserved && tokenDefs[i].name.equals(word)) {
        return tokenDefs[i].token;
      }
    }
    return -1;
  }

  int findReservedWordIgnoreCase(String word) {
    for (int i = 0; i < tokenDefs.length; i++) {
      if (tokenDefs[i].reserved && tokenDefs[i].name.equalsIgnoreCase(word)) {
        return tokenDefs[i].token;
      }
    }
    return -1;
  }

  /*
   *
   * End of packed skeleton for java
   *
   */




  /**
   * Get next token
   * 
   * @param init is set to true in the first call
   * @return the next token, changing mode as needed
   */
  protected int parserElement(boolean init) {
    int rc;
  
    if (init) {
      getNextChar(init);
    }
    switch (lexerMode) {
      case REGEX:
        rc = getRegexSymbol();
        if (environment.isDebug()) {
          System.out.printf("* RegexScanner: %s(%d) %s\n", getTokenName(rc), rc, rc == CHAR ? "\'" + lexicalValue.c + "\'":"");
        }
        break;
      
      case CHARACTER_CLASS:
        rc = getCharacterClassSymbol();
        lexicalValue = new StackElement(currentCharacterClass, runtimeData.currentStringValue);
        if (environment.isDebug()) {
          System.out.printf("* CharacterClassScanner %s(%d): %s\n", getTokenName(rc), rc, runtimeData.currentStringValue);
        }
        break;
        
      case NORMAL:
      default:
        rc = getNormalSymbol();
        lexicalValue = new StackElement(-1, tokenNumber, mustClose, runtimeData.currentStringValue, '\0');
        if (environment.isDebug()) {
          System.out.printf("* StdScanner: %s(%d) {%s}\n",
            getTokenName(rc), rc, lexicalValue != null ? lexicalValue.toString() : "");
        }
        break;
    }
    return rc;
  }

  /**
   * report an error
   * 
   * @param state the state of the error
   * @param sym causing token
   * @param stackTop the position in the stack when the error happened
   * @param errorMessage is the suggested message
   * @return always zero
   */
  protected int parserError(int state, int sym, int stackTop, String errorMessage) {
    if (errorMessage != null) {
      environment.error(-1, "Syntax error %d :\'%s\'.", state, errorMessage);
    } else {
      System.err.printf("%s(%05d) : Unknown error on state %d\n", environment.getSourceFile().toString(),
          runtimeData.lineNumber + 1, state);
    }
    isError = true;
    return 0; /*
               * with actions, it recovers weird. Need to change the action
               * stuff to the scanner
               */
  }
  
  /**
   * Get a symbol while in a character class
   * @return the next token
   */
  public int getCharacterClassSymbol() {

	while (2 > 1) { //forever
      if (currentChar == '\0') {
        isError = true;
        environment.error(-1, "Statement \"" + runtimeData.currentStringValue + "\" not properly ended with ']'.");
        return ERROR;
      }

      if (currentChar == ']') {
        // do not advance string
        lexerMode = LexerMode.REGEX;
        return CHARACTER_CLASS;
      }
      
      char c1 = currentChar;
      currentRegex += currentChar;
      getNextCharacter();
      if (currentChar == '-') {
        currentRegex += currentChar;
        getNextCharacter();
        if (currentChar == ']' || currentChar == '\0') { // last char, so use as a char
          // do not advance string
          currentCharacterClass.character(c1);
          currentCharacterClass.character('-');
          runtimeData.currentStringValue += "" + c1 + '-';
        } else {
          char c2 = currentChar;
          runtimeData.currentStringValue += "" + c1 + '-' + c2;
          currentCharacterClass.range(c1, c2);
          currentRegex += currentChar;
          getNextCharacter();
        }
      } else {
        // already advanced
        currentCharacterClass.character(c1);
        runtimeData.currentStringValue += "" + c1;
      }
    }
  }
   
  
  /**
   * Get the next character
   * @param init indicates if this is the first call
   * @return the next character in the stream
   */
  public char getNextChar(boolean init) {
    return getNextCharacter();
  }

  public int getRegexSymbol() {
    char c2;
    
    lexicalValue = new StackElement();
  
    if (isRegexSlash) {
      isRegexSlash = false;
      runtimeData.currentStringValue = "";
      return TOKEN;
    }
  
    if (currentChar == '|') {
      currentRegex += currentChar;
      getNextCharacter();
      return PIPE;
    }
    if (currentChar == '(') {
      currentRegex += currentChar;
      getNextCharacter();
      return '(';
    }
    if (currentChar == ')') {
      currentRegex += currentChar;
      getNextCharacter();
      return ')';
    }
    if (currentChar == '*') {
      currentRegex += currentChar;
      getNextCharacter();
      return STAR;
    }
    if (currentChar == '+') {
      currentRegex += currentChar;
      getNextCharacter();
      return PLUS;
    }
    if (currentChar == '?') {
      currentRegex += currentChar;
      getNextCharacter();
      return HUH;
    }
    if (currentChar == '.') {
      currentRegex += currentChar;
      getNextCharacter();
      return ANY;
    }
    if (currentChar == '/') {
      lexerMode = LexerMode.NORMAL;
      isRegexSlash = true;
      getNextCharacter();
      return '/';
    }
    if (currentChar == '[') {
      runtimeData.currentStringValue = "";
      currentCharacterClass = new CharacterClass();
      lexerMode = LexerMode.CHARACTER_CLASS;
      currentRegex += currentChar;
      getNextCharacter();
      if (currentChar == '^') {
        runtimeData.currentStringValue += '^';
        currentCharacterClass.negate();
        currentRegex += currentChar;
        getNextCharacter();
      }
      if (currentChar == '-') {
        runtimeData.currentStringValue += '-';
        currentCharacterClass.character('-');
        currentRegex += currentChar;
        getNextCharacter();
      } else if (currentChar == ']') {
        runtimeData.currentStringValue += ']';
        currentCharacterClass.character(']');
        currentRegex += currentChar;
        getNextCharacter();
        if (currentChar == '-') {
          runtimeData.currentStringValue += '-';
          currentCharacterClass.character('-');
          currentRegex += currentChar;
          getNextCharacter();
        }
      }
      return '[';
    }
    if (currentChar == ']') {
     currentRegex += currentChar;
      getNextCharacter();
      return ']';
    }
  
    if (currentChar == '\\') {
      currentRegex += currentChar;
      getNextCharacter();
      c2 = decodeEscape();
      if (c2 == 0) {
        return '\0';
      }
    }
    /*
     * StxValue.node = malloc(sizeof(REGEXNODE)); StxValue.node.parent = NULL;
     * StxValue.node.child1 = NULL; StxValue.node.child2 = NULL;
     * StxValue.node.nodeType = NODE_LEX; StxValue.node.ranges =
     * malloc(sizeof(REGEXRANGE)); StxValue.node.ranges.next = NULL;
     * StxValue.node.ranges.prev = NULL; StxValue.node.ranges.charStart = c2;
     * StxValue.node.ranges.charEnd = c2;
     */
    lexicalValue = new StackElement(-1, tokenNumber, false, "", currentChar);
    currentRegex += currentChar;
    getNextCharacter();
    return CHAR;
  }

  @Override
  public int getNormalSymbol() {
    char c2;
    String s2;
    boolean end;
  
    s2 = runtimeData.currentStringValue;
    runtimeData.currentStringValue = "";
  
    if (markers >= 2) {
      return 0;
    }
  
    if (isCurlyBrace) {
      isCurlyBrace = false;
      return ';';
    }
  
    if (isEqual) {
      isEqual = false;
      runtimeData.currentStringValue = "";
      return TOKEN;
    }
  
    if (isRegexSlash) {
      isRegexSlash = false;
      runtimeData.currentStringValue = "";
      return TOKEN;
    }
  
    while (2 > 1) {
      while (Character.isWhitespace(currentChar)) {
        getNextCharacter();
      }
      if (currentChar == '/') {
        if ((getNextCharacter()) == '*') {
          getNextCharacter();
          end = false;
          while (!end) {
            while (currentChar == '*') {
              if ((getNextCharacter()) == '/') {
                end = true;
              }
            }
            getNextCharacter();
          }
        } else {
          ungetChar(currentChar);
          currentChar = '/';
          break;
        }
      } else {
        break;
      }
    }
  
    if (currentChar == '\0') {
      return 0;
    }
  
    if (currentChar == '%' || currentChar == '\\') {
      getNextCharacter();
      switch (currentChar) {
        case '0':
          getNextCharacter();
          return TERM;
        case '<':
          getNextCharacter();
          return LEFT;
        case '2':
          getNextCharacter();
          return BINARY;
        case '>':
          getNextCharacter();
          return RIGHT;
        case '%':
        case '\\':
          getNextCharacter();
          markers++;
          return MARK;
        case '=':
          getNextCharacter();
          return PREC;
        case '@':
          getNextCharacter();
          return NAME;
        case '{':
          getNextCharacter();
          isCurlyBrace = true;
          return '{';
        case '!':
          getNextCharacter();
          return ERRDEF;
      }
      while (Character.isLetterOrDigit(currentChar)) {
        runtimeData.currentStringValue += currentChar;
        getNextCharacter();
      }
      for (ReservedWord rw : reservedWords) {
        if (runtimeData.currentStringValue.equals(rw.word)) {
          if (rw.token == UNION) {
            isCurlyBrace = true;
          }
          return rw.token;
        }
      }
      isError = true;
      environment.error(-1, "Reserved word \'%s\' is incorrect.", runtimeData.currentStringValue);
      return ERROR;
    }
  
    if (currentChar == ';') {
      getNextCharacter();
      return ';';
    }
  
    if (currentChar == ',') {
      getNextCharacter();
      return ',';
    }
  
    if (currentChar == ':') {
      currentNonTerminalName = s2;
      getNextCharacter();
      return ':';
    }
  
    if (currentChar == '|') {
      getNextCharacter();
      return '|';
    }
  
    if (currentChar == '[') {
      getNextCharacter();
      return '[';
    }
  
    if (currentChar == ']') {
      getNextCharacter();
      return ']';
    }
  
    if (currentChar == '=') {
      getNextCharacter();
      isEqual = true;
      return '=';
    }
  
    if (currentChar == '{') {
      isEqual = true;
      return '=';
    }
  
    if (currentChar == '<') {
      getNextCharacter();
      runtimeData.currentStringValue = "";
      while (currentChar != '\0' && currentChar != '>' && currentChar != '\n') {
        runtimeData.currentStringValue += currentChar;
        getNextCharacter();
      }
      if (currentChar != '>') {
        isError = true;
        environment.error(-1, "Statement < .. > not ended.");
        return ERROR;
      }
      getNextCharacter();
      return TYPENAME;
    }
  
    if (currentChar == '/') {
      lexerMode = LexerMode.REGEX;
      isRegexSlash = true;
      currentRegex = "";
      getNextCharacter();
      return '/';
    }
  
    if (Character.isDigit(currentChar)) {
      runtimeData.currentStringValue = "";
      while (Character.isDigit(currentChar)) {
        runtimeData.currentStringValue += currentChar;
        getNextCharacter();
      }
      tokenNumber = Integer.parseInt(runtimeData.currentStringValue);
      return NUM;
    }
  
    mustClose = false;
    if (currentChar == '\'' || currentChar == '"') {
      c2 = currentChar;
      mustClose = true;
      getNextCharacter();
    } else {
      c2 = ':';
    }
  
    runtimeData.currentStringValue = "";
    do { /* TOKEN */
      runtimeData.currentStringValue += currentChar;
      getNextCharacter();
      if (currentChar == '\0') {
        break;
      }
      if (!mustClose && "%\\;,:[]|={</ \r\t\n".indexOf(currentChar) >= 0) {
        break;
      }
    } while (currentChar != c2);
  
    if (mustClose && currentChar != c2) {
      isError = true;
      environment.error(-1, "Statement ' .. ' or \" .. \" not ended.");
      return ERROR;
    }
    tokenNumber = -1;
    if (runtimeData.currentStringValue.equals("\\a")) {
      tokenNumber = 7;
    } else if (runtimeData.currentStringValue.equals("\\b")) {
      tokenNumber = '\b';
    } else if (runtimeData.currentStringValue.equals("\\n")) {
      tokenNumber = '\n';
    } else if (runtimeData.currentStringValue.equals("\\t")) {
      tokenNumber = '\t';
    } else if (runtimeData.currentStringValue.equals("\\f")) {
      tokenNumber = '\f';
    } else if (runtimeData.currentStringValue.equals("\\r")) {
      tokenNumber = '\r';
    } else if (runtimeData.currentStringValue.length() >= 2 && runtimeData.currentStringValue.substring(0, 2).equals("\\x")) {
      int p = 2;
      tokenNumber = 0;
      while (2 > 1) {
        if (runtimeData.currentStringValue.charAt(p) >= '0' && runtimeData.currentStringValue.charAt(p) <= '9') {
          tokenNumber = tokenNumber * 16 + runtimeData.currentStringValue.charAt(p++) - '0';
        } else if (runtimeData.currentStringValue.charAt(p) >= 'A' && runtimeData.currentStringValue.charAt(p) <= 'F') {
          tokenNumber = tokenNumber * 16 + runtimeData.currentStringValue.charAt(p++) - 'A' + 10;
        } else if (runtimeData.currentStringValue.charAt(p) >= 'a' && runtimeData.currentStringValue.charAt(p) <= 'f') {
          tokenNumber = tokenNumber * 16 + runtimeData.currentStringValue.charAt(p++) - 'a' + 10;
        } else {
          break;
        }
      }
    } else if (runtimeData.currentStringValue.length() >= 2 && runtimeData.currentStringValue.substring(0, 2).equals("\\0")) {
      int p = 2;
      tokenNumber = 0;
      while (runtimeData.currentStringValue.charAt(p) >= '0' && runtimeData.currentStringValue.charAt(p) <= '7') {
        tokenNumber = tokenNumber * 8 + runtimeData.currentStringValue.charAt(p++) - '0';
      }
    }
  
    if (mustClose) {
      getNextCharacter();
      if (runtimeData.currentStringValue.length() == 1) {
        tokenNumber = runtimeData.currentStringValue.charAt(0);
      }
    }
  
    return TOKEN;
  }

  /**
   * Execute this phase
   * @throws ParsingException on error.  Check cause and message.
   */
  public void execute() throws ParsingException {
    if (environment.isVerbose()) {
      System.out.println("Parse");
    }
    setVerbose(environment.isDebug());
    runtimeData.lineNumber = 1;
    DfaNode.reset();
    NfaNode.reset();
    markers = 0;
    Terminal terminal = new Terminal("$");
    runtimeData.getTerminals().add(terminal);
    terminal.setCount(0);
    terminal.setToken(0);
    if (parse() == 0 || isError) {
      throw new ParsingException("Parser returned errors.  Please see messages from parser");
    }
    reviewDeclarations();
    computeRootSymbol();
    generateTopRecoveryTable();
    finalizeSymbols();
    finalizeRules();
    generateTokenDefinitions();
    runtimeData.setNumberOfErrors(errorCount);
    runtimeData.setFinalActions(finalActions);
  }
  
  static class ReservedWord {
    String word;
    int token;

    ReservedWord(String word, int token) {
      super();
      this.word = word;
      this.token = token;
    }
  }
  
  static final ReservedWord reservedWords[] = { 
    new ReservedWord("token", TERM),
    new ReservedWord("term", TERM), 
    new ReservedWord("left", LEFT), 
    new ReservedWord("nonassoc", BINARY),
    new ReservedWord("binary", BINARY),
    new ReservedWord("right", RIGHT), 
    new ReservedWord("prec", PREC),
    new ReservedWord("start", START), 
    new ReservedWord("type", TYPE), 
    new ReservedWord("symbol", TYPE), 
    new ReservedWord("declare", DECLARE), 
    new ReservedWord("union", UNION),
    new ReservedWord("stack", UNION), 
    new ReservedWord("class", UNION),
    new ReservedWord("struct", UNION),
    new ReservedWord("name", NAME), 
    new ReservedWord("error", ERRDEF), 
    new ReservedWord("lexer", LEXER),
    new ReservedWord("group", GROUP)
  };
  
}