/*
 Syntax is distributed under the Revised, or 3-clause BSD license
 ===============================================================================
 Copyright (c) 1985, 2012, 2016, Jaime Garza
 All rights reserved.
 
 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions are met:
     * Redistributions of source code must retain the above copyright
       notice, this list of conditions and the following disclaimer.
     * Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.
     * Neither the name of the copyright holder nor the
       names of its contributors may be used to endorse or promote products
       derived from this software without specific prior written permission.
 
 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ===============================================================================
*/
package me.jaimegarza.syntax.graph;

/**
 * Simple representation of a point in space
 * @author jgarza
 *
 */
public class Point {
  private double x;
  private double y;
  
  public Point(double x, double y) {
    this.x = x;
    this.y = y;
  }

  /**
   * Distance between two points
   * @param p2 the other point
   * @return the distance
   */
  public double distance(Point p2) {
    return Math.sqrt((x - p2.x)*(x - p2.x) + (y - p2.y)*(y - p2.y));
  }
  
  /**
   * Gets a new point that is the diference between two points
   * @param p2 the other point
   * @return (x - p2.x, y-p2.y)
   */
  public Point subtract(Point p2) {
    return new Point(x - p2.x, y - p2.y);
  }
  
  /**
   * Add two points
   * @param p2 is the other point
   * @return (x+p2x, y+p2y)
   */
  Point add(Point p2) {
    return new Point(x + p2.x, y + p2.y);
  }

  /**
   * Scale the point
   * @param s is the scale
   * @return (x*s, y*s)
   */
  public Point scale(double s) {
    return new Point(x*s, y*s);
  }

  /**
   * @return the x
   */
  public double getX() {
    return x;
  }

  /**
   * @return the y
   */
  public double getY() {
    return y;
  }
  
  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }

    try {
      Point p = (Point) obj;
      return x == p.x && y == p.y;
    } catch (NullPointerException unused) {
      return false;
    } catch (ClassCastException unused) {
      return false;
    }
  }

  @Override
  public int hashCode() {
    return 31 + Double.hashCode(x) + Double.hashCode(y);
  }

  @Override
  public String toString() {
    return "(" + x + "," + y + ")";
  }

}

