package com.zing.zalo.zalosdk.service.client;

import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.Bundle;
import android.os.Handler;
import android.os.IBinder;
import android.os.Looper;
import android.os.Message;
import android.os.Messenger;
import android.os.RemoteException;

import com.zing.zalo.zalosdk.core.helper.AppInfo;
import com.zing.zalo.zalosdk.core.log.Log;

public class PlatformServiceClient implements ServiceConnection {
    private final Context context;
    private final Handler handler;
    private CompletedListener listener;
    private int requestMessage;
    private boolean running;
    private Messenger sender;

    public interface CompletedListener {
        void completed(Bundle bundle);
    }

    PlatformServiceClient(Context context, int requestMessage) {
        Context applicationContext = context.getApplicationContext();
        if (applicationContext == null) {
            applicationContext = context;
        }
        this.context = applicationContext;
        this.requestMessage = requestMessage;
        this.handler = new Handler(Looper.getMainLooper()) {
            public void handleMessage(Message message) {
                PlatformServiceClient.this.handleMessage(message);
            }
        };
    }

    void setCompletedListener(CompletedListener listener) {
        this.listener = listener;
    }

    protected Context getContext() {
        return this.context;
    }

    public boolean start() {
        if (this.running) {
            return false;
        }
        Intent intent = NativeProtocol.createPlatformServiceIntent(this.context);
        if (intent == null) {
            return false;
        }
        this.running = true;
        this.context.bindService(intent, this, Context.BIND_AUTO_CREATE);
        return true;
    }

    public void cancel() {
        this.running = false;
    }

    public void onServiceConnected(ComponentName name, IBinder service) {
        this.sender = new Messenger(service);
        sendMessage();
    }

    public void onServiceDisconnected(ComponentName name) {
        this.sender = null;
        try {
            this.context.unbindService(this);
        } catch (IllegalArgumentException e) {
        }
        callback(null);
    }

    private void sendMessage() {
        Bundle data = new Bundle();
        data.putString(NativeProtocol.KEY_REQUEST_APPLICATION_ID, AppInfo.getAppId(getContext()));
        populateRequestBundle(data);
        Message request = Message.obtain(null, this.requestMessage);
        request.arg1 = 1;
        request.arg2 = 1;
        request.setData(data);
        request.replyTo = new Messenger(this.handler);
        try {
            this.sender.send(request);
        } catch (RemoteException e) {
            callback(null);
        }
    }

    private void populateRequestBundle(Bundle data) {
    }

    private void handleMessage(Message message) {
        if (message.what == this.requestMessage && message.arg2 == 2) {
            Bundle extras = message.getData();
            if (extras.getInt(NativeProtocol.KEY_RESULT_ERROR_CODE) != 0) {
                callback(null);
            } else {
                callback(extras);
            }
            try {
                this.context.unbindService(this);
            } catch (IllegalArgumentException e) {
                Log.v(e.toString());
            }
        }
    }

    private void callback(Bundle result) {
        if (this.running) {
            this.running = false;
            CompletedListener callback = this.listener;
            if (callback != null) {
                callback.completed(result);
            }
        }
    }
}

