package money.link.linkaccount

import android.net.Uri
import android.view.View
import android.webkit.WebResourceRequest
import android.webkit.WebResourceResponse
import android.webkit.WebView
import android.webkit.WebViewClient
import android.widget.ProgressBar

internal class ClientWebView(
    private val progressBar: ProgressBar,
    private val redirectUrl: String,
    private val complete: LinkAccountComplete,
    private val toggleBackButton: (Boolean) -> Unit
) : WebViewClient() {

    private val clientUrl = "https://client.${BuildConfig.ClientHost}"

    override fun onPageFinished(view: WebView?, url: String?) {
        progressBar.visibility = View.GONE
        super.onPageFinished(view, url)
    }

    override fun shouldInterceptRequest(
        view: WebView?,
        request: WebResourceRequest?
    ): WebResourceResponse? {
        val (status, result) = parseRequest(request)

        return when (status) {
            ParseRequestType.LINKING_COMPLETE -> result?.let {
                complete(it)
                WebResourceResponse("text/javascript", "utf-8", null)
            }
            ParseRequestType.SHOW_BACK_BUTTON -> {
                toggleBackButton(true)
                super.shouldInterceptRequest(view, request)
            }
            ParseRequestType.HIDE_BACK_BUTTON -> {
                toggleBackButton(false)
                super.shouldInterceptRequest(view, request)
            }

            ParseRequestType.PASS_THRU -> super.shouldInterceptRequest(view, request)
        }
    }

    private fun parseRequest(request: WebResourceRequest?): Pair<ParseRequestType, Result<LinkCustomerId>?> {
        return request?.url?.let { url ->
            if (url.toString().startsWith(redirectUrl)) {
                parseCustomerId(url)?.let { customerId ->
                    Pair(ParseRequestType.LINKING_COMPLETE, Result.success(customerId))
                } ?: Pair(ParseRequestType.LINKING_COMPLETE, Result.failure(parseError(url)))
            } else if (parsePolicy(url)) {
                Pair(ParseRequestType.SHOW_BACK_BUTTON, null)
            } else if (parseClientFlow(url)) {
                Pair(ParseRequestType.HIDE_BACK_BUTTON, null)
            } else {
                Pair(ParseRequestType.PASS_THRU, null)
            }
        } ?: Pair(ParseRequestType.PASS_THRU, null)
    }

    private fun parseCustomerId(url: Uri): LinkCustomerId? {
        return url.getQueryParameter("customerId")
    }

    private fun parseError(redirectUrl: Uri): LinkError {
        val statusCode = redirectUrl.getQueryParameter("status")?.toInt()
        return LinkError(statusCode = statusCode)
    }

    private fun parsePolicy(redirectUrl: Uri): Boolean {
        return PolicyUrl.firstOrNull { redirectUrl.toString().startsWith(it) }?.let {
            true
        } ?: false
    }

    private fun parseClientFlow(url: Uri): Boolean {
        return url.toString().startsWith(clientUrl)
    }
}
