package money.link.linkaccount

import android.annotation.SuppressLint
import android.os.Bundle
import android.view.LayoutInflater
import android.view.MenuItem
import android.view.View
import android.view.ViewGroup
import android.webkit.WebView
import android.widget.ProgressBar
import androidx.appcompat.view.menu.MenuView
import androidx.appcompat.widget.Toolbar
import androidx.fragment.app.DialogFragment
import androidx.fragment.app.FragmentManager

typealias LinkCustomerId = String
typealias LinkAccountComplete = (Result<LinkCustomerId>) -> Unit

class LinkError(
    override val message: String = "There was an error linking the customer account",
    val statusCode: Int?
) : Error()

internal enum class ParseRequestType {
    LINKING_COMPLETE,
    SHOW_BACK_BUTTON,
    HIDE_BACK_BUTTON,
    PASS_THRU
}

internal val PolicyUrl = arrayListOf(
    "https://${BuildConfig.ClientHost}/legal/terms-and-conditions",
    "https://${BuildConfig.ClientHost}/legal/privacy-policy",
    "https://connect2.finicity.com/assets/html/connect-eula.html",
    "https://www.finicity.com/privacy",
    "https://www.dwolla.com/legal/tos"
)

class LinkAccountDialog(sessionKey: String) : DialogFragment() {

    private var toolbar: Toolbar? = null
    private var backButton: MenuView.ItemView? = null
    private var webView: WebView? = null
    private var progressBar: ProgressBar? = null
    private var complete: LinkAccountComplete? = null

    private val userAgentVersion = "link-native-sdk/${BuildConfig.VERSION_NAME} (Android)"
    private val redirectUrl = "https://link.money/native-sdk/complete"
    private val linkPayUrl = "https://client.${BuildConfig.ClientHost}?sessionKey=$sessionKey&redirect=$redirectUrl"

    fun display(fragmentManager: FragmentManager, complete: LinkAccountComplete) {
        this.complete = complete
        show(fragmentManager, "fullscreen_dialog")
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setStyle(STYLE_NORMAL, R.style.AppTheme_FullScreenDialog)
    }

    override fun onStart() {
        super.onStart()

        dialog?.let {
            val width = ViewGroup.LayoutParams.MATCH_PARENT
            val height = ViewGroup.LayoutParams.MATCH_PARENT
            it.window?.let { window ->
                window.setLayout(width, height)
                window.setWindowAnimations(R.style.Slide)
            }
        }
    }

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        super.onCreateView(inflater, container, savedInstanceState)
        val view: View = inflater.inflate(R.layout.link_account_dialog, container, false)
        toolbar = view.findViewById(R.id.toolbar)
        toolbar?.setLogo(R.drawable.ic_link_logo_white)

        webView = view.findViewById(R.id.webView)
        progressBar = view.findViewById(R.id.progressBar)

        return view
    }

    @SuppressLint("SetJavaScriptEnabled", "RestrictedApi")
    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        toolbar?.let {
            it.setNavigationOnClickListener { _: View? -> dismiss() }
            it.title = " Pay by Bank"
            it.inflateMenu(R.menu.link_account_dialog)
            backButton = it.findViewById(R.id.back_button)

            it.setOnMenuItemClickListener { _: MenuItem? ->
                webView?.goBack()
                true
            }
        }

        progressBar?.let { progressBar ->
            webView?.let {
                it.settings.javaScriptEnabled = true
                it.webViewClient = ClientWebView(progressBar, redirectUrl, complete!!) { enabled ->
                    activity?.runOnUiThread {
                        if (enabled) {
                            backButton?.setTitle("Back")
                        } else {
                            backButton?.setTitle("")
                        }
                    }
                }

                it.settings.userAgentString = "${it.settings.userAgentString} $userAgentVersion"
                it.loadUrl(linkPayUrl)
            }
        }
    }
}
