/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.ossindex.service.client.cache;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Charsets;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.hash.Hashing;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.io.Reader;
import java.io.Writer;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.util.Map;
import javax.annotation.Nullable;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonatype.goodies.packageurl.PackageUrl;
import org.sonatype.ossindex.service.api.componentreport.ComponentReport;
import org.sonatype.ossindex.service.client.cache.Cache;
import org.sonatype.ossindex.service.client.cache.CacheConfiguration;
import org.sonatype.ossindex.service.client.marshal.GsonMarshaller;
import org.sonatype.ossindex.service.client.marshal.Marshaller;
import org.sonatype.ossindex.service.client.util.FileLocker;
import org.sonatype.ossindex.service.client.util.UserDataLocation;

public class DirectoryCache
implements Cache {
    private static final Logger log = LoggerFactory.getLogger(DirectoryCache.class);
    private final Marshaller marshaller;
    private final Path baseDir;
    private final Duration expireAfter;
    private volatile boolean closed;

    public DirectoryCache(Marshaller marshaller, Configuration config) throws IOException {
        Preconditions.checkNotNull((Object)config);
        this.marshaller = (Marshaller)Preconditions.checkNotNull((Object)marshaller);
        this.baseDir = (Path)Preconditions.checkNotNull((Object)config.getBaseDir(), (Object)"Missing required base-directory");
        this.expireAfter = (Duration)Preconditions.checkNotNull((Object)config.getExpireAfter(), (Object)"Missing required expiration duration");
        if (!Files.exists(this.baseDir, new LinkOption[0])) {
            Files.createDirectories(this.baseDir, new FileAttribute[0]);
        } else {
            Preconditions.checkState((boolean)Files.isDirectory(this.baseDir, new LinkOption[0]), (String)"Not a directory: %s", (Object)this.baseDir);
        }
        log.debug("Marshaller: {}", (Object)marshaller);
        log.debug("Base-directory: {}", (Object)this.baseDir);
        log.debug("Expire after: {}", (Object)this.expireAfter);
    }

    private void ensureNotClosed() {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0, (Object)"Closed");
    }

    @Override
    public synchronized void close() throws Exception {
        this.closed = true;
    }

    public String toString() {
        return MoreObjects.toStringHelper((Object)this).add("baseDir", (Object)this.baseDir).add("expireAfter", (Object)this.expireAfter).toString();
    }

    @Override
    @Nullable
    public synchronized ComponentReport getIfPresent(PackageUrl coordinates) {
        Preconditions.checkNotNull((Object)coordinates);
        this.ensureNotClosed();
        Path file = this.entryFile(coordinates);
        if (Files.exists(file, new LinkOption[0])) {
            try {
                return this.loadEntry(file);
            }
            catch (IOException e) {
                log.warn("Failed to load entry: {}", (Object)file, (Object)e);
            }
        }
        return null;
    }

    @Override
    public synchronized void putAll(Map<PackageUrl, ComponentReport> reports) {
        Preconditions.checkNotNull(reports);
        this.ensureNotClosed();
        for (Map.Entry<PackageUrl, ComponentReport> entry : reports.entrySet()) {
            Path file = this.entryFile(entry.getKey());
            try {
                this.storeEntry(entry.getValue(), file);
            }
            catch (IOException e) {
                log.warn("Failed to store entry: {}", (Object)file, (Object)e);
            }
        }
    }

    private String entryKey(PackageUrl coordinates) {
        return Hashing.sha1().hashUnencodedChars((CharSequence)coordinates.toString()).toString();
    }

    private Path entryFile(PackageUrl coordinates) {
        String key = this.entryKey(coordinates);
        Path path = Paths.get(key.substring(0, 2), key.substring(2, 4), key);
        return this.baseDir.resolve(path);
    }

    @Nullable
    private ComponentReport loadEntry(final Path file) throws IOException {
        log.trace("Loading entry: {}", (Object)file);
        return FileLocker.readLock(file, new FileLocker.FileFunction<ComponentReport>(){

            @Override
            public ComponentReport apply(RandomAccessFile raf) throws IOException {
                ComponentReport report = null;
                boolean delete = false;
                if (DirectoryCache.this.isEntryStale(file)) {
                    log.trace("Expiring entry: {}", (Object)file);
                    delete = true;
                } else {
                    try {
                        BufferedReader reader = new BufferedReader(Channels.newReader((ReadableByteChannel)raf.getChannel(), Charsets.UTF_8.name()));
                        report = DirectoryCache.this.marshaller.unmarshal((Reader)reader, ComponentReport.class);
                    }
                    catch (IOException e) {
                        log.warn("Corrupt entry: {}", (Object)file, (Object)e);
                        delete = true;
                    }
                }
                if (delete) {
                    raf.close();
                    Files.deleteIfExists(file);
                }
                return report;
            }
        });
    }

    private boolean isEntryStale(Path file) throws IOException {
        BasicFileAttributes attributes = Files.readAttributes(file, BasicFileAttributes.class, new LinkOption[0]);
        FileTime lastModified = attributes.lastModifiedTime();
        log.trace("Last-modified: {}", (Object)lastModified);
        long age = System.currentTimeMillis() - lastModified.toMillis();
        log.trace("Age: {} ms", (Object)age);
        return age > this.expireAfter.getMillis();
    }

    private void storeEntry(final ComponentReport report, final Path file) throws IOException {
        log.trace("Storing entry: {} -> {}", (Object)report, (Object)file);
        Files.createDirectories(file.getParent(), new FileAttribute[0]);
        try {
            Files.createFile(file, new FileAttribute[0]);
        }
        catch (FileAlreadyExistsException e) {
            log.trace("File already exists: {}", (Object)file, (Object)e);
        }
        FileLocker.writeLock(file, new FileLocker.FileFunction<Void>(){

            @Override
            public Void apply(RandomAccessFile raf) throws IOException {
                try {
                    BufferedWriter writer = new BufferedWriter(Channels.newWriter((WritableByteChannel)raf.getChannel(), Charsets.UTF_8.name()));
                    DirectoryCache.this.marshaller.marshal(report, writer);
                    ((Writer)writer).flush();
                }
                catch (IOException e) {
                    log.warn("Failed to store entry: {}", (Object)file, (Object)e);
                    raf.close();
                    Files.deleteIfExists(file);
                }
                return null;
            }
        });
    }

    @VisibleForTesting
    void purgeEntry(PackageUrl coordinates) throws IOException {
        Path file = this.entryFile(coordinates);
        log.trace("Purge entry: {}", (Object)file);
        Files.deleteIfExists(file);
    }

    public static class Configuration
    implements CacheConfiguration {
        public static final Path DEFAULT_BASE_DIR = new UserDataLocation("Sonatype", "Ossindex").get().resolve("report-cache");
        public static final Duration DEFAULT_EXPIRE_AFTER = Duration.standardHours((long)12L);
        @JsonProperty
        private Path baseDir = DEFAULT_BASE_DIR;
        @JsonProperty
        private Duration expireAfter = DEFAULT_EXPIRE_AFTER;

        public Path getBaseDir() {
            return this.baseDir;
        }

        public void setBaseDir(Path baseDir) {
            this.baseDir = baseDir;
        }

        public Duration getExpireAfter() {
            return this.expireAfter;
        }

        public void setExpireAfter(Duration expireAfter) {
            this.expireAfter = expireAfter;
        }

        @Override
        public Cache create() throws Exception {
            return new DirectoryCache(new GsonMarshaller(), this);
        }
    }
}

