package net.aihelp.core.db;

import android.content.Context;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;

import net.aihelp.utils.ListUtil;

import java.util.List;

public class BaseSQLiteHelper extends LocaleIgnoredOpenHelper {

    private IDatabaseContract contract;
    private BaseSQLiteHelper.IDbMigrationListener listener;

    public BaseSQLiteHelper(Context context, IDatabaseContract contract) {
        super(context, contract.getDatabaseName(), null, contract.getDatabaseVersion());
        this.contract = contract;
    }

    public void setListener(BaseSQLiteHelper.IDbMigrationListener listener) {
        this.listener = listener;
    }

    public void onCreate(SQLiteDatabase db) {
        try {
            db.beginTransaction();
            this.createAllTables(db);
            db.setTransactionSuccessful();
        } finally {
            try {
                if (db.inTransaction()) {
                    db.endTransaction();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        List<IMigrateContract> migrators = this.contract.getMigratorsForUpgrade(oldVersion);
        if (!ListUtil.isListEmpty(migrators)) {
            boolean isMigrated = this.migrate(db, migrators, oldVersion);
            if (this.listener != null) {
                if (isMigrated) {
                    this.listener.onDbMigrationSuccess(BaseSQLiteHelper.MigrationType.UPGRADE, this.contract.getDatabaseName());
                } else {
                    this.listener.onDbMigrationFailed(BaseSQLiteHelper.MigrationType.UPGRADE, this.contract.getDatabaseName());
                }
            }
        }
    }

    public void onDowngrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        boolean allTablesRecreated = this.dropAndCreateAllTables(db, true);
        if (this.listener != null) {
            if (allTablesRecreated) {
                this.listener.onDbMigrationSuccess(BaseSQLiteHelper.MigrationType.DOWNGRADE, this.contract.getDatabaseName());
            } else {
                this.listener.onDbMigrationFailed(BaseSQLiteHelper.MigrationType.DOWNGRADE, this.contract.getDatabaseName());
            }
        }

    }

    private boolean dropAndCreateAllTables(SQLiteDatabase db, boolean throwException) {
        if (db.isOpen()) {
            boolean var4;
            try {
                db.beginTransaction();
                this.dropAllTables(db);
                this.createAllTables(db);
                db.setTransactionSuccessful();
                return true;
            } catch (Exception var14) {
                if (throwException) {
                    throw var14;
                }
                var4 = false;
            } finally {
                try {
                    if (db.inTransaction()) {
                        db.endTransaction();
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }

            }

            return var4;
        } else {
            return true;
        }
    }

    public boolean dropAndCreateAllTables(SQLiteDatabase db) {
        return this.dropAndCreateAllTables(db, false);
    }

    private void dropAllTables(SQLiteDatabase db) {
        for (String tableName : this.contract.getTableNames()) {
            db.execSQL("DROP TABLE IF EXISTS " + tableName);
        }
    }

    private void createAllTables(SQLiteDatabase db) {
        for (String query : this.contract.getCreateTableQueries()) {
            db.execSQL(query);
        }
    }

    private boolean migrate(SQLiteDatabase db, List<IMigrateContract> migrators, int oldVersion) {
        boolean isMigrationSuccess = true;

        try {
            db.beginTransaction();

            for (IMigrateContract migrator : migrators) {
                migrator.migrate(db);
            }

            db.setTransactionSuccessful();
        } catch (Exception var15) {
            isMigrationSuccess = false;
        } finally {
            try {
                if (db.inTransaction()) {
                    db.endTransaction();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }

        }

        if (!isMigrationSuccess) {
            this.dropAndCreateAllTables(db, true);
        }

        return isMigrationSuccess;
    }

    public interface IDbMigrationListener {
        void onDbMigrationSuccess(BaseSQLiteHelper.MigrationType var1, String var2);

        void onDbMigrationFailed(BaseSQLiteHelper.MigrationType var1, String var2);
    }

    public enum MigrationType {
        UPGRADE,
        DOWNGRADE
    }

    public synchronized void clearDatabase() {
        try {
            SQLiteDatabase database = getWritableDatabase();
            dropAndCreateAllTables(database);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

}
