package net.aihelp.core.net.http.interceptor;

import android.text.TextUtils;

import net.aihelp.BuildConfig;
import net.aihelp.common.Const;
import net.aihelp.core.net.http.config.HttpConfig;
import net.aihelp.utils.TLog;

import org.json.JSONArray;
import org.json.JSONObject;

import java.io.IOException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;
import java.util.UUID;
import java.util.regex.Pattern;

import androidx.annotation.NonNull;
import okhttp3.FormBody;
import okhttp3.Interceptor;
import okhttp3.MediaType;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okio.Buffer;

public class SignInterceptor implements Interceptor {

    @Override
    public Response intercept(@NonNull Chain chain) throws IOException {
        Request request = chain.request();

        Request.Builder newBuilder = request.newBuilder();
        long timestamp = System.currentTimeMillis();
        String nonce = UUID.randomUUID().toString();

        try {
            if (request.method().equals("GET")) {
                String url = request.url().toString();
                if (!Pattern.matches("^/.+\\.(ini|json|aiml)$", url)) {
                    String sign = getRequestSign(getParamDictionary(url, timestamp, nonce));
                    request = addHeaders(newBuilder, timestamp, nonce, sign).url(url).build();
                }
            } else if (request.method().equals("POST")) {
                RequestBody requestBody = request.body();
                if (requestBody instanceof FormBody) {
                    FormBody formBody = (FormBody) requestBody;
                    String sign = getRequestSign(getParamDictionary(formBody, timestamp, nonce));
                    request = addHeaders(newBuilder, timestamp, nonce, sign).post(requestBody).build();
                    return chain.proceed(request);
                }
                MediaType mediaType = null;
                if (requestBody != null) mediaType = request.body().contentType();
                if (mediaType != null && "json".equals(mediaType.subtype())) {
                    String postJson = bodyToString(request.body());
                    if (!TextUtils.isEmpty(postJson)) {
                        try {
                            new JSONArray(postJson);
                        } catch (Exception e) {
                            JSONObject jsonObject = new JSONObject(postJson);
                            String sign = getRequestSign(getParamDictionary(jsonObject, timestamp, nonce));
                            request = addHeaders(newBuilder, timestamp, nonce, sign).post(requestBody).build();
                        }
                    }
                } else {
                    JSONObject jsonObject = new JSONObject();
                    jsonObject.put("appId", Const.APP_ID);  //添加公共参数
                    jsonObject.put("appkey", Const.APP_KEY);
                    jsonObject.put("lan", Const.CORRECT_LANGUAGE);
                    jsonObject.put("l", Const.CORRECT_LANGUAGE);
                    jsonObject.put("platform", BuildConfig.SDK_PLATFORM);
                    jsonObject.put("sdkVersion", BuildConfig.SDK_VERSION);
                    String sign = getRequestSign(getParamDictionary(jsonObject, timestamp, nonce));
                    request = addHeaders(newBuilder, timestamp, nonce, sign).post(requestBody).build();
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        return chain.proceed(request);

    }

    private String getPostRawString(Object obj) {
        StringBuilder result = new StringBuilder();
        if (obj instanceof FormBody) {
            FormBody formBody = (FormBody) obj;
            for (int i = 0; i < formBody.size(); i++) {
                result.append(formBody.encodedName(i)).append("=").append(formBody.encodedValue(i)).append("&");
            }
        }
        if (obj instanceof JSONObject) {
            JSONObject jsonObject = (JSONObject) obj;
            Iterator<String> keys = jsonObject.keys();
            while (keys.hasNext()) {
                String key = keys.next();
                result.append(key).append("=").append(jsonObject.opt(key)).append("&");
            }
        }
        return result.toString();
    }

    private Map<String, Object> getParamDictionary(Object obj, long timestamp, String nonce) {
        TreeMap<String, Object> sortedMap = new TreeMap<>();
        try {
            sortedMap.put("timestamp", String.valueOf(timestamp));
            sortedMap.put("nonce", nonce);
            if (obj instanceof String) {
                String requestUrl = (String) obj;
                String[] params = requestUrl.substring(requestUrl.indexOf("?") + 1).split("&");
                for (String param : params) {
                    if (!TextUtils.isEmpty(param)) {
                        String[] split = param.split("=");
                        if (split.length > 0) {
                            sortedMap.put(split[0], split.length == 2 ? split[1] : "");
                        }
                    }
                }
            }
            if (obj instanceof FormBody) {
                FormBody formBody = (FormBody) obj;
                for (int i = 0; i < formBody.size(); i++) {
                    sortedMap.put(formBody.encodedName(i), formBody.encodedValue(i));
                }
            }
            if (obj instanceof JSONObject) {
                JSONObject jsonObject = (JSONObject) obj;
                Iterator<String> keys = jsonObject.keys();
                while (keys.hasNext()) {
                    String key = keys.next();
                    sortedMap.put(key, jsonObject.opt(key));
                }
            }
            if (!sortedMap.containsKey("appkey")) {
                sortedMap.put("appkey", Const.APP_KEY);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return sortedMap;
    }

    private Request.Builder addHeaders(Request.Builder originBuilder, long timestamp, String nonce, String sign) {

//        TLog.e("Sign", "appkey ------ " + Const.APP_KEY);
//        TLog.e("Sign", "nonce ------ " + nonce);
//        TLog.e("Sign", "timestamp ------ " + timestamp);
//        TLog.e("Sign", "sign ------ " + sign);

        return originBuilder.addHeader("appkey", Const.APP_KEY)
                .addHeader("nonce", nonce)
                .addHeader("timestamp", String.valueOf(timestamp))
                .addHeader("sign", sign);
    }

    private String getRequestSign(Map<String, Object> paramDictionary) {
        int i = 0;
        StringBuilder rawStrBuilder = new StringBuilder();
        for (String key : paramDictionary.keySet()) {
            Object value = paramDictionary.get(key);
            if (i > 0) rawStrBuilder.append("&");
            rawStrBuilder = rawStrBuilder.append(key).append("=");
            if (value instanceof String) {
                rawStrBuilder.append(URLDecoder.decode(String.valueOf(value)));
            } else if (value instanceof JSONObject) {
                JSONObject jsonObject = (JSONObject) value;
                rawStrBuilder.append(jsonObject.toString()
                        .replace("\\/", "/"));
            }else {
                rawStrBuilder.append(value);
            }
            i++;
        }

//        TLog.e("Sign", "rawStrBuilder ------ " + rawStrBuilder.toString());

        String secret = HttpConfig.md5(Const.APP_KEY + paramDictionary.get("timestamp")).toLowerCase();

//        TLog.e("Sign", "secret ------ " + secret);

        return HttpConfig.md5((rawStrBuilder.toString() + secret)).toLowerCase();
    }

    private String bodyToString(final RequestBody request) {
        try {
            final Buffer buffer = new Buffer();
            if (request != null)
                request.writeTo(buffer);
            else
                return "";
            return buffer.readUtf8();
        } catch (final IOException e) {
            return "";
        }
    }

}
