package net.aihelp.core.net.monitor;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.ConnectivityManager;
import android.net.LinkProperties;
import android.net.Network;
import android.net.NetworkCapabilities;
import android.net.NetworkInfo;
import android.net.NetworkRequest;
import android.os.Build;
import android.telephony.TelephonyManager;

import net.aihelp.config.AIHelpContext;
import net.aihelp.init.AIHelpSupport;
import net.aihelp.utils.TLog;

import java.lang.ref.WeakReference;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import androidx.annotation.RequiresApi;

public class NetworkMonitorManager {

    public static final String TAG = "NetWorkMonitor >>> : ";
    private static NetworkMonitorManager ourInstance;
    private WeakReference<Context> mContext;

    private NetworkCallback mNetworkCallback;

    private boolean isFirstReceived;

    public static NetworkMonitorManager getInstance() {
        synchronized (NetworkMonitorManager.class) {
            if (ourInstance == null) {
                ourInstance = new NetworkMonitorManager();
            }
        }
        return ourInstance;
    }

    private Map<Object, NetworkStateReceiverMethod> netWorkStateChangedMethodMap = new ConcurrentHashMap<>();

    private NetworkMonitorManager() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            mNetworkCallback = new NetworkCallback();
        }
    }

    public void init(Context context) {
        if (context == null) {
            TLog.e("NetworkMonitorManager init Context can not be null");
            return;
        }
        this.mContext = new WeakReference<>(context);
    }

    /**
     * 初始化网络监听 根据不同版本做不同的处理
     */
    private void initMonitor() {
        if (mContext != null && mContext.get() != null) {
            ConnectivityManager connectivityManager = (ConnectivityManager) mContext.get().getSystemService(Context.CONNECTIVITY_SERVICE);
            if (connectivityManager != null && mNetworkCallback != null) {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {//API 大于26时
                    connectivityManager.registerDefaultNetworkCallback(mNetworkCallback);
                } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {//API 大于21时
                    NetworkRequest.Builder builder = new NetworkRequest.Builder();
                    NetworkRequest request = builder.build();
                    connectivityManager.registerNetworkCallback(request, mNetworkCallback);
                }
            }
        }
    }

    /**
     * 反注册广播
     */
    private void onDestroy() {
        // Receiver not registered: net.aihelp.core.net.monitor.NetworkMonitorManager$1@9d3da470
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP && mContext != null && mContext.get() != null) {
            mContext.get().unregisterReceiver(receiver);
            isFirstReceived = false;
        }
    }

    public void register(Object object) {
        try {
            if (mContext != null && mContext.get() != null) {
                ConnectivityManager connectivityManager = (ConnectivityManager) mContext.get().getSystemService(Context.CONNECTIVITY_SERVICE);
                if (connectivityManager != null && mNetworkCallback != null) {
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                        connectivityManager.registerDefaultNetworkCallback(mNetworkCallback);
                    } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                        NetworkRequest.Builder builder = new NetworkRequest.Builder();
                        NetworkRequest request = builder.build();
                        connectivityManager.registerNetworkCallback(request, mNetworkCallback);
                    }
                }

                if (object != null) {
                    Class myClass = object.getClass();
                    while (myClass != null) {
                        NetworkStateReceiverMethod NetworkStateReceiverMethod = findMethod(object, myClass);
                        if (NetworkStateReceiverMethod != null) {
                            netWorkStateChangedMethodMap.put(object, NetworkStateReceiverMethod);
                        }
                        myClass = moveToSuperclass(myClass);
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void unregister(Object object) {
        try {
            if (mContext != null && mContext.get() != null) {
                if (object != null && netWorkStateChangedMethodMap != null) {
                    netWorkStateChangedMethodMap.remove(object);
                }
                ConnectivityManager connectivityManager = (ConnectivityManager) mContext.get().getSystemService(Context.CONNECTIVITY_SERVICE);
                if (connectivityManager != null && mNetworkCallback != null && Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                    connectivityManager.unregisterNetworkCallback(mNetworkCallback);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private Class moveToSuperclass(Class clazz) {
        if (clazz == null) return null;
        clazz = clazz.getSuperclass();
        if (clazz != null) {
            String clazzName = clazz.getName();
            // Skip system classes, this degrades performance.
            // Also we might avoid some ClassNotFoundException (see FAQ for background).
            if (clazzName.startsWith("java.") || clazzName.startsWith("javax.") ||
                    clazzName.startsWith("android.") || clazzName.startsWith("androidx.")) {
                clazz = null;
            }
        }
        return clazz;
    }

    /**
     * 网络状态发生变化，需要去通知更改
     */
    private void postNetState(NetworkState netWorkState) {
        try {
            Set<Object> set = netWorkStateChangedMethodMap.keySet();
            for (Object object : set) {
                NetworkStateReceiverMethod NetworkStateReceiverMethod = netWorkStateChangedMethodMap.get(object);
                invokeMethod(NetworkStateReceiverMethod, netWorkState);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void invokeMethod(NetworkStateReceiverMethod NetworkStateReceiverMethod, NetworkState netWorkState) {
        if (NetworkStateReceiverMethod != null) {
            try {
                NetworkState[] netWorkStates = NetworkStateReceiverMethod.getNetworkState();
                for (NetworkState myState : netWorkStates) {
                    if (myState == netWorkState) {
                        NetworkStateReceiverMethod.getMethod().invoke(NetworkStateReceiverMethod.getObject(), netWorkState);
                        return;
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    private NetworkStateReceiverMethod findMethod(Object object, Class myClass) {
        NetworkStateReceiverMethod targetMethod = null;
        Method[] methods = myClass.getDeclaredMethods();
        for (Method method : methods) {
            //如果参数个数不是1个 直接忽略
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                if (method.getParameterCount() != 1) {
                    continue;
                }
            }
            //获取方法参数
            Class[] parameters = method.getParameterTypes();
            if (parameters.length != 1) {
                continue;
            }
            //参数的类型需要时NetWorkState类型
            if (parameters[0].getName().equals(NetworkState.class.getName())) {
                //是NetWorkState类型的参数
                NetworkMonitor netWorkMonitor = method.getAnnotation(NetworkMonitor.class);
                targetMethod = new NetworkStateReceiverMethod();
                //如果没有添加注解，默认就是所有网络状态变化都通知
                if (netWorkMonitor != null) {
                    NetworkState[] netWorkStates = netWorkMonitor.monitorFilter();
                    targetMethod.setNetworkState(netWorkStates);
                }
                targetMethod.setMethod(method);
                targetMethod.setObject(object);
                //只添加第一个符合的方法
                return targetMethod;
            }
        }
        return targetMethod;
    }

    private static final String ANDROID_NET_CHANGE_ACTION = "android.net.conn.CONNECTIVITY_CHANGE";

    private BroadcastReceiver receiver = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            if (isFirstReceived) return;
            if (ANDROID_NET_CHANGE_ACTION.equalsIgnoreCase(intent.getAction())) {
                //网络发生变化 没有网络-0：WIFI网络1：4G网络-4：3G网络-3：2G网络-2
                int netType = getAPNType(context);
                NetworkState netWorkState;
                switch (netType) {
                    case 0:
                        netWorkState = NetworkState.NONE;
                        break;
                    case 1:
                        netWorkState = NetworkState.WIFI;
                        break;
                    default:
                        netWorkState = NetworkState.CELLULAR;
                        break;
                }
                postNetState(netWorkState);
            }
        }
    };

    @RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
    private class NetworkCallback extends ConnectivityManager.NetworkCallback {

        @Override
        public void onAvailable(Network network) {
            super.onAvailable(network);
            if (mContext != null && mContext.get() != null) {
                int netType = getAPNType(mContext.get());
                NetworkState netWorkState;
                switch (netType) {
                    case 0://None
                        netWorkState = NetworkState.NONE;
                        break;
                    case 1://Wifi
                        netWorkState = NetworkState.WIFI;
                        break;
                    default://CELLULAR
                        netWorkState = NetworkState.CELLULAR;
                        break;
                }
                postNetState(netWorkState);
            }
        }

        /**
         * 网络不可用时调用和onAvailable成对出现
         */
        @Override
        public void onLost(Network network) {
            super.onLost(network);
            postNetState(NetworkState.NONE);
        }

        /**
         * 在网络连接正常的情况下，丢失数据会有回调 即将断开时
         */
        @Override
        public void onLosing(Network network, int maxMsToLive) {
            super.onLosing(network, maxMsToLive);
        }

        /**
         * 网络功能更改 满足需求时调用
         *
         * @param network
         * @param networkCapabilities
         */
        @Override
        public void onCapabilitiesChanged(Network network, NetworkCapabilities networkCapabilities) {
            super.onCapabilitiesChanged(network, networkCapabilities);
        }

        /**
         * 网络连接属性修改时调用
         *
         * @param network
         * @param linkProperties
         */
        @Override
        public void onLinkPropertiesChanged(Network network, LinkProperties linkProperties) {
            super.onLinkPropertiesChanged(network, linkProperties);
        }

        /**
         * 网络缺失network时调用
         */
        @Override
        public void onUnavailable() {
            super.onUnavailable();
        }
    }

    private static int getAPNType(Context context) {
        if (context == null) return 0;
        //结果返回值
        int netType = 0;
        //获取手机所有连接管理对象
        ConnectivityManager manager = (ConnectivityManager) context.getSystemService(Context
                .CONNECTIVITY_SERVICE);
        //获取NetworkInfo对象
        NetworkInfo networkInfo = null;
        if (manager != null) {
            networkInfo = manager.getActiveNetworkInfo();
        }
        //NetworkInfo对象为空 则代表没有网络
        if (networkInfo == null) {
            return netType;
        }
        //否则 NetworkInfo对象不为空 则获取该networkInfo的类型
        int nType = networkInfo.getType();
        if (nType == ConnectivityManager.TYPE_WIFI) {
            //WIFI
            netType = 1;
        } else if (nType == ConnectivityManager.TYPE_MOBILE) {
            int nSubType = networkInfo.getSubtype();
            TelephonyManager telephonyManager = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
            if (telephonyManager != null) {

                //3G   联通的3G为UMTS或HSDPA 电信的3G为EVDO
                if (nSubType == TelephonyManager.NETWORK_TYPE_LTE
                        && !telephonyManager.isNetworkRoaming()) {
                    netType = 4;
                } else if (nSubType == TelephonyManager.NETWORK_TYPE_UMTS
                        || nSubType == TelephonyManager.NETWORK_TYPE_HSDPA
                        || nSubType == TelephonyManager.NETWORK_TYPE_EVDO_0
                        && !telephonyManager.isNetworkRoaming()) {
                    netType = 3;
                    //2G 移动和联通的2G为GPRS或EGDE，电信的2G为CDMA
                } else if (nSubType == TelephonyManager.NETWORK_TYPE_GPRS
                        || nSubType == TelephonyManager.NETWORK_TYPE_EDGE
                        || nSubType == TelephonyManager.NETWORK_TYPE_CDMA
                        && !telephonyManager.isNetworkRoaming()) {
                    netType = 2;
                } else {
                    netType = 2;
                }
            }
        }
        return netType;
    }

}
