package net.aihelp.data.logic;

import android.app.Activity;
import android.content.Context;
import android.text.TextUtils;
import android.view.Gravity;
import android.view.View;
import android.widget.TextView;
import android.widget.Toast;

import net.aihelp.common.API;
import net.aihelp.common.Const;
import net.aihelp.common.CustomConfig;
import net.aihelp.common.UserProfile;
import net.aihelp.common.IntentValues;
import net.aihelp.core.net.check.Ping;
import net.aihelp.core.net.check.TraceRoute;
import net.aihelp.core.net.mqtt.AIHelpMqtt;
import net.aihelp.core.net.mqtt.callback.IMqttCallback;
import net.aihelp.core.net.mqtt.config.MqttConfig;
import net.aihelp.core.ui.dialog.AlertDialog;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.core.util.concurrent.ApiExecutorFactory;
import net.aihelp.data.event.SupportActionEvent;
import net.aihelp.data.local.ElvaRepository;
import net.aihelp.data.localize.util.LocalizeUtil;
import net.aihelp.data.model.faq.FaqListEntity;
import net.aihelp.data.model.cs.storyline.BotTag;
import net.aihelp.data.model.cs.ConversationMsg;
import net.aihelp.data.model.cs.ElvaBotMsg;

import net.aihelp.db.bot.ConversationDBHelper;
import net.aihelp.ui.cs.BaseCSFragment;
import net.aihelp.ui.helper.ConversationHelper;
import net.aihelp.ui.helper.StatisticHelper;
import net.aihelp.utils.DeviceUuidFactory;
import net.aihelp.data.localize.LocalizeHelper;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.TLog;
import net.aihelp.utils.ToastUtil;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;


public class ElvaBotPresenter extends MqttPresenter<BaseCSFragment, ElvaRepository> {

    // To avoid adding welcome msg in every onResume
    private boolean isLocalElvaReplied;

    public ElvaBotPresenter(Context context) {
        super(context);
    }

    public void prepareMqttConnection(IMqttCallback callback) {
        callback.showMqttLoading();

        if (TextUtils.isEmpty(Const.CUSTOM_STORY_NODE)) {
            // LocalizeHelper.goFetchElvaBotData(mContext);
            if (LocalizeUtil.isAIMLLocalized() && !isLocalElvaReplied) {
                ArrayList<ConversationMsg> list = new ArrayList<>();
                ElvaBotMsg defaultMsg = ConversationHelper.getDefaultMsg();
                if (!defaultMsg.isBotStupid()) {
                    list.add(defaultMsg);
                    Const.isLocalWelcomeAvailable = true;
                }
                // clear all elva message histories to avoid sending old messages to server
                ConversationDBHelper.clearElvaMsg();
                mView.updateChatList(list);
                isLocalElvaReplied = true;
            }
        }

        if (isNetworkAvailable()) {
            callback.updateHostView(mView);
            AIHelpMqtt.getInstance().prepare(MqttConfig.TYPE_ELVA_BOT, callback);
        } else {
            mView.showError(ResResolver.getString("aihelp_network_no_connect"));
        }

    }

    public void askForAnswer(String msg) {
        try {
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("msg", msg);
//        jsonObject.put("imgFlag", "0");
            jsonObject.put("answer_type", "wildcard");
            mqtt(API.TOPIC_BOT_CHAT, jsonObject);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void markWhetherFaqHelpful(boolean helpful, ElvaBotMsg botMsg) {
        try {
            JSONObject faqFeedback = new JSONObject();
            faqFeedback.put("isLike", helpful ? "1" : "2");
            faqFeedback.put("timeMillis", botMsg.getTimeStamp());
            faqFeedback.put("isClickDetail", botMsg.isFaqViewed());
            faqFeedback.put("contentId", botMsg.getFaqContentId());
            mqtt(API.TOPIC_BOT_FAQ, faqFeedback);
            if (!helpful && CustomConfig.CustomerService.isUnhelpfulFaqStoryNodeEnable) {
                askForAnswer("UnhelpfulFaqStoryNode");
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void postFeedbackOnFaq(ElvaBotMsg botMsg, String feedbackMsg) {
        try {
            JSONObject params = new JSONObject();
            params.put("Language", Const.CORRECT_LANGUAGE);
            params.put("PlayerId", String.format("%s|%s", Const.APP_ID, UserProfile.USER_ID));
            params.put("PlayerName", UserProfile.USER_NAME);
            params.put("FaqId", botMsg.getFaqMainId());
            params.put("contentId", botMsg.getFaqContentId());
            params.put("Message", feedbackMsg);
            params.put("Type", "1");
            params.put("CreateTime", String.valueOf(botMsg.getTimeStamp()));
            // only for bot
            params.put("PlayerQuestion", botMsg.getUserInput());
            params.put("TicketId", botMsg.getFaqTicketId());
            params.put("PitchonQuestion", botMsg.getBotMsg());
            post(API.FAQ_FEEDBACK_URL, params, null);
            StatisticHelper.whenBotFAQGotNegativeFeedback(String.valueOf(botMsg.getTimeStamp()),
                    botMsg.getFaqContentId(), botMsg.getFaqMainId());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 1、Save elva chat history to database;
     * 2、Log chat msg group (bot & user) to server, log similar match status;
     * 3、check whether to start network checking
     */
    public void handleBotMsg(ElvaBotMsg botMsg) {
        ConversationDBHelper.storeElvaMsg(botMsg.getTimeStamp(), botMsg.getRawResponse());
        StatisticHelper.whenBotMessagePrepared(botMsg);
        prepareNetworkCheck(botMsg);
    }

    private void prepareNetworkCheck(ElvaBotMsg botMsg) {
        if (!Const.TOGGLE_NET_CHECK || TextUtils.isEmpty(Const.NET_CHECK_HOST) || !botMsg.isHasTag())
            return;
        if (mView.isNetCheckingInProgress()) {
            ToastUtil.INSTANCE.makeRawToast(mContext, ResResolver.getString("aihelp_network_check_in_progress"));
            return;
        }
        tagLoop:
        for (BotTag botTag : botMsg.getBotTagList()) {
            for (String pingTags : Const.NET_PING.split(",")) {
                if (pingTags.equalsIgnoreCase(botTag.getTagName())) {

                    if (!isNetworkAvailable()) {
                        ToastUtil.INSTANCE.makeRawToast(mContext, ResResolver.getString("aihelp_network_no_connect"));
                        return;
                    }

                    mView.updateNetCheckingStatus(true);
                    Ping.start(Const.NET_CHECK_HOST, new Ping.Callback() {
                        @Override
                        public void complete(Ping.Result result) {
                            handleNetworkResult(result);
                        }
                    });
                    break tagLoop;
                }
            }
            for (String pingTags : Const.NET_TRACE_ROUTE.split(",")) {
                if (pingTags.equalsIgnoreCase(botTag.getTagName())) {

                    if (!isNetworkAvailable()) {
                        ToastUtil.INSTANCE.makeRawToast(mContext, ResResolver.getString("aihelp_network_no_connect"));
                        return;
                    }

                    mView.updateNetCheckingStatus(true);
                    TraceRoute.start(Const.NET_CHECK_HOST, new TraceRoute.Callback() {
                        @Override
                        public void complete(TraceRoute.Result result) {
                            handleNetworkResult(result);
                        }
                    });
                    break tagLoop;
                }
            }
        }
    }

    private void handleNetworkResult(final Object result) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                mView.updateNetCheckingStatus(false);
                if (result instanceof Ping.Result) {
                    Ping.Result pingResult = (Ping.Result) result;
                    if (Const.sCheckResultListener != null) {
                        Const.sCheckResultListener.onNetworkCheckResult(pingResult.result);
                    }
                    if (pingResult.sent > 0 && pingResult.avg < 300) {
                        showNetworkFineToast();
                        return;
                    }
                    showNetworkSlowDialog(pingResult.result, true);
                }

                if (result instanceof TraceRoute.Result) {
                    TraceRoute.Result traceRouteResult = (TraceRoute.Result) result;
                    if (Const.sCheckResultListener != null) {
                        Const.sCheckResultListener.onNetworkCheckResult(traceRouteResult.content());
                    }
                    showNetworkSlowDialog(traceRouteResult.content(), false);
                }
            }
        });

    }

    private void showNetworkFineToast() {
        if (mContext instanceof Activity) {
            Toast toast = new Toast(mContext);
            View view = View.inflate(mContext, ResResolver.getLayoutId("aihelp_toast_network_fine"), null);
            TextView textView = view.findViewById(ResResolver.getViewId("aihelp_toast_txt"));
            textView.setText(ResResolver.getString("aihelp_network_check_fine"));
            toast.setView(view);
            toast.setGravity(Gravity.CENTER, 0, 0);
            Activity activity = (Activity) mContext;
            if (!activity.isFinishing()) {
                toast.show();
            } else {
                TLog.e("Activity finished when network toast is about to show.");
            }
        }
    }

    private void showNetworkSlowDialog(final String result, final boolean isPing) {
        // FIXME: 2020/11/18 show dialog when in manual support page
        if (mContext instanceof Activity && mView.isVisible()) {

            final AlertDialog uploadLogDialog = new AlertDialog.Builder(mContext)
                    .setContentView(ResResolver.getLayoutId("aihelp_dia_upload_net_check"))
                    .setWidthByDevice()
                    .create();

            TextView tvTitle = uploadLogDialog.findViewById(ResResolver.getViewId("aihelp_tv_title"));
            TextView tvSubTitle = uploadLogDialog.findViewById(ResResolver.getViewId("aihelp_tv_sub_title"));
            TextView tvCancel = uploadLogDialog.findViewById(ResResolver.getViewId("aihelp_tv_cancel"));
            TextView tvConfirm = uploadLogDialog.findViewById(ResResolver.getViewId("aihelp_tv_confirm"));

            tvTitle.setText(ResResolver.getString("aihelp_network_upload_log_title"));
            tvSubTitle.setText(ResResolver.getString("aihelp_network_check_poor"));
            tvCancel.setText(ResResolver.getString("aihelp_no"));
            tvConfirm.setText(ResResolver.getString("aihelp_yes"));

            uploadLogDialog.setOnClickListener(ResResolver.getViewId("aihelp_tv_cancel"), new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    uploadLogDialog.dismiss();
                }
            });
            uploadLogDialog.setOnClickListener(ResResolver.getViewId("aihelp_tv_confirm"), new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    EventBus.getDefault().post(new SupportActionEvent(IntentValues.SHOW_SUPPORT_ACTION,
                            IntentValues.SUPPORT_ACTION_MSG_UNREAD));
                    postNetworkResultToServer(result, isPing ? "network,ping" : "network,traceroute");
                    logCheckResultSent(isPing ? "ping" : "traceroute");
                    uploadLogDialog.dismiss();
                }
            });

            Activity activity = (Activity) mContext;
            if (!activity.isFinishing()) {
                uploadLogDialog.show();
            } else {
                TLog.e("Activity finished when network dialog is about to show.");
            }

        }
    }

    private void logCheckResultSent(String checkType) {
        try {
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("deviceId", DeviceUuidFactory.id(mContext));
            jsonObject.put("playerId", UserProfile.USER_ID);
            jsonObject.put("type", checkType);
            post(API.LOG_NETWORK_RESULT_SENT, jsonObject, null);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void postNetworkResultToServer(String userSay, String chatTags) {
        try {
            JSONObject chatParams = new JSONObject();
            chatParams.put("msg", userSay);
            chatParams.put("timeStamp", System.currentTimeMillis());
            chatParams.put("chatTags", chatTags);
            chatParams.put("imgFlag", "0");
            mqtt(API.TOPIC_CONVERSATION_SEND, chatParams);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public List<FaqListEntity> getMatchedFaqListForAlert(String input) {
        return mRepo.getMatchedFaqListForAlert(input);
    }

}
