package net.aihelp.init;

import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.content.Intent;
import android.text.TextUtils;
import android.util.Log;

import net.aihelp.BuildConfig;
import net.aihelp.common.Const;
import net.aihelp.common.SpKeys;
import net.aihelp.config.AIHelpContext;
import net.aihelp.config.FaqConfig;
import net.aihelp.common.IntentValues;
import net.aihelp.config.OperationConfig;
import net.aihelp.config.ConversationConfig;
import net.aihelp.config.UserConfig;
import net.aihelp.config.enums.PublishCountryOrRegion;
import net.aihelp.config.enums.PushPlatform;
import net.aihelp.core.net.http.config.HttpConfig;
import net.aihelp.core.net.monitor.NetworkMonitorManager;
import net.aihelp.core.net.mqtt.config.MqttConfig;
import net.aihelp.core.util.concurrent.ApiExecutorFactory;
import net.aihelp.data.logic.InitPresenter;
import net.aihelp.core.util.crash.AIHelpCrashHandler;
import net.aihelp.exception.AIHelpInitException;
import net.aihelp.ui.helper.StatisticHelper;
import net.aihelp.ui.listener.OnAIHelpInitializedCallback;
import net.aihelp.ui.listener.OnAIHelpSessionCloseCallback;
import net.aihelp.ui.listener.OnAIHelpSessionOpenCallback;
import net.aihelp.ui.listener.OnSpecificUrlClickedCallback;
import net.aihelp.ui.listener.OnMessageCountArrivedCallback;
import net.aihelp.ui.listener.OnNetworkCheckResultCallback;
import net.aihelp.ui.listener.OnOperationUnreadChangedCallback;
import net.aihelp.ui.listener.OnSpecificFormSubmittedCallback;
import net.aihelp.utils.AppInfoUtil;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.SchemaUtil;
import net.aihelp.utils.TLog;
import net.aihelp.utils.SpUtil;
import net.aihelp.utils.ToastUtil;

import java.lang.ref.WeakReference;
import java.lang.reflect.Method;

class AIHelpCore {

    private WeakReference<Activity> mActivity;
    private Context mAppContext;
    private InitPresenter mInitPresenter;
    private OnAIHelpInitializedCallback initListener;

    public void init(Context context, String appKey, String domain, String appId, String language) throws AIHelpInitException {
        try {

            final String trimApiKey = !TextUtils.isEmpty(appKey) ? appKey.trim() : appKey;
            final String trimDomain = !TextUtils.isEmpty(domain) ? domain.trim() : domain;
            final String trimAppId = generateAppIdFromAppKey(appKey, domain, appId);
            SchemaUtil.validateInitializeCredentials(context, trimApiKey, trimDomain, trimAppId);

            ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    if (context instanceof Activity) {
                        mActivity = new WeakReference<>((Activity) context);
                    }
                    mAppContext = context.getApplicationContext();
                    AIHelpContext.getInstance().setContext(mAppContext);

                    AIHelpCrashHandler.INSTANCE.init(mAppContext);

                    mInitPresenter = new InitPresenter(mAppContext, trimApiKey, trimDomain, trimAppId, language);

                    if (SpUtil.getInstance().getBoolean(SpKeys.TOGGLE_LOG)) TLog.initLog(true);

                    NetworkMonitorManager.getInstance().init(mAppContext);
                    mInitPresenter.loadUpListeners(initListener);
                    mInitPresenter.doInitForAIHelp();
                }
            });

        } catch (AIHelpInitException e) {
            throw e;
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private String generateAppIdFromAppKey(String appKey, String domain, String appId) {
        if (TextUtils.isEmpty(appId) && !TextUtils.isEmpty(appKey) && !TextUtils.isEmpty(domain)) {
            String updatedDomain = domain.replace("https://", "").replace("http://", "");
            String newAppKey = HttpConfig.md5(appKey.toLowerCase() + "AIHelpSDKToAndroidAppId");
            String newDomain = updatedDomain.toLowerCase().substring(0, updatedDomain.indexOf("."));
            return newDomain + "_platform_" + newAppKey;
        }
        return appId.trim();
    }

    void updateSDKLanguage(String sdkLanguage) {
        try {
            if (isInitStillInProgress()) return;
            mInitPresenter.updateSDKLanguage(sdkLanguage);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void showConversation(ConversationConfig conversationConfig) {
        try {
            if (isInitStillInProgress() || MqttConfig.getInstance().isConnected()) return;
            mInitPresenter.updateConversationFields(conversationConfig);
            Intent showSupportIntent = IntentValues.getShowSupportIntent(mAppContext, conversationConfig);
            if (mActivity != null && mActivity.get() != null) {
                mActivity.get().startActivity(showSupportIntent);
            } else {
                mAppContext.startActivity(IntentValues.wrapForApplicationContext(showSupportIntent));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void showAllFAQSections(FaqConfig faqConfig) {
        try {
            if (isInitStillInProgress()) return;
            StatisticHelper.whenAllFAQSectionsVisible(false);
            mInitPresenter.updateConversationFields(faqConfig.getConversationConfig());
            Intent showFAQIntent = IntentValues.getShowFAQIntent(mAppContext, faqConfig);
            if (mActivity != null && mActivity.get() != null) {
                mActivity.get().startActivity(showFAQIntent);
            } else {
                mAppContext.startActivity(IntentValues.wrapForApplicationContext(showFAQIntent));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void showFAQSection(String sectionId, FaqConfig faqConfig) {
        try {
            if (isInitStillInProgress()) return;
            mInitPresenter.updateConversationFields(faqConfig.getConversationConfig());
            Intent showFAQSectionIntent = IntentValues.getShowFAQSectionIntent(mAppContext, sectionId, faqConfig);
            if (mActivity != null && mActivity.get() != null) {
                mActivity.get().startActivity(showFAQSectionIntent);
            } else {
                mAppContext.startActivity(IntentValues.wrapForApplicationContext(showFAQSectionIntent));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void showSingleFAQ(String mainOrDisplayId, FaqConfig faqConfig) {
        try {
            if (isInitStillInProgress()) return;
            mInitPresenter.updateConversationFields(faqConfig.getConversationConfig());
            Intent showSingleFAQIntent = IntentValues.getShowSingleFAQIntent(mAppContext, mainOrDisplayId, faqConfig);
            if (mActivity != null && mActivity.get() != null) {
                mActivity.get().startActivity(showSingleFAQIntent);
            } else {
                mAppContext.startActivity(IntentValues.wrapForApplicationContext(showSingleFAQIntent));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void showOperation(OperationConfig operationConfig) {
        try {
            if (isInitStillInProgress() || MqttConfig.getInstance().isConnected()) return;
            mInitPresenter.updateConversationFields(operationConfig.getConversationConfig());
            Intent showOperateIntent = IntentValues.getShowOperateIntent(mAppContext, operationConfig);
            if (mActivity != null && mActivity.get() != null) {
                mActivity.get().startActivity(showOperateIntent);
            } else {
                mAppContext.startActivity(IntentValues.wrapForApplicationContext(showOperateIntent));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void showUrl(String url) {
        try {
            if (mAppContext != null && !TextUtils.isEmpty(url)) {
                Intent showUrlIntent = IntentValues.getUrlIntent(mAppContext, url);
                if (mActivity != null && mActivity.get() != null) {
                    mActivity.get().startActivity(showUrlIntent);
                } else {
                    mAppContext.startActivity(IntentValues.wrapForApplicationContext(showUrlIntent));
                }
            } else {
                TLog.e("AIHelp", "AIHelpSupport showUrl illegal argument");
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void updateUserInfo(UserConfig config) {
        try {
            if (isPresenterStillNull()) return;
            mInitPresenter.updateUserProfile(config);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void setUploadLogPath(String logPath) {
        try {
            if (isPresenterStillNull()) return;
            mInitPresenter.setUploadLogPath(logPath);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void setNetworkCheckHostAddress(String hostAddress, OnNetworkCheckResultCallback listener) {
        try {
            if (isPresenterStillNull()) return;
            mInitPresenter.setNetworkCheckHostAddress(hostAddress, listener);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void setPushTokenAndPlatform(String pushToken, PushPlatform pushPlatform) {
        try {
            if (isPresenterStillNull() || pushPlatform == null) return;
            mInitPresenter.postCrmPushTokenInfo(pushToken, pushPlatform.getValue());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void additionalSupportFor(PublishCountryOrRegion countryOrRegion) {
        Const.countryOrRegion = countryOrRegion;
    }

    void startUnreadMessageCountPolling(OnMessageCountArrivedCallback listener) {
        try {
            if (isPresenterStillNull()) return;
            mInitPresenter.startUnreadMessagePolling(listener);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void setOnSpecificFormSubmittedCallback(OnSpecificFormSubmittedCallback callback) {
        try {
            if (isPresenterStillNull()) return;
            mInitPresenter.setOnSpecificFormSubmittedCallback(callback);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void setOnAIHelpSessionOpenCallback(OnAIHelpSessionOpenCallback callback) {
        try {
            if (isPresenterStillNull()) return;
            mInitPresenter.setOnAIHelpSessionOpenCallback(callback);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void setOnAIHelpSessionCloseCallback(OnAIHelpSessionCloseCallback callback) {
        try {
            if (isPresenterStillNull()) return;
            mInitPresenter.setOnAIHelpSessionCloseCallback(callback);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void setOnAIHelpInitializedCallback(OnAIHelpInitializedCallback listener) {
        try {
            if (mInitPresenter == null) {
                initListener = listener;
                return;
            }
            mInitPresenter.loadUpListeners(listener);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void setOnOperationUnreadChangedCallback(OnOperationUnreadChangedCallback callback) {
        try {
            if (isPresenterStillNull()) return;
            mInitPresenter.setOnOperationUnreadChangedCallback(callback);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void setOnSpecificUrlClickedCallback(OnSpecificUrlClickedCallback callback) {
        try {
            if (isPresenterStillNull()) return;
            mInitPresenter.setOnSpecificUrlClickedCallback(callback);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void enableLogging(boolean enable) {
        TLog.initLog(enable);
    }

    private boolean isInitStillInProgress() {
        if (!AIHelpContext.successfullyInit.get()) {
            Log.e("AIHelp", "AIHelp is during initialization process at this time, the API you called is not available, please try again later.");
            return true;
        }
        if (!AppInfoUtil.isNetworkAvailable(mAppContext)) {
            ToastUtil.INSTANCE.makeRawToast(mAppContext, ResResolver.getString("aihelp_network_no_connect"));
            return true;
        }
        return false;
    }

    private boolean isPresenterStillNull() {
        try {
            if (mInitPresenter == null) {
                Log.e("AIHelp", "Please ensure AIHelpSupport#init is called at the very first place before you call any other API.");
                return true;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return false;
    }

    // ===================== getInstance =======================

    public static AIHelpCore getInstance() {
        return Holder.INSTANCE;
    }

    private AIHelpCore() {
    }

    private static class Holder {
        private static final AIHelpCore INSTANCE = new AIHelpCore();
    }

}
