package net.aihelp.init;

import android.app.ActivityManager;
import android.content.Context;
import android.text.TextUtils;
import android.util.Log;

import net.aihelp.BuildConfig;
import net.aihelp.common.Const;
import net.aihelp.common.UserProfile;
import net.aihelp.config.AIHelpContext;
import net.aihelp.config.FaqConfig;
import net.aihelp.config.OperationConfig;
import net.aihelp.config.ConversationConfig;
import net.aihelp.config.UserConfig;
import net.aihelp.config.enums.PublishCountryOrRegion;
import net.aihelp.config.enums.PushPlatform;
import net.aihelp.core.util.concurrent.ApiExecutorFactory;
import net.aihelp.core.util.logger.AIHelpLogger;
import net.aihelp.exception.AIHelpInitException;
import net.aihelp.ui.listener.OnAIHelpInitializedCallback;
import net.aihelp.ui.listener.OnAIHelpSessionCloseCallback;
import net.aihelp.ui.listener.OnAIHelpSessionOpenCallback;
import net.aihelp.ui.listener.OnSpecificUrlClickedCallback;
import net.aihelp.ui.listener.OnMessageCountArrivedCallback;
import net.aihelp.ui.listener.OnNetworkCheckResultCallback;
import net.aihelp.ui.listener.OnOperationUnreadChangedCallback;
import net.aihelp.ui.listener.OnSpecificFormSubmittedCallback;
import net.aihelp.utils.DeviceUuidFactory;
import net.aihelp.utils.TLog;

import java.util.List;
import java.util.Map;

public class AIHelpSupport {

    /**
     * Initialize AIHelp support
     * <p>
     * When initializing AIHelp you must at least pass four tokens.
     * You should initialize AIHelp at the beginning of your app lifecycle, ideally at the top of Application#onCreate.
     * This method can throw the InstallException asynchronously if the install keys are not in the correct format.
     *
     * @param context This should be your application instance
     * @param appKey  This is your developer API Key
     * @param domain  This is your domain name without any http:// or forward slashes
     * @param appId   This is the unique ID assigned to your app
     */
    public static void init(Context context, String appKey, String domain, String appId) {
        init(context, appKey, domain, appId, Const.CORRECT_LANGUAGE);
    }

    /**
     * Initialize AIHelp support
     * <p>
     * When initializing AIHelp you must at least pass four tokens.
     * You should initialize AIHelp at the beginning of your app lifecycle, ideally at the top of Application#onCreate.
     * This method can throw the InstallException asynchronously if the install keys are not in the correct format.
     *
     * @param context  This should be your application instance
     * @param appKey   This is your developer API Key
     * @param domain   This is your domain name without any http:// or forward slashes
     * @param appId    This is the unique ID assigned to your app
     * @param language This is your expected init language
     */
    public static void init(final Context context, final String appKey, final String domain,
                            final String appId, final String language) throws AIHelpInitException {
        AIHelpCore.getInstance().init(context, appKey, domain, appId, language);
    }

    /**
     * Register callback for the process of AIHelp's initialization.
     * <p>
     * After you register this callback, SDK will let you know if the init work is done.
     * You can call this method either before or after the {@link #init(Context, String, String, String)} call.
     *
     * @param listener callback for AIHelp initialization
     */
    public static void setOnAIHelpInitializedCallback(OnAIHelpInitializedCallback listener) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().setOnAIHelpInitializedCallback(listener);
            }
        });
    }

    /**
     * Show the AIHelp support screen, which is Elva bot by default.
     * <p>
     * If you want to change your support intent or custom your welcome message,
     * please check {@link #showConversation(ConversationConfig)} and {@link ConversationConfig} for more information.
     */
    public static void showConversation() {
        showConversation(new ConversationConfig.Builder().build());
    }

    /**
     * Show the AIHelp support screen, with Optional Arguments.
     * <p>
     * {@link #showConversation()} shows Elva Bot support by default.
     * In addition, user CAN NOT get in touch with customer service in Elva Bot page, unless there is
     * an unfinished ticket or they submit a form from elva bot.
     * <p>
     * But You could configure your default support intent with {@link ConversationConfig}.
     * For instance, You can show different welcome msg for different customer, make your customer service always online for your VIPs,
     * even skip bot support and show customer service by default for them.
     *
     * @param conversationConfig configure different support actions
     * @see ConversationConfig for more information.
     */
    public static void showConversation(final ConversationConfig conversationConfig) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().showConversation(conversationConfig);
            }
        });
    }

    /**
     * Show the AIHelp screen with only the faqs, which is root section by default.
     * <p>
     * To show the AIHelp screen with only the faq sections with search with optional arguments, you can use this api.
     * If you want to pass some options to configure your faqs display, check {@link #showAllFAQSections(FaqConfig)} for more information.
     */
    public static void showAllFAQSections() {
        showAllFAQSections(new FaqConfig.Builder().build());
    }

    /**
     * Show the AIHelp screen with only the faqs, with Optional Arguments.
     * <p>
     * User can get in touch with your customer service via FAQs, you can configure this with {@link FaqConfig}.
     * Such as the EXACT moment user can see the CONTACT US button.
     * <p>
     * As you can get support via FAQs, you can configure your support configs just as same as {@link #showConversation()}.
     *
     * @param faqConfig the configs which contains custom configurations for faqs & supports
     * @see FaqConfig
     * @see ConversationConfig
     */
    public static void showAllFAQSections(final FaqConfig faqConfig) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().showAllFAQSections(faqConfig);
            }
        });
    }

    /**
     * Show the AIHelp screen with faqs from a particular section.
     * <p>
     * To show the AIHelp screen for showing a particular faq section you need to pass
     * the section-id of the faq section.
     * <p>
     * This api takes on the default options, if you want to pass some options to configure your faqs display,
     * check {@link #showFAQSection(String, FaqConfig)} for more information.
     *
     * @param sectionId the publish id associated with the faq section which is shown in the FAQ page on the admin side.(https://aihelp.net/dashboard/#/FAQ)
     */
    public static void showFAQSection(String sectionId) {
        showFAQSection(sectionId, new FaqConfig.Builder().build());
    }

    /**
     * Show the AIHelp screen with faqs from a particular section, with Optional Arguments.
     * <p>
     * To show the AIHelp screen for showing a particular faq section you need to pass
     * the section-id of the faq section.
     * <p>
     * In addition, user can get in touch with your customer service via FAQs, you can configure this with {@link FaqConfig}.
     * Such as the EXACT moment user can see the CONTACT US button.
     * Besides, as you can get support via FAQs, you can configure your support configs just as same as {@link #showConversation()}.
     *
     * @param sectionId the publish id associated with the faq section which is shown in the FAQ page on the admin side.(https://aihelp.net/dashboard/#/FAQ)
     * @param faqConfig the configs which contains custom configurations for faqs & supports
     * @see FaqConfig
     * @see ConversationConfig
     */
    public static void showFAQSection(final String sectionId, final FaqConfig faqConfig) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().showFAQSection(sectionId, faqConfig);
            }
        });
    }

    /**
     * Show the AIHelp screen with a single faq.
     * <p>
     * To show the AIHelp screen for showing a specific FAQ, you need to pass the faq id.
     * <p>
     * This api takes on the default options, if you want to pass some options to configure your faq display,
     * check {@link #showSingleFAQ(String, FaqConfig)} for more information.
     *
     * @param faqId the publish id associated with the faq which is shown when you expand a single FAQ.(https://aihelp.net/dashboard/#/FAQ)
     */
    public static void showSingleFAQ(String faqId) {
        showSingleFAQ(faqId, new FaqConfig.Builder().build());
    }

    /**
     * Show the AIHelp screen with a single faq, with Optional Arguments.
     * <p>
     * To show the AIHelp screen for showing a specific FAQ, you need to pass the faq id.
     * <p>
     * In addition, user can get in touch with your customer service via FAQs, you can configure this with {@link FaqConfig}.
     * Such as the EXACT moment user can see the CONTACT US button.
     * Besides, as you can get support via FAQs, you can configure your support configs just as same as {@link #showConversation()}.
     *
     * @param faqId     the publish id associated with the faq which is shown when you expand a single FAQ.(https://aihelp.net/dashboard/#/FAQ)
     * @param faqConfig the configs which contains custom configurations for faqs & supports
     * @see FaqConfig
     * @see ConversationConfig
     */
    public static void showSingleFAQ(final String faqId, final FaqConfig faqConfig) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().showSingleFAQ(faqId, faqConfig);
            }
        });
    }

    /**
     * Show the AIHelp operation screen.
     * <p>
     * This api takes on the default options, which selects Elva Bot by default.
     * If you want to change the operation default selection or elva bot title,
     * please check {@link #showOperation(OperationConfig)} for more information.
     */
    public static void showOperation() {
        showOperation(new OperationConfig.Builder().build());
    }

    /**
     * Show the AIHelp operation screen, with Optional Arguments.
     * <p>
     * You can change your default selection or elva bot title by passing in {@link OperationConfig}.
     * Besides, as you can get support via operation, you can configure your support configs just as same as {@link #showConversation()}.
     *
     * @param operationConfig the configs which contains custom configurations for operations & supports
     * @see OperationConfig
     * @see ConversationConfig
     */
    public static void showOperation(final OperationConfig operationConfig) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().showOperation(operationConfig);
            }
        });
    }

    public static void showUrl(final String url) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().showUrl(url);
            }
        });
    }

    /**
     * Update a user's profile via UserConfig.
     * <p>
     * Please check {@link UserConfig} for more detail information.
     *
     * @param config configs which contains all information about a user.
     * @see UserConfig
     */
    public static void updateUserInfo(final UserConfig config) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().updateUserInfo(config);
            }
        });
    }

    /**
     * Clear the values set to a user, reset the userId to deviceId, userName to 'anonymous'.
     */
    public static void resetUserInfo() {
        Context context = AIHelpContext.getInstance().getContext();
        if (context != null) {
            UserConfig resetUser = new UserConfig.Builder()
                    .setUserId(DeviceUuidFactory.id(context))
                    .setServerId("-1")
                    .setUserName("anonymous")
                    .build();
            updateUserInfo(resetUser);
        }
    }

    /**
     * Change the SDK language. By default, the device's prefered language is used.
     * The call will fail in the following cases :
     * 1. If a AIHelp session is already active at the time of invocation
     * 2. Language code is incorrect
     * 3. Corresponding localization file is not found
     *
     * @param sdkLanguage the string representing the language code. For example, use 'fr' for French.
     */
    public static void updateSDKLanguage(String sdkLanguage) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().updateSDKLanguage(sdkLanguage);
            }
        });
    }

    /**
     * start in-app unread message count polling
     * <p>
     * This is a schedule work, get unread message count every five minutes.
     *
     * @param listener callback for unread message polling
     */
    public static void startUnreadMessageCountPolling(OnMessageCountArrivedCallback listener) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().startUnreadMessageCountPolling(listener);
            }
        });
    }

    /**
     * Set log path for uploading.
     * <p>
     * In order to serve your customers well, you can upload customer-related-logs when tickets are created or
     * specific forms are submitted.
     *
     * @param logPath the absolute path of log, which will be uploaded when needed
     */
    public static void setUploadLogPath(String logPath) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().setUploadLogPath(logPath);
            }
        });
    }

    /**
     * @param pushToken    the pushToken received from the push notification servers.
     * @param pushPlatform the specific push platform, must be of
     *                     {@link PushPlatform#APNS}, {@link PushPlatform#FIREBASE},
     *                     {@link PushPlatform#JPUSH}, {@link PushPlatform#GETUI}
     * @deprecated Use {@link #updateUserInfo(UserConfig)} instead.
     * <p>
     * set the pushToken and platform to enable push notifications.
     * NOTE: You must get the specific push sdk in your App,
     * and call {@link #updateUserInfo(UserConfig)} BEFORE this invocation.
     */
    public static void setPushTokenAndPlatform(String pushToken, PushPlatform pushPlatform) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().setPushTokenAndPlatform(pushToken, pushPlatform);
            }
        });
    }

    /**
     * Set up host address for network check, without result callback.
     * Check {@link #setNetworkCheckHostAddress(String, OnNetworkCheckResultCallback)} for more information.
     *
     * @param hostAddress host address for network checking, without schemes such 'https://' or 'http://'.
     *                    For example, you can pass in 'www.google.com' or just 'google.com', no schemes are needed.
     */
    public static void setNetworkCheckHostAddress(String hostAddress) {
        setNetworkCheckHostAddress(hostAddress, null);
    }

    /**
     * Set up host address for network check with result callback.
     * <p>
     * With this api, you can get the network check result passing back to you.
     *
     * @param hostAddress host address for network checking, without schemes such 'https://' or 'http://'.
     *                    For example, you can pass in 'www.google.com' or just 'google.com', no schemes are needed.
     * @param listener    network check result callback, you can get the check result via this listener
     */
    public static void setNetworkCheckHostAddress(String hostAddress, OnNetworkCheckResultCallback listener) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().setNetworkCheckHostAddress(hostAddress, listener);
            }
        });
    }

    /**
     * Register callback for the submit of particular forms.
     * <p>
     * With this api, you can get callback when the specific forms you configured in AIHelp dashboard are submitted.
     *
     * @param callback callback for form submitted
     */
    public static void setOnSpecificFormSubmittedCallback(OnSpecificFormSubmittedCallback callback) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().setOnSpecificFormSubmittedCallback(callback);
            }
        });
    }

    /**
     * Register callback for AIHelp's session visibility
     * @param callback callback when AIHelp session is visible
     */
    public static void setOnAIHelpSessionOpenCallback(OnAIHelpSessionOpenCallback callback) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().setOnAIHelpSessionOpenCallback(callback);
            }
        });
    }

    /**
     * Register callback for AIHelp's session visibility
     * @param callback callback when AIHelp session is closed by users
     */
    public static void setOnAIHelpSessionCloseCallback(OnAIHelpSessionCloseCallback callback) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().setOnAIHelpSessionCloseCallback(callback);
            }
        });
    }

    /**
     * Register callback for Operation module's data source update
     * @param callback callback when there is update in Operation articles,
     *                 adding new articles or updating existing ones are both included.
     */
    public static void setOnOperationUnreadChangedCallback(OnOperationUnreadChangedCallback callback) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().setOnOperationUnreadChangedCallback(callback);
            }
        });
    }

    /**
     * Register callback for specific url clicked by users.
     *
     * @param callback Whenever a url contains 'js-bridge=enable' is clicked
     */
    public static void setOnSpecificUrlClickedCallback(OnSpecificUrlClickedCallback callback) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().setOnSpecificUrlClickedCallback(callback);
            }
        });
    }

    /**
     * AIHelp provide additional support for some country or regions.
     *
     * @param countryOrRegion ISO country code, please check https://www.iso.org/iso-3166-country-codes.html to learn more.
     */
    public static void additionalSupportFor(PublishCountryOrRegion countryOrRegion) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().additionalSupportFor(countryOrRegion);
            }
        });
    }

    /**
     * Get current AIHelp SDK version
     *
     * @return AIHelp SDK version
     */
    public static String getSDKVersion() {
        return BuildConfig.SDK_VERSION;
    }

    /**
     * Whether AIHelp session is visible to users
     *
     * @return whether sdk is active
     */
    public static boolean isAIHelpShowing() {
        return Const.IS_SDK_SHOWING;
    }

    /**
     * Enable develop logs of AIHelp
     */
    public static void enableLogging(boolean enable) {
        AIHelpCore.getInstance().enableLogging(enable);
    }

}
