package net.aihelp.ui;

import android.annotation.SuppressLint;
import android.content.Intent;
import android.content.pm.ActivityInfo;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.graphics.Color;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.ColorDrawable;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.os.Bundle;
import android.text.TextUtils;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;
import android.webkit.JavascriptInterface;

import net.aihelp.BuildConfig;
import net.aihelp.common.ConfigValues;
import net.aihelp.common.Const;
import net.aihelp.common.CustomConfig;
import net.aihelp.core.ui.BaseActivity;
import net.aihelp.core.ui.image.Picasso;
import net.aihelp.core.ui.image.Target;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.core.util.concurrent.ApiExecutorFactory;
import net.aihelp.core.util.crash.LeakChecker;
import net.aihelp.data.event.OrientationChangeEvent;
import net.aihelp.data.localize.LocalizeHelper;
import net.aihelp.ui.helper.BitmapHelper;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.Styles;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentManager;
import androidx.fragment.app.FragmentTransaction;

public class HostActivity extends BaseActivity {

    FragmentManager fragmentManager;

    @Override
    public int getLayoutId() {
        return ResResolver.getLayoutId("aihelp_act_host");
    }

    @SuppressLint({"SourceLockedOrientationActivity"})
    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {

        LeakChecker.INSTANCE.watch(this);

        Picasso.setSingletonInstance(new Picasso.Builder(this).build());

        if (Const.sSessionOpenListener != null) {
            Const.sSessionOpenListener.onAIHelpSessionOpened();
        }

        Const.IS_SDK_SHOWING = true;
        getWindow().setFlags(WindowManager.LayoutParams.FLAG_HARDWARE_ACCELERATED,
                WindowManager.LayoutParams.FLAG_HARDWARE_ACCELERATED);

        switch (CustomConfig.CommonSetting.screenOrientation) {
            case ConfigValues.SCREEN_ORIENTATION_LANDSCAPE:
                CustomConfig.CommonSetting.isLandscape = true;
                setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_SENSOR_LANDSCAPE);
                break;
            case ConfigValues.SCREEN_ORIENTATION_PORTRAIT:
                CustomConfig.CommonSetting.isLandscape = false;
                setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_PORTRAIT);
                break;
            case ConfigValues.SCREEN_ORIENTATION_SENSOR:
                CustomConfig.CommonSetting.isLandscape = getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE;
                setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_UNSPECIFIED);
                break;
        }

        super.onCreate(savedInstanceState);

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            getWindow().clearFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_STATUS);
            getWindow().addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS);
            getWindow().setStatusBarColor(Styles.getColorWithAlpha(CustomConfig.CommonSetting.navigationBarBackground,
                    CustomConfig.CommonSetting.navigationBarAlpha));
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                getWindow().getDecorView().setSystemUiVisibility(
                        // 状态栏的颜色应该和文字颜色保持一致，如果文字颜色是浅色，那状态栏就应该是白色，反之亦然
                        Styles.isLightColor(Color.parseColor(CustomConfig.CommonSetting.textColor)) ?
                                View.SYSTEM_UI_FLAG_VISIBLE : View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR
                );
            }
        }

        if (CustomConfig.CommonSetting.isBackgroundRenderedWithImage) {
            updateWindowBackgroundImage();
        } else {
            getWindow().setBackgroundDrawable(new ColorDrawable(Styles.getColor(CustomConfig.CommonSetting.backgroundColorForAll)));
        }

    }

    @Override
    public void onConfigurationChanged(@NonNull Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        // 只有在确实开启了重力感应的前提下，才会发送全局通知；
        // 否则就会在横屏进竖屏或者竖屏进横屏的时候触发两次页面渲染，带来意料之外的问题
        if (CustomConfig.CommonSetting.screenOrientation == ConfigValues.SCREEN_ORIENTATION_SENSOR) {
            CustomConfig.CommonSetting.isLandscape = newConfig.orientation == Configuration.ORIENTATION_LANDSCAPE;
            EventBus.getDefault().post(new OrientationChangeEvent(newConfig.orientation));
            if (CustomConfig.CommonSetting.isBackgroundRenderedWithImage) {
                updateWindowBackgroundImage();
            }
        }
    }

    private void updateWindowBackgroundImage() {
        String resource;
        if (Styles.isLandscape()) {
            resource = CustomConfig.CommonSetting.backgroundImageForLandscape;
        } else {
            resource = CustomConfig.CommonSetting.backgroundImageForPortrait;
        }
        if (!TextUtils.isEmpty(resource)) {
            Picasso.get().load(resource).into(new Target() {
                @Override
                public void onBitmapLoaded(Bitmap bitmap, Picasso.LoadedFrom from) {
                    getWindow().setBackgroundDrawable(new BitmapDrawable(getResources(), bitmap));
                }

                @Override
                public void onBitmapFailed(Exception e, Drawable errorDrawable) {
                }

                @Override
                public void onPrepareLoad(Drawable placeHolderDrawable) {

                }
            });
        }
    }

    @Override
    public void initView() {
        fragmentManager = getSupportFragmentManager();
        Bundle bundle = getIntent().getExtras();
        if (bundle == null) bundle = new Bundle();
        FragmentTransaction fraTransaction = fragmentManager.beginTransaction();
        fraTransaction.add(ResResolver.getViewId("aihelp_root_container"), SupportFragment.getInstance(bundle));
        fraTransaction.commit();
    }

    @JavascriptInterface
    public void finishFormPage() {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                onBackPressed();
            }
        });
    }

    public void onBackPressed() {
        List<Fragment> fragments = fragmentManager.getFragments();

        for (Fragment fragment : fragments) {
            if (fragment != null && fragment.isVisible() && fragment instanceof SupportFragment) {
                if (((SupportFragment) fragment).onBackPressed()) {
                    FragmentManager childFragmentManager = fragment.getChildFragmentManager();
                    if (childFragmentManager.getBackStackEntryCount() > 0) {
                        childFragmentManager.popBackStack();
                        return;
                    }
                } else {
                    return;
                }
            }
        }

        super.onBackPressed();
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, @Nullable Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        Picasso.get().shutdown();
        Const.IS_SDK_SHOWING = false;
        if (Const.sSessionCloseListener != null) {
            Const.sSessionCloseListener.onAIHelpSessionClosed();
        }
        LeakChecker.INSTANCE.stopWatching();
    }
}
