package net.aihelp.utils;

import android.content.Context;
import android.content.res.ColorStateList;
import android.content.res.TypedArray;
import android.graphics.Color;
import android.graphics.PorterDuff;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.GradientDrawable;
import android.graphics.drawable.StateListDrawable;
import android.os.Build;
import android.text.TextUtils;
import android.view.View;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.TextView;


import net.aihelp.common.CustomConfig;
import net.aihelp.core.ui.image.Picasso;
import net.aihelp.data.model.faq.FaqContentEntity;

import java.util.Random;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import androidx.annotation.NonNull;
import androidx.annotation.RestrictTo;
import androidx.appcompat.content.res.AppCompatResources;
import androidx.core.graphics.drawable.DrawableCompat;
import androidx.core.view.ViewCompat;
import androidx.vectordrawable.graphics.drawable.VectorDrawableCompat;

@RestrictTo(RestrictTo.Scope.LIBRARY)
public class Styles {

    public static void reRenderTextView(TextView targetView, String text) {
        reRenderTextView(targetView, text, true);
    }

    public static void reRenderTextView(TextView targetView, String text, boolean visible) {
        reRenderTextView(targetView, text, Color.parseColor(CustomConfig.CommonSetting.textColor), visible, 15);
    }

    public static void reRenderTextView(TextView targetView, String text, float alpha) {
        reRenderTextView(targetView, text, getColorWithAlpha(CustomConfig.CommonSetting.textColor, alpha));
    }

    public static void reRenderTextView(TextView targetView, String text, int color) {
        reRenderTextView(targetView, text, color, true, 15);
    }

    public static void reRenderTextView(TextView targetView, String text, int color, boolean visible, int textSize) {
        if (targetView != null) {
            targetView.setVisibility(visible ? View.VISIBLE : View.GONE);
            targetView.setText(text);
            targetView.setTextColor(color);
            targetView.setTextSize(textSize);
            if (targetView instanceof EditText) {
                EditText editText = (EditText) targetView;
                editText.setText("");
                editText.setHint(text);
                editText.setHintTextColor(getColorWithAlpha(getHexColor(color), 0.35f));
            }
        }
    }

    public static void reRenderImageView(ImageView targetView, String svgResource, int color, boolean clickable) {
        try {
            int drawableId = ResResolver.getDrawableId(svgResource);
            if (targetView != null && drawableId != 0) {
                targetView.setImageDrawable(getClickableDrawable(targetView.getContext(), svgResource, color, clickable));
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
                    if (targetView.getContext().getResources().getConfiguration().getLayoutDirection() == View.LAYOUT_DIRECTION_RTL) {
                        targetView.setScaleX(-1); // mirror icon for right-to-left layout
                    }
                }
            }
        } catch (Exception e) {
            // ignored
        }
    }

    public static StateListDrawable getClickableDrawableForList() {
        int upperBackgroundColor = getColorWithAlpha(CustomConfig.CommonSetting.upperBackgroundColor,
                CustomConfig.CommonSetting.upperBackgroundAlpha);
        return getClickableDrawable(upperBackgroundColor, Color.TRANSPARENT, 0);
    }

    public static StateListDrawable getClickableDrawableForButton() {
        int buttonColor = Color.parseColor(CustomConfig.CommonSetting.interactElementTextColor);
        return getClickableDrawable(buttonColor, buttonColor, 8);
    }

    public static StateListDrawable getClickableDrawable(int anchorColor, int normalBgColor, int cornerRadius) {
        StateListDrawable drawable = new StateListDrawable();
        int pressedBgColor = getColorWithAlpha(isLightColor(anchorColor) ? Color.BLACK : Color.WHITE, 0.1);
        drawable.addState(new int[]{android.R.attr.state_pressed}, Styles.getDrawable(pressedBgColor, cornerRadius));
        drawable.addState(new int[]{-android.R.attr.state_pressed}, Styles.getDrawable(normalBgColor, cornerRadius));
        return drawable;
    }

    public static Drawable getClickableDrawable(Context context, String svgResource, int color, boolean clickable) {
        StateListDrawable drawable = new StateListDrawable();
        int drawableId = ResResolver.getDrawableId(svgResource);
        if (drawableId != 0) {
            VectorDrawableCompat normalDrawable = VectorDrawableCompat.create(context.getResources(), drawableId, null);
            if (normalDrawable != null) {
                DrawableCompat.setTint(DrawableCompat.wrap(normalDrawable).mutate(), Styles.getColorWithAlpha(color, 1f));
            }
            drawable.addState(new int[]{-android.R.attr.state_pressed}, normalDrawable);
            if (clickable) {
                VectorDrawableCompat pressedDrawable = VectorDrawableCompat.create(context.getResources(), drawableId, null);
                if (pressedDrawable != null) {
                    DrawableCompat.setTint(DrawableCompat.wrap(pressedDrawable).mutate(), Styles.getColorWithAlpha(color, 0.7f));
                }
                drawable.addState(new int[]{android.R.attr.state_pressed}, pressedDrawable);
            }
        }
        return drawable;
    }

    public static ColorStateList getClickableTextColor(String color) {
        return getClickableTextColor(Color.parseColor(color));
    }

    public static ColorStateList getClickableTextColor(int color) {
        int[][] states = new int[2][];
        states[0] = new int[]{android.R.attr.state_pressed};
        states[1] = new int[]{};
        int[] colors = new int[]{getColorWithAlpha(color, 0.8f), getColorWithAlpha(color, 1f)};
        return new ColorStateList(states, colors);
    }

    public static void loadIcon(ImageView targetView, String iconResource) {
        loadIcon(targetView, iconResource, true);
    }

    public static void loadIcon(ImageView targetView, String iconResource, boolean visible) {
        loadIcon(targetView, iconResource, visible, "aihelp_svg_ic_placeholder");
    }

    public static void loadIcon(ImageView targetView, String iconResource, boolean visible, String placeHolderRes) {
        try {
            if (targetView != null) {
                targetView.setVisibility(visible ? View.VISIBLE : View.GONE);
                if (!TextUtils.isEmpty(iconResource)) {
                    String adjustedUrl = DomainSupportHelper.getAdjustedUrl(iconResource);
                    Picasso.get().load(adjustedUrl).placeholder(ResResolver.getDrawableId(placeHolderRes)).into(targetView);
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
                        if (targetView.getContext().getResources().getConfiguration().getLayoutDirection() == View.LAYOUT_DIRECTION_RTL) {
                            targetView.setScaleX(-1); // mirror icon for right-to-left layout
                        }
                    }
                } else {
                    targetView.setImageDrawable(AppCompatResources.getDrawable(targetView.getContext(), ResResolver.getDrawableId(placeHolderRes)));
                }
            }
        } catch (Exception e) {
            // ignored
        }
    }

    public static void reRenderImageView(ImageView targetView, String svgResource, boolean clickable) {
        reRenderImageView(targetView, svgResource, Color.parseColor(CustomConfig.CommonSetting.interactElementTextColor), clickable);
    }

    public static void reRenderImageView(ImageView targetView, String svgResource) {
        reRenderImageView(targetView, svgResource, false);
    }

    public static void reRenderImageView(ImageView targetView, String svgResource, int color) {
        reRenderImageView(targetView, svgResource, color, false);
    }

    public static boolean isLandscape() {
        return CustomConfig.CommonSetting.isLandscape;
    }

    public static int getScreenWidth(Context context) {
        int widthPixels = context.getResources().getDisplayMetrics().widthPixels;
        int heightPixels = context.getResources().getDisplayMetrics().heightPixels;
        int screenWidth = Math.min(widthPixels, heightPixels);
        int screenHeight = Math.max(widthPixels, heightPixels);
        return isLandscape() ? screenHeight : screenWidth;
    }

    public static int getScreenHeight(Context context) {
        int widthPixels = context.getResources().getDisplayMetrics().widthPixels;
        int heightPixels = context.getResources().getDisplayMetrics().heightPixels;
        int screenWidth = Math.min(widthPixels, heightPixels);
        int screenHeight = Math.max(widthPixels, heightPixels);
        return isLandscape() ? screenWidth : screenHeight;
    }

    public static int getColorFromAttr(Context context, int attribute) {
        TypedArray typedArray = context.getTheme().obtainStyledAttributes(new int[]{attribute});
        int color = typedArray.getColor(0, -1);
        typedArray.recycle();
        return color;
    }

    public static int getColor(Context context, int color) {
        return context.getResources().getColor(color);
    }

    public static String getHexColor(Context context, int attribute) {
        return getHexColor(getColor(context, attribute));
    }

    public static String getHexColor(int color) {
        return String.format("#%06X", 16777215 & color);
    }

    public static boolean isLightColor(int color) {
        double darkness = 1 - (0.299 * Color.red(color) + 0.587 * Color.green(color) + 0.114 * Color.blue(color)) / 255;
        return darkness < 0.5;
    }

    public static int getColor(String colorString) {
        return getColorWithAlpha(colorString, 1.0f);
    }

    public static int getColorWithAlpha(int color, double alpha) {
        try {
            float red = ((color & 0xff0000) >> 16) / 255.0f;
            float green = ((color & 0x00ff00) >> 8) / 255.0f;
            float blue = ((color & 0x0000ff)) / 255.0f;
            return argb(alpha, red, green, blue);
        } catch (Exception e) {
            return Color.WHITE;
        }
    }

    public static int getColorWithAlpha(String colorString, double alpha) {
        return getColorWithAlpha(Color.parseColor(colorString), alpha);
    }

    public static int[] getColorRGB(String colorString) {
        int color = Color.parseColor(colorString);
        int red = ((color & 0xff0000) >> 16);
        int green = ((color & 0x00ff00) >> 8);
        int blue = ((color & 0x0000ff));
        return new int[]{red, green, blue};
    }

    public static int argb(double alpha, float red, float green, float blue) {
        return ((int) (alpha * 255.0f + 0.5f) << 24) |
                ((int) (red * 255.0f + 0.5f) << 16) |
                ((int) (green * 255.0f + 0.5f) << 8) |
                (int) (blue * 255.0f + 0.5f);
    }

    public static void setColorFilter(Context context, Drawable drawable, int attr) {
        if (drawable != null) {
            drawable.setColorFilter(getColor(context, attr), PorterDuff.Mode.SRC_ATOP);
        }
    }

    public static void setColorFilter(Drawable drawable, int color) {
        if (drawable != null) {
            drawable.setColorFilter(color, PorterDuff.Mode.SRC_ATOP);
        }
    }

    public static int dpToPx(Context context, float dp) {
        if (context == null) return 0;
        return (int) (dp * context.getResources().getDisplayMetrics().density);
    }

    public static int px2dp(Context context, float pxVal) {
        float scale = context.getResources().getDisplayMetrics().density;
        return (int) (pxVal / scale + 0.5f);
    }


    public static StateListDrawable makeSelector(Context context) {
        StateListDrawable res = new StateListDrawable();

        VectorDrawableCompat checkedDrawable = VectorDrawableCompat.create(context.getResources(), ResResolver.getDrawableId("aihelp_svg_ic_bill_checked"), null);
        if (checkedDrawable != null) {
            DrawableCompat.setTint(DrawableCompat.wrap(checkedDrawable).mutate(), getColor(CustomConfig.CommonSetting.interactElementTextColor));
        }

        VectorDrawableCompat uncheckedDrawable = VectorDrawableCompat.create(context.getResources(), ResResolver.getDrawableId("aihelp_svg_ic_bill_unchecked"), null);
        if (uncheckedDrawable != null) {
            DrawableCompat.setTint(DrawableCompat.wrap(uncheckedDrawable).mutate(), getColorWithAlpha(CustomConfig.CommonSetting.textColor, 0.3));
        }

        res.addState(new int[]{android.R.attr.state_checked}, checkedDrawable);
        res.addState(new int[]{}, uncheckedDrawable);
        return res;
    }

    public static Drawable getDrawable(int color, int cornerRadius) {
        GradientDrawable drawable = new GradientDrawable();
        drawable.setCornerRadius(cornerRadius);//设置4个角的弧度
        drawable.setColor(color);// 设置颜色
        return drawable;
    }

    public static Drawable getDrawableWithCorner(int color, int topLeft, int topRight, int bottomRight, int bottomLeft) {
        GradientDrawable drawable = new GradientDrawable();
        drawable.setCornerRadii(new float[]{topLeft, topLeft, topRight, topRight, bottomRight, bottomRight, bottomLeft, bottomLeft});
        drawable.setColor(color);// 设置颜色
        return drawable;
    }

    public static void setGradientBackground(@NonNull View view, int startColor, int endColor, GradientDrawable.Orientation orientation) {
        int[] colors = new int[]{startColor, endColor};
        GradientDrawable gradientDrawable = new GradientDrawable(orientation, colors);
        ViewCompat.setBackground(view, gradientDrawable);
    }

    public static FaqContentEntity getFAQWithHighlightedSearchTerms(Context context, FaqContentEntity faq, String searchTerms) {
        FaqContentEntity newQuestion = null;
        if (!TextUtils.isEmpty(searchTerms)) {
            String titleText = faq.getFaqTitle();
            String bodyText = faq.getFaqContent();
            String hexColor = CustomConfig.CommonSetting.highlightedColor;
            String bodyTextCopy;
            String word;

            bodyText = ">" + bodyText + "<";
            titleText = ">" + titleText + "<";
            Pattern pattern = Pattern.compile(">[^<]+<");

            String titleTextCopy = titleText;

            Matcher matcher;
            String content;
            for (matcher = pattern.matcher(titleText); matcher.find(); titleText = titleText.replace(bodyTextCopy, content)) {
                bodyTextCopy = titleTextCopy.substring(matcher.start(), matcher.end());
                content = bodyTextCopy.replaceAll("(?i)(" + searchTerms + ")", "<span style=\"color: " + hexColor + "\"></span>");
            }

            bodyTextCopy = bodyText;

            for (matcher = pattern.matcher(bodyText); matcher.find(); bodyText = bodyText.replace(content, word)) {
                content = bodyTextCopy.substring(matcher.start(), matcher.end());
                word = content.replaceAll("(?i)(" + searchTerms + ")", "<span style=\"color: " + hexColor + "\">$1</span>");
            }

            titleText = titleText.substring(1, titleText.length() - 1);
            bodyText = bodyText.substring(1, bodyText.length() - 1);
            newQuestion = new FaqContentEntity(faq.getSecId(), titleText, faq.getFaqKeywords(), faq.getFaqMainId(),
                    faq.getFaqDisplayId(), faq.getFaqContentId(), bodyText, faq.isHelpful(), faq.getSearchTerm());
        }

        return newQuestion;
    }

    public static String getNoTemplateFaqContent(String originFaqContent) {
        StringBuilder stringBuilder = new StringBuilder();
        originFaqContent = ">" + originFaqContent + "<";
        Matcher matcher = Pattern.compile(">[^<]+").matcher(originFaqContent);
        while (matcher.find()) {
            String group = matcher.toMatchResult().group().replace(">", "").replace("<", "");
            if (group.trim().contains("body {") || TextUtils.isEmpty(group.trim())) continue;
            stringBuilder.append(group);
        }

        if (TextUtils.isEmpty(stringBuilder.toString())) {
            stringBuilder.append(originFaqContent);
        }
        return stringBuilder.toString();
    }

    public static int getRandomColor() {
        Random random = new Random();

        int r = 0;
        int g = 0;
        int b = 0;
        for (int i = 0; i < 2; i++) {
            int temp = random.nextInt(16);
            r = r * 16 + temp;
            temp = random.nextInt(16);
            g = g * 16 + temp;
            temp = random.nextInt(16);
            b = b * 16 + temp;
        }
        return Color.rgb(r, g, b);
    }

}

