package net.aihelp.core.net.check;

import android.os.AsyncTask;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Locale;

public final class Ping implements Task {

    private final String address;
    private final int count;
    private final int size;
    private final Callback complete;
    private volatile boolean stopped;
    private int interval;

    private Ping(String address, int count, Callback complete) {
        this(address, count, 56, 200, complete);
    }

    private Ping(String address, int count, int size,
                 int interval, Callback complete) {
        this.address = address;
        this.count = count;
        this.size = size;
        this.interval = interval;
        this.complete = complete;
        this.stopped = false;
    }

    public static Ping.Result startSync(String address) {
        return startSync(address, 1);
    }

    public static Ping.Result startSync(String address, int count) {
        final Ping p = new Ping(address, count, null);
        return p.pingCmd();
    }

    public static Task start(String address, Callback complete) {
        return start(address, 5, complete);
    }

    public static Task start(String address, int count, Callback complete) {
        final Ping p = new Ping(address, count, complete);
        AsyncTask.execute(new Runnable() {
            @Override
            public void run() {
                p.run();
            }
        });
        return p;
    }

    private static String getIp(String host) throws UnknownHostException {
        InetAddress i = InetAddress.getByName(host);
        return i.getHostAddress();
    }

    private void run() {
        Result r = pingCmd();
        if (complete != null) {
            complete.complete(r);
        }
    }

    private Result pingCmd() {
        String ip;
        try {
            ip = getIp(address);
        } catch (UnknownHostException e) {
            return new Result(e.toString(), "", 0, 0);
        }
        boolean isIpv6 = false;
        try {
            InetAddress address = InetAddress.getByName(ip);
            if (address instanceof java.net.Inet6Address) {
                // It's ipv6
                isIpv6 = true;
            } else if (address instanceof java.net.Inet4Address) {
                // It's ipv4
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        String strFmt = "ping -n -i %f -s %d -c %d %s";
        if (isIpv6) {
            strFmt = "ping6 -n -i %f -s %d -c %d %s";
        }
        String cmd = String.format(Locale.getDefault(), strFmt, ((double) interval / 1000), size, count, ip);
        Process process = null;
        StringBuilder str = new StringBuilder();
        BufferedReader reader = null;
        BufferedReader errorReader = null;
        try {
            process = Runtime.getRuntime().exec(cmd);
            reader = new BufferedReader(new InputStreamReader(
                    process.getInputStream()));
            String line;
            errorReader = new BufferedReader(new InputStreamReader(
                    process.getErrorStream()));

            if ("".equals(str.toString())) {
                str.append("Ping host address is ")
                        .append(address).append(".").append("\n");
            }

            while ((line = reader.readLine()) != null) {
                str.append(line).append("\n");
            }
            while ((line = errorReader.readLine()) != null) {
                str.append(line);
            }
            reader.close();
            errorReader.close();
            process.waitFor();

        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                if (reader != null) {
                    reader.close();
                }
                if (process != null) {
                    process.destroy();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return new Result(str.toString(), ip, size, interval);
    }

    @Override
    public void stop() {
        stopped = true;
    }

    public interface Callback {
        void complete(Result r);
    }

    public static class Result {

        public final String result;
        public final String ip;
        public final int size;
        public final int interval;
        private final String lastLinePrefix = "rtt min/avg/max/mdev = ";
        private final String packetWords = " packets transmitted";
        private final String receivedWords = " received";
        public int sent;
        public int dropped;
        public float max;
        public float min;
        public float avg;
        public float stddev;
        public int count;
        public boolean isSuccess;

        Result(String result, String ip, int size, int interval) {
            this.result = result;
            this.ip = ip;
            this.size = size;
            this.interval = interval;
            parseResult();
        }

        static String trimNoneDigital(String s) {
            if (s == null || s.length() == 0) {
                return "";
            }
            char[] v = s.toCharArray();
            char[] v2 = new char[v.length];
            int j = 0;
            for (char aV : v) {
                if ((aV >= '0' && aV <= '9') || aV == '.') {
                    v2[j++] = aV;
                }
            }
            return new String(v2, 0, j);
        }

        private void parseRttLine(String s) {
            String s2 = s.substring(lastLinePrefix.length(), s.length() - 3);
            String[] l = s2.split("/");
            if (l.length != 4) {
                return;
            }
            min = Float.parseFloat(trimNoneDigital(l[0]));
            avg = Float.parseFloat(trimNoneDigital(l[1]));
            max = Float.parseFloat(trimNoneDigital(l[2]));
            stddev = Float.parseFloat(trimNoneDigital(l[3]));
        }

        private void parsePacketLine(String s) {
            String[] l = s.split(",");
            if (l.length != 4) {
                return;
            }
            if (l[0].length() > packetWords.length()) {
                String s2 = l[0].substring(0, l[0].length() - packetWords.length());
                count = Integer.parseInt(s2);
            }
            if (l[1].length() > receivedWords.length()) {
                String s3 = l[1].substring(0, l[1].length() - receivedWords.length());
                sent = Integer.parseInt(s3.trim());
            }
            dropped = count - sent;
        }

        private void parseResult() {
            String[] rs = result.split("\n");
            try {
                for (String s : rs) {
                    if (s.contains(packetWords)) {
                        parsePacketLine(s);
                    } else if (s.contains(lastLinePrefix)) {
                        parseRttLine(s);
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
            this.isSuccess = ip != null && ip.length() > 0 && sent > 0 && avg > 0;
        }
    }
}
