package net.aihelp.core.net.http;

import android.os.Looper;
import android.text.TextUtils;

import net.aihelp.common.API;
import net.aihelp.core.net.http.callback.AIHelpCallback;
import net.aihelp.core.net.http.callback.BaseCallback;
import net.aihelp.core.net.http.callback.DownloadCallback;
import net.aihelp.core.net.http.callback.UploadCallback;
import net.aihelp.core.net.http.config.HttpConfig;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.core.util.concurrent.ApiExecutor;
import net.aihelp.core.util.concurrent.ApiExecutorFactory;
import net.aihelp.data.localize.util.LocalizeUtil;
import net.aihelp.utils.TLog;

import org.json.JSONObject;

import java.io.File;
import java.net.URLEncoder;
import java.util.Iterator;
import java.util.regex.Pattern;

import okhttp3.Call;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;

public class AIHelpRequest {

    private static final String ENCODE = "utf-8";

    private final OkHttpClient mOkHttpClient;
    private final OkHttpClient mOkHttpUploadClient;

    private AIHelpRequest() {
        mOkHttpClient = HttpConfig.getOkHttpClient(true);
        mOkHttpUploadClient = HttpConfig.getOkHttpClient(false);
    }

    public static AIHelpRequest getInstance() {
        return Holder.INSTANCE;
    }

    private static class Holder {
        private static final AIHelpRequest INSTANCE = new AIHelpRequest();
    }

    public <T> void requestGetByAsync(String actionUrl, final BaseCallback<T> callBack) {
        requestGetByAsync(actionUrl, null, callBack);
    }

    public <T> void requestGetByAsync(String actionUrl, JSONObject jsonObject, final BaseCallback<T> callBack) {
        if (!TextUtils.isEmpty(actionUrl)) {
            if (!actionUrl.contains("//")) {
                actionUrl = API.REQUEST_SCHEME + API.HOST_URL + actionUrl;
            }
            StringBuilder tempParams = new StringBuilder();
            try {
                int pos = 0;
                if (jsonObject != null) {
                    Iterator<String> keys = jsonObject.keys();
                    while (keys.hasNext()) {
                        String key = keys.next();
                        if (pos > 0) {
                            tempParams.append("&");
                        }
                        tempParams.append(String.format("%s=%s", key, URLEncoder.encode(jsonObject.optString(key), "utf-8")));
                        pos++;
                    }
                }
                if (!TextUtils.isEmpty(tempParams)) {
                    actionUrl = String.format("%s?%s", actionUrl, tempParams.toString());
                }
                final Request request = new Request.Builder().url(actionUrl).build();
                Call call = mOkHttpClient.newCall(request);
                onRequest(callBack, call);
            } catch (Exception e) {
                TLog.d("AIHelpRequest requestGetByAsync catch Exception: " + e.toString());
                failedCallBack(actionUrl, e.getMessage(), callBack);
            }
        }
    }

    public <T> Call requestPostByJson(String actionUrl, String jsonString, final BaseCallback<T> callBack) {
        if (!TextUtils.isEmpty(actionUrl)) {
            try {
                if (!actionUrl.contains("//")) {
                    actionUrl = API.REQUEST_SCHEME + API.HOST_URL + actionUrl;
                }
                RequestBody body = RequestBody.create(HttpConfig.MEDIA_TYPE_JSON, jsonString);
                final Request request = new Request.Builder().url(actionUrl).post(body).build();
                Call call = mOkHttpClient.newCall(request);
                return onRequest(callBack, call);
            } catch (Exception e) {
                TLog.d("AIHelpRequest requestPostByAsync catch Exception: " + e.toString());
                failedCallBack(actionUrl, e.getMessage(), callBack);
            }
        }
        return null;
    }

    public <T> Call requestPostByJson(String actionUrl, JSONObject json, final BaseCallback<T> callBack) {
        if (json == null) json = new JSONObject();
        return requestPostByJson(actionUrl, json.toString(), callBack);
    }

    public <T> Call requestUpLoadFile(String actionUrl, File targetFile, final UploadCallback callBack) {
        if (!actionUrl.contains("//")) {
            actionUrl = API.REQUEST_SCHEME + API.HOST_URL + actionUrl;
        }
        final Request request = HttpConfig.getUploadRequest(actionUrl, targetFile);
        if (request == null) {
            failedCallBack(actionUrl, "", callBack);
            return null;
        }
        Call call = mOkHttpUploadClient.newCall(request);
        return onRequest(callBack, call);
    }

    public <T> void requestDownloadFile(int mode, final BaseCallback<T> callBack) {
        String url = LocalizeUtil.getUrl(mode);
        if (!TextUtils.isEmpty(url) && LocalizeUtil.isFallbackUrl(mode, url)) {
            failedCallBack(url, "The cdn file is not working, requesting data via API.", callBack);
        } else {
            File file = new File(LocalizeUtil.getFileLocation(mode));
            if (file.exists() && file.length() > 0) {
                successCallBack(callBack);
            } else {
                if (!TextUtils.isEmpty(url)) {
                    TLog.d(String.format("LocalizeUrl: %s", url));
                    if (Pattern.compile(".+\\.(json|aiml)").matcher(url).matches()) {
                        Request request = new Request.Builder().url(url).build();
                        Call call = mOkHttpClient.newCall(request);
                        call.enqueue(new DownloadCallback<>(callBack, LocalizeUtil.getFileLocation(mode)));
                    }
                } else {
                    failedCallBack(url, "bad request for mode: " + mode, callBack);
                }
            }
        }
    }

    public void requestDownloadFile(String url, String path, final BaseCallback callBack) {
        File file = new File(path);
        if (file.exists() && file.length() > 0) {
            progressCallback(callBack, file.length(), file.length());
            successCallBack(callBack);
        } else {
            Request request = new Request.Builder().url(url).build();
            Call call = mOkHttpClient.newCall(request);
            call.enqueue(new DownloadCallback<>(callBack, path));
        }
    }

    private <T> Call onRequest(final BaseCallback<T> callBack, Call call) {
        call.enqueue(new AIHelpCallback<>(callBack));
        return call;
    }

    private <T> void progressCallback(final BaseCallback<T> callBack, long totalContentLength, long currentTransferred) {
        if (callBack == null) return;
        int progress = (int) ((currentTransferred * 1.0f / totalContentLength) * 100);
        ApiExecutor apiExecutor = ApiExecutorFactory.getHandlerExecutor();
        if (Looper.getMainLooper().getThread() == Thread.currentThread()) {
            apiExecutor.runAsync(new Runnable() {
                @Override
                public void run() {
                    callBack.onAsyncReqProgress(totalContentLength, currentTransferred, progress);
                }
            });
            callBack.onReqProgress(totalContentLength, currentTransferred, progress);
        } else {
            callBack.onAsyncReqProgress(totalContentLength, currentTransferred, progress);
            apiExecutor.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    callBack.onReqProgress(totalContentLength, currentTransferred, progress);
                }
            });
        }
    }

    private <T> void successCallBack(final BaseCallback<T> callBack) {
        if (callBack == null) return;
        ApiExecutor apiExecutor = ApiExecutorFactory.getHandlerExecutor();
        if (Looper.getMainLooper().getThread() == Thread.currentThread()) {
            apiExecutor.runAsync(new Runnable() {
                @Override
                public void run() {
                    callBack.onAsyncReqSuccess(null);
                }
            });
            callBack.onReqSuccess(null);
        } else {
            callBack.onAsyncReqSuccess(null);
            apiExecutor.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    callBack.onReqSuccess(null);
                }
            });
        }
    }

    private <T> void failedCallBack(final String url, final String errorMsg, final BaseCallback<T> callBack) {
        if (callBack == null) return;
        ApiExecutor apiExecutor = ApiExecutorFactory.getHandlerExecutor();
        if (Looper.getMainLooper().getThread() == Thread.currentThread()) {
            apiExecutor.runAsync(new Runnable() {
                @Override
                public void run() {
                    callBack.onAsyncFailure(url, -1, errorMsg);
                }
            });
            callBack.onFailure(url, -1, errorMsg);
        } else {
            callBack.onAsyncFailure(url, -1, errorMsg);
            apiExecutor.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    callBack.onFailure(url, -1, errorMsg);
                }
            });
        }
    }

}
