package net.aihelp.core.net.http.callback;

import android.text.TextUtils;

import net.aihelp.core.util.concurrent.ApiExecutor;
import net.aihelp.core.util.concurrent.ApiExecutorFactory;
import net.aihelp.core.util.crash.logger.AIHelpCrashLogger;
import net.aihelp.utils.TLog;

import java.io.Closeable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;

import androidx.annotation.NonNull;
import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.Response;

public class DownloadCallback<T> implements Callback {

    private static final String TAG = "RequestManager";

    private final BaseCallback<T> reqCallBack;
    private final String targetPath;

    public DownloadCallback(BaseCallback<T> reqCallBack, String targetPath) {
        this.reqCallBack = reqCallBack;
        this.targetPath = targetPath;
    }

    @Override
    public void onFailure(@NonNull Call call, @NonNull IOException e) {
        TLog.d(TAG, "DownloadCallback onFailure: " + e.toString());
        String requestUrl = call.request().url().toString();
        failedCallBack(requestUrl, e.getMessage(), reqCallBack);
        AIHelpCrashLogger.error(requestUrl, e);
    }

    @Override
    public void onResponse(@NonNull Call call, @NonNull Response response) {

        String requestUrl = call.request().url().toString();

        if (TextUtils.isEmpty(targetPath)) return;

        if (response.body() == null || !response.isSuccessful()) {
            failedCallBack(requestUrl, "response.isSuccessful() returning false", reqCallBack);
            return;
        }

        long total = response.body().contentLength();
        long curr = 0;

        File file = new File(targetPath);
        if (file.exists() && file.length() > 0) {
            progressCallback(reqCallBack, total, total);
            successCallBack(reqCallBack);
            return;
        }

        InputStream input = null;
        FileOutputStream output = null;

        try {
            input = response.body().byteStream();
            output = new FileOutputStream(file);
            byte[] buffer = new byte[8192];

            int bytesRead;
            while ((bytesRead = input.read(buffer, 0, buffer.length)) >= 0) {
                output.write(buffer, 0, bytesRead);
                curr += bytesRead;
                progressCallback(reqCallBack, total, curr);
            }

            successCallBack(reqCallBack);

        } catch (Throwable e) {
            TLog.d(TAG, "DownloadCallback onResponse catch Exception: " + e.toString());
            failedCallBack(requestUrl, e.getMessage(), reqCallBack);
            AIHelpCrashLogger.error(requestUrl, e);
        } finally {
            closeQuietly(input);
            closeQuietly(output);
        }
    }

    private void successCallBack(final BaseCallback<T> callBack) {
        if (callBack == null) return;
        callBack.onAsyncReqSuccess(null);
        ApiExecutor apiExecutor = ApiExecutorFactory.getHandlerExecutor();
        apiExecutor.runOnUiThread(new Runnable() {
            @Override
            public void run() {
                callBack.onReqSuccess(null);
            }
        });
    }

    private void failedCallBack(final String url, final String errorMsg, final BaseCallback<T> callBack) {
        if (callBack == null) return;
        callBack.onAsyncFailure(url, -1, errorMsg);
        ApiExecutor apiExecutor = ApiExecutorFactory.getHandlerExecutor();
        apiExecutor.runOnUiThread(new Runnable() {
            @Override
            public void run() {
                callBack.onFailure(url, -1, errorMsg);
            }
        });
    }

    private void progressCallback(final BaseCallback<T> callBack, long totalContentLength, long currentTransferred) {
        if (callBack == null) return;
        int progress = (int) ((currentTransferred * 1.0f / totalContentLength) * 100);
        callBack.onAsyncReqProgress(totalContentLength, currentTransferred, progress);
        ApiExecutor apiExecutor = ApiExecutorFactory.getHandlerExecutor();
        apiExecutor.runOnUiThread(new Runnable() {
            @Override
            public void run() {
                callBack.onReqProgress(totalContentLength, currentTransferred, progress);
            }
        });
    }

    private void closeQuietly(Closeable stream) {
        try {
            if (stream != null) {
                stream.close();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

}
