package net.aihelp.core.net.json;

import android.text.TextUtils;

import net.aihelp.core.util.crash.logger.AIHelpCrashLogger;

import org.json.JSONArray;
import org.json.JSONObject;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

public class JsonHelper {

    public static <T> T toJavaObject(String jsonString, Type type) {
        try {
            String clazzName = type.toString().split(" ")[1].trim();
            Class<?> typeClass = Class.forName(clazzName);
            return (T) toJavaObject(jsonString, typeClass);
        } catch (Throwable e) {
            e.printStackTrace();
            AIHelpCrashLogger.error("JsonHelper toJavaObject#30", e);
        }
        return null;
    }

    public static <T> T toJavaObject(String jsonString, Class<T> clazz) {
        if (TextUtils.isEmpty(jsonString)) return null;
        if (clazz == String.class || clazz == Integer.class || clazz == Double.class || clazz == Float.class) {
            return (T) jsonString;
        }

        try {
            T returnObject = clazz.newInstance();
            JSONObject jsonObject = new JSONObject(jsonString);
            for (Field field : clazz.getDeclaredFields()) {
                field.setAccessible(true);

                Object optObject = jsonObject.opt(field.getName());

                if (optObject instanceof JSONObject) {
                    if ("JSONObject".equals(field.getType().getSimpleName())) {
                        field.set(returnObject, optObject);
                    } else {
                        field.set(returnObject, toJavaObject(optObject.toString(), field.getType()));
                    }
                } else if (optObject instanceof JSONArray && "List".equals(field.getType().getSimpleName())) {
                    GenericType genericType = field.getAnnotation(GenericType.class);
                    if (genericType != null) {
                        field.set(returnObject, toJavaList(optObject.toString(), genericType.value()));
                    }
                } else {
                    if ("String".equals(field.getType().getSimpleName())) {
                        field.set(returnObject, optObject != null && optObject != JSONObject.NULL ? String.valueOf(optObject) : "");
                    } else if (optObject != null && optObject != JSONObject.NULL) {
                        field.set(returnObject, optObject);
                    }
                }
            }
            return returnObject;
        } catch (Exception e) {
            e.printStackTrace();
            AIHelpCrashLogger.error("JsonHelper toJavaObject#66", e);
        }
        return null;
    }

    public static <T> List<T> toJavaList(String jsonString, Class<T> clazz) {
        ArrayList<T> javaList = new ArrayList<>();
        if (clazz != null) {
            try {
                JSONArray jsonArray = new JSONArray(jsonString);
                for (int i = 0; i < jsonArray.length(); i++) {
                    Object arrObject = jsonArray.opt(i);
                    if (arrObject != null) {
                        if (arrObject instanceof JSONObject) {
                            javaList.add(toJavaObject(arrObject.toString(), clazz));
                        } else if (arrObject.getClass() == clazz) {
                            javaList.add((T) arrObject);
                        }
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
                AIHelpCrashLogger.error("JsonHelper toJavaList#82", e);
            }
        }
        return javaList;
    }


    public static JSONObject getJsonObject(JSONArray jsonArray, int index) {
        if (jsonArray == null) return new JSONObject();
        JSONObject jsonObject = jsonArray.optJSONObject(index);
        return jsonObject == null ? new JSONObject() : jsonObject;
    }

    public static JSONObject getJsonObject(JSONObject object, String key) {
        if (object == null) return new JSONObject();
        JSONObject jsonObject = object.optJSONObject(key);
        return jsonObject == null ? new JSONObject() : jsonObject;
    }

    public static JSONArray getJsonArray() {
        return new JSONArray();
    }

    public static JSONArray getJsonArray(JSONObject object, String key) {
        if (object == null) return new JSONArray();
        JSONArray jsonArray = object.optJSONArray(key);
        return jsonArray == null ? new JSONArray() : jsonArray;
    }

    public static String optString(JSONObject object, String key) {
        if (object == null) return "";
        String s = object.optString(key);
        return (TextUtils.isEmpty(s) || s.equals(JSONObject.NULL.toString())) ? "" : s;
    }

    public static JSONArray wrap(Collection<? extends Jsonable> collection) {
        JSONArray jsonArray = new JSONArray();
        if (collection != null) {
            for (Jsonable o : collection) {
                if (o != null) {
                    jsonArray.put(o.toJsonObject());
                }
            }
        }
        return jsonArray;
    }

    public static JSONObject getJsonObject() {
        return new JSONObject();
    }

    public static JSONObject getJsonObject(String jsonString) {
        if (!TextUtils.isEmpty(jsonString)) {
            try {
                return new JSONObject(jsonString);
            } catch (Exception e) {
                // ignored
            }
        }
        return new JSONObject();
    }

    public static JSONObject shallowCopy(JSONObject from) {
        if (from != null) {
            try {
                JSONObject to = new JSONObject();
                Iterator<String> keys = from.keys();
                while (keys.hasNext()) {
                    String key = keys.next();
                    Object value = from.opt(key);
                    put(to, key, value);
                }
                return to;
            } catch (Exception e) {
                // ignored
                return from;
            }
        }
        return new JSONObject();
    }

    public static JSONArray getJsonArray(String jsonString) {
        if (!TextUtils.isEmpty(jsonString)) {
            try {
                return new JSONArray(jsonString);
            } catch (Exception e) {
                // ignored
            }
        }
        return new JSONArray();
    }

    public static <T> T opt(JSONObject object, String key, T fallback) {
        if (object == null) return fallback;
        try {
            String methodName = String.format("opt%s", fallback.getClass().getSimpleName());
            if ("optString".equals(methodName)) {
                return (T) optString(object, key);
            }
            Method method = object.getClass().getDeclaredMethod(methodName, String.class, fallback.getClass());
            method.setAccessible(true);
            return (T) method.invoke(object, key, fallback);
        } catch (Exception e) {
            try {
                Field typeField = fallback.getClass().getDeclaredField("TYPE");
                typeField.setAccessible(true);
                Object o = typeField.get(null);
                if (o instanceof Class) {
                    String methodName = String.format("opt%s", captureName(((Class<?>) o).getSimpleName()));
                    Method method = object.getClass().getDeclaredMethod(methodName, String.class, (Class<?>) o);
                    method.setAccessible(true);
                    return (T) method.invoke(object, key, fallback);
                }
            } catch (Exception ignored) {
                return fallback;
            }
            return fallback;
        }
    }

    public static String captureName(String name) {
        if (!TextUtils.isEmpty(name)) {
            char[] cs = name.toCharArray();
            cs[0] -= 32; // ASCII 偏移 32 位即可偏移到大写字母
            return String.valueOf(cs);
        }
        return "";
    }

    public static boolean hasKey(JSONObject object, String key) {
        if (object != null && object.has(key)) {
            return !TextUtils.isEmpty(object.optString(key));
        }
        return false;
    }

    public static void put(JSONObject object, String key, Object value) {
        try {
            if (object != null) {
                object.put(key, value);
            }
        } catch (Exception e) {
            // ignored
        }
    }

}
