package net.aihelp.core.ui;

import android.app.Activity;
import android.content.Context;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.animation.AlphaAnimation;
import android.view.animation.Animation;

import net.aihelp.common.Const;
import net.aihelp.common.IntentValues;
import net.aihelp.config.AIHelpContext;
import net.aihelp.core.mvp.IPresenter;
import net.aihelp.core.mvp.IView;
import net.aihelp.core.net.monitor.NetworkState;
import net.aihelp.core.ui.loading.helper.VaryViewHelperController;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.core.util.bus.Subscribe;
import net.aihelp.core.util.bus.ThreadMode;
import net.aihelp.core.util.bus.event.EventCenter;
import net.aihelp.core.util.concurrent.ApiExecutorFactory;
import net.aihelp.data.event.TemplateReadyEvent;
import net.aihelp.data.track.resource.tracker.TemplateResTracker;
import net.aihelp.ui.SupportFragment;
import net.aihelp.ui.faq.IFaqEventListener;
import net.aihelp.ui.faq.IFaqParentView;
import net.aihelp.ui.helper.FragmentHelper;
import net.aihelp.ui.wrapper.FaqEventListenerWrapper;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.TLog;
import net.aihelp.utils.ToastUtil;

import java.lang.ref.WeakReference;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentManager;

public abstract class BaseFragment<P extends IPresenter> extends Fragment implements IView {

    private boolean isInflateFinished;
    protected View mInflateView;
    private static boolean shouldRetainChildFragmentManager;
    private FragmentManager retainedChildFragmentManager;

    protected P mPresenter;
    protected MyHandler mHandler = new MyHandler(this);

    protected VaryViewHelperController mVaryViewHelperController = null;

    protected int dip2px(Context context, double dpValue) {
        if (context == null) return (int) dpValue;
        float density = context.getResources().getDisplayMetrics().density;
        return (int) (dpValue * density + 0.5);
    }

    protected IFaqEventListener getFaqFlowListener() {
        SupportFragment supportFragment = FragmentHelper.getSupportFragment(this);
        if (supportFragment != null) {
            return ((IFaqParentView) supportFragment).getFaqEventListener();
        }
        return new FaqEventListenerWrapper();
    }

    public FragmentManager getRetainedChildFragmentManager() {
        if (shouldRetainChildFragmentManager) {
            if (retainedChildFragmentManager != null) {
                retainedChildFragmentManager = getChildFragmentManager();
            }
            return retainedChildFragmentManager;
        } else {
            return getChildFragmentManager();
        }
    }

    public Context getContext() {
        Context context = super.getContext();
        return context != null ? context : AIHelpContext.getInstance().getContext();
    }

    private void initPresenter() {
        Type type = getClass().getGenericSuperclass();
        if (type instanceof ParameterizedType) {
            Type[] types = ((ParameterizedType) type).getActualTypeArguments();
            Class<P> presenterClass = (Class<P>) types[0];

            if (presenterClass == IPresenter.class) {
                return;
            }

            try {
                Constructor<P> constructor = presenterClass.getDeclaredConstructor(Context.class);
                mPresenter = constructor.newInstance(getContext());
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        if (mPresenter != null) {
            mPresenter.attachView(this);
        }
    }

    public void onAttach(Context context) {
        Const.isNestedFragmentOnResume = false;
        super.onAttach(AIHelpContext.getLocaleUpdatedContext(context, Const.CORRECT_LANGUAGE));

        if (isBindEventBus()) {
            EventBus.getDefault().register(this);
        }

        try {
            setRetainInstance(true);
        } catch (Exception e) {
            shouldRetainChildFragmentManager = true;
        }

        if (shouldRetainChildFragmentManager && retainedChildFragmentManager != null) {
            try {
                Field childFMField = Fragment.class.getDeclaredField("mChildFragmentManager");
                childFMField.setAccessible(true);
                childFMField.set(this, retainedChildFragmentManager);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

    }

    @Override
    public void onResume() {
        super.onResume();
        if (mInflateView != null) {
            mInflateView.postDelayed(new Runnable() {
                @Override
                public void run() {
                    Const.isNestedFragmentOnResume = true;
                }
            }, 300);
        }
    }

    public Animation onCreateAnimation(int transit, boolean enter, int nextAnim) {
        if (!enter && !this.isRemoving()) {
            Animation doNothingAnimation = new AlphaAnimation(1.0F, 1.0F);
            doNothingAnimation.setDuration(200);
            return doNothingAnimation;
        } else {
            return super.onCreateAnimation(transit, enter, nextAnim);
        }
    }

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, @Nullable ViewGroup container,
                             @Nullable Bundle savedInstanceState) {
        if (mInflateView == null && !isInflateFinished) {
            if (getLayout() != 0) {
                mInflateView = inflater.inflate(getLayout(), null);
            } else {
                return super.onCreateView(inflater, container, savedInstanceState);
            }
        }
        if (mInflateView != null) {
            ViewGroup parent = (ViewGroup) mInflateView.getParent();
            if (parent != null) parent.removeView(mInflateView);
        }
        return mInflateView;
    }

    protected int getLoadingTargetViewId() {
        return 0;
    }

    @Override
    public void onViewCreated(View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);

        if (!isInflateFinished) {
            initPresenter();

            View targetView = view.findViewById(getLoadingTargetViewId());
            if (targetView != null && mVaryViewHelperController == null) {
                mVaryViewHelperController = new VaryViewHelperController(targetView);
            }

            mInflateView = view;

            boolean isTemplateReady = TemplateResTracker.INSTANCE.isReady();
            if (null != getArguments()) {
                getBundleBeforeDataPrepared(getArguments());
            }

            if (isTemplateReady) {
                initEventAndData(view);
            } else {
                showLoading("");
            }

            if (null != getArguments()) {
                getBundleAfterDataPrepared(getArguments());
            }

            isInflateFinished = true;
        }

    }

    private boolean doNotCheckTemplate() {
        if (getArguments() != null) {
            int supportMode = getArguments().getInt(IntentValues.SUPPORT_MODE);
            return supportMode == IntentValues.MODE_SHOW_SINGLE_FAQ ||
                    supportMode == IntentValues.MODE_URL;
        }
        return true;
    }

    @Override
    public void onDetach() {
        super.onDetach();
        mHandler.removeCallbacksAndMessages(null);
        if (isBindEventBus()) {
            EventBus.getDefault().unregister(this);
        }
        if (mPresenter != null) {
            mPresenter.detachView();
        }
        isInflateFinished = false;
        mInflateView = null;
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEventComing(EventCenter event) {

    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onNetworkStateChanged(NetworkState networkState) {

    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onTemplateReady(TemplateReadyEvent templateReadyEvent) {
        if (getActivity() != null && !getActivity().isFinishing()) {
            restoreViewState();
            onTemplateResourceReady();
        }
    }

    public <T extends View> T get(String viewId) {
        if (mInflateView != null) {
            return (T) mInflateView.findViewById(ResResolver.getViewId(viewId));
        }
        return null;
    }

    public Activity getActivity(Fragment fragment) {
        if (fragment == null) {
            return null;
        }
        while (fragment.getParentFragment() != null) {
            fragment = fragment.getParentFragment();
        }
        return fragment.getActivity();
    }

    @Override
    public void showLoading(String msg) {
        if (checkVaryViewHelper()) {
            mVaryViewHelperController.showLoading(msg);
        }
    }

    @Override
    public void showEmpty(int... layoutId) {
        if (checkVaryViewHelper()) {
            mVaryViewHelperController.showEmpty(layoutId);
        }
    }

    @Override
    public void showEmpty(View view) {
        if (checkVaryViewHelper()) {
            mVaryViewHelperController.showEmpty(view);
        }
    }

    @Override
    public void restoreViewState() {
        if (checkVaryViewHelper()) {
            mVaryViewHelperController.restore();
        }
    }

    @Override
    public void showError(String msg) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                ToastUtil.INSTANCE.makeRawToast(getContext(), msg);
            }
        });
    }

    @Override
    public void showNetError() {
        if (checkVaryViewHelper()) {
            mVaryViewHelperController.showNetworkError();
        }
    }

    @Override
    public boolean isViewRestored() {
        return checkVaryViewHelper() && mVaryViewHelperController.isViewRestored();
    }

    @NonNull
    @Override
    public String toString() {
        String simpleName = this.getClass().getSimpleName();
        if (simpleName.length() <= 0) {
            simpleName = this.getClass().getName();
            int end = simpleName.lastIndexOf('.');
            if (end > 0) {
                simpleName = simpleName.substring(end + 1);
            }
        }
        return simpleName;
    }

    private boolean checkVaryViewHelper() {
        if (null == mVaryViewHelperController) {
            TLog.d("You must return a right target view for loading");
            return false;
        }
        return true;
    }

    public static class MyHandler extends Handler {
        private WeakReference<BaseFragment> mActivity;

        public MyHandler(BaseFragment fragment) {
            mActivity = new WeakReference<>(fragment);
        }

        @Override
        public void handleMessage(Message msg) {
            if (mActivity.get() != null) {
                mActivity.get().handleMsg(msg);
            }
        }
    }

    protected boolean isBindEventBus() {
        return false;
    }

    protected void handleMsg(Message msg) {

    }

    protected void getBundleBeforeDataPrepared(Bundle extras) {

    }

    protected void getBundleAfterDataPrepared(Bundle extras) {

    }

    protected void onTemplateResourceReady() {
        if (mInflateView != null) {
            initEventAndData(mInflateView);
        }
    }

    protected abstract void initEventAndData(View contentView);

    protected abstract int getLayout();

}
