package net.aihelp.data.attachment;

import android.app.Activity;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.text.TextUtils;

import net.aihelp.common.IntentValues;
import net.aihelp.config.AIHelpContext;
import net.aihelp.core.util.luban.Luban;
import net.aihelp.core.util.luban.OnCompressListener;
import net.aihelp.ui.preview.data.PreviewInfo;
import net.aihelp.utils.RegexUtil;

import java.io.File;
import java.io.Serializable;
import java.lang.ref.WeakReference;

import androidx.fragment.app.Fragment;


public enum AttachmentPicker {

    INSTANCE;

    public static final int ATTACHMENT_TYPE_MEDIA = 1;
    public static final int ATTACHMENT_TYPE_FILE = 2;
    public static final int ATTACHMENT_TYPE_RPA = 3;

    public static final int ATTACHMENT_FILE_NOT_FOUND = -1;
    public static final int ATTACHMENT_FILE_SIZE_LIMIT_EXCEEDED = -2;
    public static final int NO_APPS_TO_OPEN_ATTACHMENTS_INTENT = -3;
    public static final int INVALID_URI = -4;

    private static final long MAX_ATTACHMENT_FILE_SIZE_LIMIT = 26214400L;

    private WeakReference<Fragment> pickerHostRef;
    private WeakReference<IAttachmentPickerListener> attachmentPickerListenerRef;

    private int attachmentType;

    public <T extends Fragment> AttachmentPicker setPickerHost(T fragment) {
        pickerHostRef = new WeakReference<>(fragment);
        return this;
    }

    public <T extends IAttachmentPickerListener> AttachmentPicker setAttachmentPickerListener(T listener) {
        if (listener != null) {
            attachmentPickerListenerRef = new WeakReference<>(listener);
        }
        return this;
    }

    public void launchPicker(int attachmentType) {
        this.attachmentType = attachmentType;
        try {
            if (pickerHostRef != null) {
                Fragment fragment = pickerHostRef.get();
                if (fragment != null && fragment.getActivity() != null) {
                    fragment.startActivityForResult(AttachmentHelper.getIntentForMedia(attachmentType),
                            IntentValues.REQUEST_CODE_ATTACHMENT);
                }
            }
        } catch (ActivityNotFoundException ignored) {
            this.sendPickFailure(NO_APPS_TO_OPEN_ATTACHMENTS_INTENT);
        }
    }

    public void onAttachmentRequestResult(int requestCode, int resultCode, Intent intent) {
        switch (requestCode) {
            case IntentValues.REQUEST_CODE_ATTACHMENT:
                if (resultCode == Activity.RESULT_OK && intent != null) {
                    onConfirmResult(intent.getData());
                }
                break;
            case IntentValues.REQUEST_CODE_PREVIEW:
                onPreviewResult(resultCode, intent);
                break;
            default:
                break;
        }
    }

    private void onConfirmResult(Uri uri) {
        if (uri != null) {
            File copiedUriFile = AttachmentHelper.getCopiedUriFile(AIHelpContext.getInstance().getContext(), uri);
            if (copiedUriFile != null && !TextUtils.isEmpty(copiedUriFile.getAbsolutePath())) {
                sendPickSuccess(copiedUriFile);
            } else {
                sendPickFailure(ATTACHMENT_FILE_NOT_FOUND);
            }
        } else {
            sendPickFailure(INVALID_URI);
        }
    }

    private void onPreviewResult(int resultCode, Intent intent) {
        if (this.attachmentPickerListenerRef != null) {
            IAttachmentPickerListener attachmentPickerListener = this.attachmentPickerListenerRef.get();
            if (attachmentPickerListener != null) {
                if (resultCode == Activity.RESULT_OK && intent != null) {
                    Serializable serializable = intent.getSerializableExtra(IntentValues.PREVIEW_INFO);
                    if (serializable instanceof PreviewInfo) {
                        PreviewInfo info = (PreviewInfo) serializable;
                        String filePath = info.getFilePath();
                        if (!TextUtils.isEmpty(filePath)) {
                            if (RegexUtil.isMatch(filePath, RegexUtil.AIHELP_SUPPORTED_IMAGE)) {
                                compressImageBeforeConfirm(filePath, attachmentPickerListener);
                            } else {
                                attachmentPickerListener.onPreviewConfirmed(filePath);
                            }
                        }
                    }
                } else {
                    attachmentPickerListener.onPreviewCanceled(attachmentType);
                }
            }
        }
    }

    private void compressImageBeforeConfirm(String filePath, IAttachmentPickerListener listener) {
        Context context = AIHelpContext.getInstance().getContext();
        if (context != null && !RegexUtil.isGifFile(filePath)) {
            Luban.with(context).load(filePath).setCompressListener(new OnCompressListener() {
                @Override
                public void onStart() {

                }

                @Override
                public void onSuccess(File compressedFile) {
                    listener.onPreviewConfirmed(compressedFile.getAbsolutePath());
                }

                @Override
                public void onError(Throwable e) {
                    listener.onPreviewConfirmed(filePath);
                }
            }).launch();
        } else {
            listener.onPreviewConfirmed(filePath);
        }
    }

    private void sendPickSuccess(File pickedFile) {
        if (this.attachmentPickerListenerRef != null) {
            IAttachmentPickerListener attachmentPickerListener = this.attachmentPickerListenerRef.get();
            if (attachmentPickerListener != null) {
                attachmentPickerListener.onPickSuccess(pickedFile);
            }
        }
    }

    private void sendPickFailure(int errorCode) {
        if (this.attachmentPickerListenerRef != null) {
            IAttachmentPickerListener attachmentPickerListener = this.attachmentPickerListenerRef.get();
            if (attachmentPickerListener != null) {
                attachmentPickerListener.onPickFailure(errorCode);
            }
        }
    }

}

