package net.aihelp.data.logic.common;

import android.content.Context;
import android.text.TextUtils;
import android.util.Log;

import net.aihelp.common.API;
import net.aihelp.common.Const;
import net.aihelp.common.SpKeys;
import net.aihelp.common.UserProfile;
import net.aihelp.config.AIHelpContext;
import net.aihelp.config.UserConfig;
import net.aihelp.core.mvp.AbsPresenter;
import net.aihelp.core.mvp.IView;
import net.aihelp.core.net.http.callback.ReqCallback;
import net.aihelp.data.local.InitRepository;

import net.aihelp.data.track.event.EventTracker;
import net.aihelp.data.track.event.utils.EventType;
import net.aihelp.init.InitHelper;
import net.aihelp.data.logic.cs.TicketStatusTracker;
import net.aihelp.ui.listener.OnMessageCountArrivedCallback;
import net.aihelp.ui.listener.OnNetworkCheckResultCallback;
import net.aihelp.utils.DeviceUuidFactory;
import net.aihelp.utils.LocaleUtil;

import org.json.JSONException;
import org.json.JSONObject;


public class InitPresenter extends AbsPresenter<IView, InitRepository> {

    public InitPresenter(Context context, String appKey, String domain, String appId, String language) {
        super(context);
        mRepo.saveInitConfig(appKey, domain, appId, language);
    }

    public void updateUserProfile(final UserConfig config) {

        String newUserId = config.getUserId();
        String oldUserId = UserProfile.USER_ID;
        mRepo.saveUserProfileConfig(config);

        // Ensure the `UserProfile.USER_ID` and `Const.TOGGLE_FETCH_MESSAGE` have updated when `fetch` is executed,
        // otherwise the request for new message count inside the Poller could be omitted
        if (!TextUtils.equals(newUserId, oldUserId)) {
            Const.TOGGLE_FETCH_MESSAGE = true;
            UnreadMessagePoller.getInstance().fetch(UnreadMessagePoller.FETCH_SOURCE_POLL);
//            EventTracker.INSTANCE.updateTargetedStatus();
        }

        TicketStatusTracker.resetTicketStatusFlags();
        if (config.isSyncCrmInfo()) {
            final String latelyCrmInfo = String.format("%s|%s", config.getUserId(), config.getUserTags());
            String cachedCrmInfo = mSp.getString(SpKeys.CRM_USER_INFO + config.getUserId());
            if (!latelyCrmInfo.equals(cachedCrmInfo)) {
                try {
                    JSONObject json = new JSONObject();
                    json.put("userName", config.getUserName());
                    json.put("uId", config.getUserId());
                    json.put("tags", config.getUserTags());
                    post(API.SYNC_VIP_INFO, json, new ReqCallback<String>() {
                        @Override
                        public void onReqSuccess(String result) {
                            mRepo.saveCrmInfo(config.getUserId(), config.getUserTags());
                        }
                    });
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }
        }
        postCrmPushTokenInfo();
        EventTracker.INSTANCE.log(EventType.API_UPDATE_USER_INFO, config.desc());
    }

    public void postCrmPushTokenInfo() {
        String pushToken = UserProfile.PUSH_TOKEN;
        int pushPlatform = UserProfile.PUSH_PLATFORM;

        if (!TextUtils.isEmpty(pushToken) && pushPlatform > 0) {
            mRepo.saveMqttPushInfo(pushToken, pushPlatform);
            if (Const.TOGGLE_CRM_TOKEN) {
                final String latelyCrmPushInfo = String.format("%s|%s|%s", UserProfile.USER_ID, pushToken, pushPlatform);
                String cachedCrmPushInfo = mSp.getString(SpKeys.PUSH_INFO_CRM);
                if (!latelyCrmPushInfo.equals(cachedCrmPushInfo)) {
                    try {
                        JSONObject params = new JSONObject();
                        params.put("token", pushToken);
                        params.put("pushTypeId", pushPlatform);
                        params.put("playerId", UserProfile.USER_ID);
                        params.put("playerName", UserProfile.USER_NAME);
                        params.put("language", Const.CORRECT_LANGUAGE);
                        params.put("deviceId", DeviceUuidFactory.id(mContext));
                        post(API.CRM_TOKEN_URL, params, new ReqCallback<String>() {
                            @Override
                            public void onReqSuccess(String result) {
                                mSp.put(SpKeys.PUSH_INFO_CRM, latelyCrmPushInfo);
                            }
                        });
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }
        }
    }

    public void updateSDKLanguage(final String sdkLanguage) {
        String formatLanguage = LocaleUtil.getFormatLanguage(sdkLanguage);
        if (!Const.ORIGINAL_LANGUAGE.equals(formatLanguage)) {
            // reset init status to avoid opening AIHelp UI before init success
            AIHelpContext.successfullyInit.set(false);

            // update memory language
            Const.ORIGINAL_LANGUAGE = formatLanguage;
            Const.CORRECT_LANGUAGE = Const.ORIGINAL_LANGUAGE;

            // request init to fetch latest data after updating language
            InitHelper.getInstance().requestInit(false);
        } else {
            Log.e("TAG", "AIHelp is using " + formatLanguage + " as global language already!");
        }
    }

    public void setUploadLogPath(String logPath) {
        mRepo.setUploadLogPath(logPath);
    }

    public void setNetworkCheckHostAddress(String hostAddress, OnNetworkCheckResultCallback listener) {
        if (TextUtils.isEmpty(hostAddress)) {
            hostAddress = "aihelp.net";
        } else {
            hostAddress = hostAddress.replace("http://", "")
                    .replace("https://", "");
        }
        mRepo.setNetworkCheckHostAddress(hostAddress);
        Const.sCheckResultListener = listener;
    }


    public void startUnreadMessagePolling(OnMessageCountArrivedCallback listener) {
        if (listener != null) {
            Const.sPollingUnreadCountListener = listener;
            UnreadMessagePoller.getInstance().start();
        }
    }

    public void stopUnreadMessagePolling() {
        if (Const.sPollingUnreadCountListener != null) {
            UnreadMessagePoller.getInstance().stop();
            Const.sPollingUnreadCountListener = null;
        }
    }

    public void fetchUnreadMessageCount(OnMessageCountArrivedCallback listener) {
        if (listener != null) {
            Const.sFetchingUnreadCountListener = listener;
            UnreadMessagePoller.getInstance().fetch(UnreadMessagePoller.FETCH_SOURCE_PROACTIVE);
        }
    }

}
