package net.aihelp.data.logic.common;

import net.aihelp.common.API;
import net.aihelp.common.Const;
import net.aihelp.common.IntentValues;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.core.net.mqtt.callback.IMqttCallback;
import net.aihelp.core.net.mqtt.config.MqttConfig;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.data.event.LoadingElvaEvent;
import net.aihelp.data.logic.cs.MessagePoller;
import net.aihelp.data.track.event.EventTracker;
import net.aihelp.data.track.event.utils.EventType;
import net.aihelp.data.model.rpa.msg.base.Message;
import net.aihelp.data.model.rpa.step.RPAStep;
import net.aihelp.data.track.statistic.StatisticTracker;
import net.aihelp.ui.cs.CustomerServiceFragment;
import net.aihelp.ui.cs.viewer.InviteRateAppHelper;
import net.aihelp.ui.faq.BaseFaqFragment;
import net.aihelp.data.logic.cs.rpa.helper.ResponseHelper;
import net.aihelp.data.logic.cs.TicketStatusTracker;
import net.aihelp.utils.TLog;
import net.aihelp.utils.UploadFileHelper;

import org.json.JSONObject;

import java.lang.ref.WeakReference;

import androidx.fragment.app.Fragment;


public class MqttCallbackImpl implements IMqttCallback {

    public static final int NOTIFICATION_TYPE_ASSIGN_CHANGE = 1;
    public static final int NOTIFICATION_TYPE_TICKET_STATUS = 2;

    private WeakReference<BaseFaqFragment> faqFragment;
    private WeakReference<CustomerServiceFragment> csFragment;

    public void updateHostView(Fragment fragment) {
        if (fragment instanceof CustomerServiceFragment) {
            csFragment = new WeakReference<>((CustomerServiceFragment) fragment);
        } else if (fragment instanceof BaseFaqFragment) {
            faqFragment = new WeakReference<>((BaseFaqFragment) fragment);
        }
    }

    @Override
    public void onMqttResponse(String topic, String response) {
        try {
            JSONObject responseObject = new JSONObject(response);
            boolean isCodeNot200 = responseObject.has("code") && responseObject.optInt("code") != 200;
            boolean hasErrorCode = responseObject.has("errorCode");
            if (isCodeNot200 || hasErrorCode) return;

            EventTracker.INSTANCE.log(EventType.CS_MESSAGE_RECEIVED, "topic", topic, "response", response);

            switch (topic) {

                case API.TOPIC_WITHDRAW:
                    if (csFragment != null && csFragment.get() != null) {
                        JSONObject pushWithdraw = new JSONObject(response);
                        csFragment.get().notifyMessageWithdrawn(pushWithdraw.optLong("withdrawkey", 0));
                    }
                    break;
                case API.TOPIC_CONVERSATION_RECEIVE:
                    if (csFragment != null && csFragment.get() != null) {
                        Message replyMsg = ResponseHelper.getRPAMessage(response);
                        RPAStep rpaStep = ResponseHelper.getRPAStep(response);
                        if (rpaStep.isEnableUpload()) {
                            UploadFileHelper.INSTANCE.tryUploadLog(true);
                        } else {
                            if (replyMsg.isNormalMessage()) {
                                csFragment.get().updateChatList(replyMsg);
                            }
                            if (rpaStep.getNextStep() != RPAStep.STEP_IGNORE_THIS) {
                                csFragment.get().updateBottomLayout(replyMsg, rpaStep);
                            }
                        }
                        ResponseHelper.notifyMqttPush("");
                    }
                    break;
                case API.TOPIC_TICKET_REJECTED:
                    TicketStatusTracker.isTicketRejected = true;
                    if (csFragment != null && csFragment.get() != null) {
                        csFragment.get().onTicketFinishedOrRejected();
                    }
                    break;
                case API.TOPIC_SUBMIT_FORM:
                    // 将填写的表单信息插入到人工消息列表中
                    if (csFragment != null && csFragment.get() != null) {
                        String formContent = responseObject.optString("msg");
                        csFragment.get().onFormSubmitted(formContent);
                    }

                    StatisticTracker.getInstance().onFormSubmitted(responseObject.optString("formId"));

                    if (Const.sSpecificFormSubmittedListener != null && responseObject.getBoolean("isSpecificForm")) {
                        Const.sSpecificFormSubmittedListener.onFormSubmitted();
                    }
                    break;

                // FAQ 未读提醒
                case API.TOPIC_FAQ_NOTIFICATION:
                    if (faqFragment != null && faqFragment.get() != null) {
                        faqFragment.get().showEntranceWithNotification(true, true);
                    }
                    break;
                // 通知
                case API.TOPIC_NOTIFICATION:
                    if (!TicketStatusTracker.isTicketRejected && !TicketStatusTracker.isTicketFinished) {
                        preparePushNotifications(response);
                    }
                    break;
            }
        } catch (Exception e) {
            // ignored
        }
    }

    private void preparePushNotifications(String response) {
        try {
            JSONObject jsonObject = new JSONObject(response);
            int notifyType = jsonObject.optInt("type");
            JSONObject notifyData = JsonHelper.getJsonObject(jsonObject, "data");
            switch (notifyType) {
                case NOTIFICATION_TYPE_ASSIGN_CHANGE:
                    TicketStatusTracker.ticketAssignType = notifyData.optInt("assignType");
                    if (csFragment != null && csFragment.get() != null) {
                        if (TicketStatusTracker.isTicketServingByAgent()) {
                            RPAStep rpaStep = new RPAStep();
                            rpaStep.setNextStep(RPAStep.STEP_MANUAL_INPUT);
                            csFragment.get().updateBottomLayout(new Message(), rpaStep);
                        }
                        csFragment.get().onTicketAssignStatusChanged();
                    }
                    break;
                case NOTIFICATION_TYPE_TICKET_STATUS:
                    int status = notifyData.optInt("status");
                    if (status == TicketStatusTracker.TICKET_STATUS_REJECTED) {
                        TicketStatusTracker.isTicketRejected = true;
                    }
                    if (csFragment != null && csFragment.get() != null) {
                        csFragment.get().onTicketFinishedOrRejected();
                        boolean goRateApp = notifyData.optBoolean("storeReview");
                        if (goRateApp) {
                            InviteRateAppHelper.showThanks(csFragment.get().getContext(), true);
                        }
                    }
                    break;
            }

        } catch (Exception e) {
            // ignored
        }
    }


    @Override
    public void onMqttConnected() {
        TLog.d("MqttCallbackImpl", "onMqttConnected");
        EventTracker.INSTANCE.log(EventType.CS_MQTT_CONNECTED);
    }

    @Override
    public void onMqttDisconnected() {
        TLog.d("MqttCallbackImpl", "onMqttDisconnected");
        EventTracker.INSTANCE.log(EventType.CS_MQTT_DISCONNECTED);
        showMqttLoading();
        if (csFragment != null && csFragment.get() != null && csFragment.get().isVisible()) {
            csFragment.get().prepareMqtt(MqttConfig.TYPE_CONNECTION_RECONNECT);
        }
    }

    @Override
    public void onMqttFailure(Throwable value) {
        TLog.d("MqttCallbackImpl", "onMqttFailure");
        EventTracker.INSTANCE.log(EventType.CS_MQTT_CONNECT_FAILURE, value);
        MessagePoller.INSTANCE.fetchMessagesSinceLatest(MessagePoller.POLL_SOURCE_FAILURE);
    }

    @Override
    public void onMqttSubscribed(int connectType) {
        TLog.d("MqttCallbackImpl", "onMqttSubscribed");
        EventTracker.INSTANCE.log(EventType.CS_MQTT_SUBSCRIBED, "ticketId", TicketStatusTracker.currentTicketId);
        MessagePoller.INSTANCE.fetchMessagesSinceLatest(connectType == MqttConfig.TYPE_CONNECTION_LOGIN ?
                MessagePoller.POLL_SOURCE_LOGIN : MessagePoller.POLL_SOURCE_RECONNECT);
    }

    @Override
    public void showMqttLoading() {
        EventBus.getDefault().post(new LoadingElvaEvent(IntentValues.SHOW_CS_LOADING));
    }

    @Override
    public void dismissMqttLoading() {
        EventBus.getDefault().post(new LoadingElvaEvent(IntentValues.HIDE_CS_LOADING));
    }

    private static final class LazyHolder {
        static final MqttCallbackImpl INSTANCE = new MqttCallbackImpl();

        private LazyHolder() {
        }
    }

    private MqttCallbackImpl() {
    }

    public static IMqttCallback getInstance() {
        return MqttCallbackImpl.LazyHolder.INSTANCE;
    }

}
