package net.aihelp.data.logic.common;

import android.os.Handler;
import android.os.Message;

import net.aihelp.data.track.event.EventTracker;
import net.aihelp.data.track.event.utils.EventType;
import net.aihelp.init.InitHelper;

import java.util.Random;

import androidx.annotation.NonNull;

/**
 * 指数退避算法:
 * https://cloud.google.com/memorystore/docs/redis/exponential-backoff
 * <p>
 * 1、向后端发出请求；
 * 2、如果请求失败，等待 1 + random_number_milliseconds 秒后再重试请求；
 * 3、如果请求失败，等待 2 + random_number_milliseconds 秒后再重试请求；
 * 4、如果请求失败，等待 4 + random_number_milliseconds 秒后再重试请求；
 * 5、依此类推，等待时间上限为 maximum_backoff（32s）；
 * 6、等待时间达到上限后，继续等待并重试，直到达到重试次数上限（但接下来的重试操作不会增加各次重试之间的等待时间）；
 */
public class RequestRetryHandler extends Handler {

    private final static int MAXIMUM_BACKOFF = 32 * 1000;

    private final OnRetryRequestListener retryListener;
    private final int maxRetryCount;

    private int retryCount;

    public RequestRetryHandler(OnRetryRequestListener retryListener, int maxRetryCount) {
        super();
        this.retryListener = retryListener;
        this.maxRetryCount = Math.max(3, maxRetryCount);
    }

    public void handleRetryRequest(int code, String errorMsg) {
        if (code >= 500) {
            InitHelper.getInstance().onAIHelpInitializedCallback(false, errorMsg);
            return;
        }
        double currentDelay = ((Math.pow(2, retryCount++) * 1000) + new Random().nextInt(1001));
        if (retryCount <= maxRetryCount) {
            sendEmptyMessageDelayed(0, (long) Math.min(currentDelay, MAXIMUM_BACKOFF));
        } else {
            removeCallbacksAndMessages(null);
            if (retryListener != null) {
                retryListener.onRetryUpToMaxCount(code, errorMsg);
            }
        }
    }

    @Override
    public void handleMessage(@NonNull Message msg) {
        if (retryListener != null) {
            retryListener.onRetryRequest();
        }
    }

    public interface OnRetryRequestListener {
        void onRetryRequest();
        void onRetryUpToMaxCount(int code, String errorMsg);
    }

}
