package net.aihelp.data.logic.common;

import android.os.Handler;
import android.os.Message;

import net.aihelp.BuildConfig;
import net.aihelp.common.API;
import net.aihelp.common.Const;
import net.aihelp.common.UserProfile;
import net.aihelp.config.AIHelpContext;
import net.aihelp.core.net.http.AIHelpRequest;
import net.aihelp.core.net.http.callback.ReqCallback;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.ui.helper.BreakReleaseHelper;
import net.aihelp.utils.DeviceUuidFactory;
import net.aihelp.utils.SpUtil;
import net.aihelp.utils.TLog;
import net.aihelp.utils.ToastUtil;

import org.json.JSONObject;

import androidx.annotation.NonNull;

public class UnreadMessagePoller extends Handler {

    private static UnreadMessagePoller sInstance;

    private final static int POLL_UNREAD_MESSAGE = 100;

    public static final int FETCH_SOURCE_POLL = 1;
    public static final int FETCH_SOURCE_PROACTIVE = 2;

    private long mLastFetchTime;
    private int unreadMessageCount;

    private UnreadMessagePoller() {
        super();
    }

    public static UnreadMessagePoller getInstance() {
        if (sInstance == null) {
            sInstance = new UnreadMessagePoller();
        }
        return sInstance;
    }

    public void start() {
        stop();
        sendEmptyMessage(POLL_UNREAD_MESSAGE);
    }

    public void fetch(int fetchSource) {
        boolean isFetched = fetchUnreadMessageCount(fetchSource);
        if (isFetched && fetchSource == FETCH_SOURCE_PROACTIVE) {
            mLastFetchTime = System.currentTimeMillis();
        }
    }

    public void stop() {
        removeCallbacksAndMessages(null);
    }

    public void reset() {
        unreadMessageCount = 0;
    }

    @Override
    public void handleMessage(@NonNull Message msg) {
        fetchUnreadMessageCount(FETCH_SOURCE_POLL);
        sendEmptyMessageDelayed(POLL_UNREAD_MESSAGE, getPollingLimit());
    }

    private boolean fetchUnreadMessageCount(int fetchSource) {
        if (interceptFetchRequest(fetchSource)) {
            onMessageUnreadCountCallback(fetchSource);
            return false;
        }
        JSONObject params = JsonHelper.getJsonObject();
        JsonHelper.put(params, "appid", Const.APP_ID);
        JsonHelper.put(params, "uid", UserProfile.USER_ID);
        AIHelpRequest.getInstance().requestGetByAsync(API.FETCH_NEW_MSG, params, new ReqCallback<String>() {
            @Override
            public void onReqSuccess(String result) {
                JSONObject jsonObject = JsonHelper.getJsonObject(result);
                Const.TOGGLE_FETCH_MESSAGE = jsonObject.optBoolean("isHaveChat");
                int newCount = jsonObject.optInt("cs_message_count");
                int cachedCount = Math.max(0, SpUtil.getInstance().getInt(UserProfile.USER_ID));
                unreadMessageCount = Math.max(0, newCount - cachedCount);
                onMessageUnreadCountCallback(fetchSource);
            }
        });
        return true;
    }

    private void onMessageUnreadCountCallback(int fetchSource) {
        switch (fetchSource) {
            case FETCH_SOURCE_POLL:
                if (Const.sPollingUnreadCountListener != null) {
                    Const.sPollingUnreadCountListener.onMessageCountArrived(unreadMessageCount);
                }
                break;
            case FETCH_SOURCE_PROACTIVE:
                if (Const.sFetchingUnreadCountListener != null) {
                    Const.sFetchingUnreadCountListener.onMessageCountArrived(unreadMessageCount);
                }
                break;
        }
    }

    private boolean interceptFetchRequest(int fetchSource) {
        if (Const.IS_SDK_SHOWING) {
            log("[UnreadMessageCount] AIHelp session is visible to user, do not need fetch for unread messages.");
            return true;
        } else if (!Const.TOGGLE_FETCH_MESSAGE) {
            log(String.format("[UnreadMessageCount] Current user(%s) does not have any active tickets at present.", UserProfile.USER_ID));
            return true;
        } else if (fetchSource == FETCH_SOURCE_POLL && !isPollingEnable()) {
            log("[UnreadMessageCount] Unread message count polling is not enabled, please contact AIHelp for more information.");
            return true;
        } else if (fetchSource == FETCH_SOURCE_PROACTIVE && !isProactiveFetchEnable()) {
            log(String.format("[UnreadMessageCount] Cached count is returned, the latest data can be obtained in %s seconds.", getDelayForNextFetch()));
            return true;
        } else if (UserProfile.USER_ID.equals(DeviceUuidFactory.id(AIHelpContext.getInstance().getContext()))) {
            log("[UnreadMessageCount] The userId you're using is AIHelp's generated deviceId, " +
                    "please verify if this is what you want.");
        }
        return false;
    }

    private void log(String msg) {
        TLog.d("AIHelp", msg);
//        if (BuildConfig.DEBUG) {
//            ToastUtil.INSTANCE.makeRawToast(AIHelpContext.getInstance().getContext(), msg, false);
//        }
    }

    private long getPollingLimit() {
        if (BreakReleaseHelper.isBreak()) {
            return 10 * 1000L;
        } else if (!isPollingEnable()) {
            return 300 * 1000L;
        } else {
            return Const.LIMIT_CHECKING_UNREAD * 1000L;
        }
    }

    private boolean isPollingEnable() {
        return Const.TOGGLE_OPEN_UNREAD_MSG && Const.LIMIT_CHECKING_UNREAD > 0;
    }

    private boolean isProactiveFetchEnable() {
        return Const.LIMIT_PROACTIVE_FETCH > 0
                && System.currentTimeMillis() - mLastFetchTime > Const.LIMIT_PROACTIVE_FETCH * 1000L;
    }

    private long getDelayForNextFetch() {
        if (Const.LIMIT_PROACTIVE_FETCH > 0) {
            return Const.LIMIT_PROACTIVE_FETCH - (System.currentTimeMillis() - mLastFetchTime) / 1000;
        }
        return -1;
    }

}
