package net.aihelp.data.logic.cs;

import net.aihelp.common.API;
import net.aihelp.core.net.http.AIHelpRequest;
import net.aihelp.core.net.http.callback.ReqCallback;
import net.aihelp.core.net.mqtt.config.MqttConfig;
import net.aihelp.data.model.rpa.msg.base.Message;
import net.aihelp.data.model.rpa.step.RPAStep;
import net.aihelp.ui.cs.CustomerServiceFragment;
import net.aihelp.data.logic.cs.rpa.helper.ResponseHelper;
import net.aihelp.ui.helper.BreakReleaseHelper;
import net.aihelp.utils.ListUtil;

import org.json.JSONObject;

import java.lang.ref.WeakReference;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import androidx.fragment.app.Fragment;

public enum MessagePoller {

    INSTANCE;

    public static final int POLL_SOURCE_LOGIN = 1;
    public static final int POLL_SOURCE_FAILURE = 2;
    public static final int POLL_SOURCE_RECONNECT = 3;
    public static final int POLL_SOURCE_SCHEDULER = 4;

    private ScheduledFuture<?> newMessageFetcher;
    private WeakReference<CustomerServiceFragment> csFragment;

    public void startMessagePolling(Fragment fragment) {
        if (fragment instanceof CustomerServiceFragment) {
            csFragment = new WeakReference<>((CustomerServiceFragment) fragment);
        }
        if (BreakReleaseHelper.isBreak()) {
            TicketStatusTracker.ticketPollInterval = 10;
        }
        if (newMessageFetcher == null) {
            newMessageFetcher = Executors.newScheduledThreadPool(1).scheduleAtFixedRate(new Runnable() {
                @Override
                public void run() {
                    if (MqttConfig.getInstance().isConnected()) {
                        return;
                    }
                    fetchMessagesSinceLatest(POLL_SOURCE_SCHEDULER);
                }
            }, TicketStatusTracker.ticketPollInterval, TicketStatusTracker.ticketPollInterval, TimeUnit.SECONDS);
        }
    }

    public void stopMessagePolling() {
        if (newMessageFetcher != null) {
            newMessageFetcher.cancel(true);
            newMessageFetcher = null;
        }
    }

    public void fetchMessagesSinceLatest(int pollSource) {
        if (TicketStatusTracker.isTicketFinished || TicketStatusTracker.isTicketRejected) {
            return;
        }
        try {
            if (csFragment != null && csFragment.get() != null) {
                CustomerServiceFragment customerServiceFragment = csFragment.get();
                JSONObject params = new JSONObject();
                params.put("messageTime", customerServiceFragment.getLastAgentMessageTimestamp());
                params.put("pollSource", pollSource);
                params.put("ticketId", TicketStatusTracker.currentTicketId);
                AIHelpRequest.getInstance().requestGetByAsync(API.POLL_MESSAGE_AFTER_MQTT_DISCONNECT, params, new ReqCallback<String>() {
                    @Override
                    public void onReqSuccess(String result) {
                        List<Message> pollResponse = ResponseHelper.getPollResponse(result);
                        Message message = ResponseHelper.getRPAMessage(result);
                        RPAStep lastRpaStep = ResponseHelper.getRPAStep(result);
                        if (TicketStatusTracker.isTicketActive && TicketStatusTracker.isTicketServingByAgent()) {
                            lastRpaStep.setNextStep(RPAStep.STEP_MANUAL_INPUT);
                        } else if (TicketStatusTracker.isTicketFinished || TicketStatusTracker.isTicketRejected) {
                            lastRpaStep.setNextStep(RPAStep.STEP_NEW_CONVERSATION);
                        }
                        if (!ListUtil.isListEmpty(pollResponse)) {
                            customerServiceFragment.updateChatList(pollResponse, false);
                        }
                        // !TicketStatusTracker.isTicketServingByAnswerBot()
                        //    - to fix the issue that user maybe typing when the poll want to update the bottom layout
                        //      which will hide keyboard, causing a very bad user experience
                        // TicketStatusTracker.isTicketFinished || TicketStatusTracker.isTicketRejected
                        //  - to fix the issue that the ticket maybe finished during user's screen is locked, and leave the
                        //      bottom layout stays as another state, and user may click a button or input something to create
                        //      a new ticket
                        if (!TicketStatusTracker.isTicketServingByAnswerBot() ||
                                (TicketStatusTracker.isTicketFinished || TicketStatusTracker.isTicketRejected)) {
                            customerServiceFragment.updateBottomLayout(message, lastRpaStep, false);
                        }
                    }
                });
            }
        } catch (Exception e) {
            // ignored
        }
    }

}
