package net.aihelp.data.logic.cs.rpa.factory;

import android.text.TextUtils;

import net.aihelp.data.model.rpa.msg.agent.AgentMessage;
import net.aihelp.data.model.rpa.msg.agent.AgentRichMessage;
import net.aihelp.data.model.rpa.msg.base.FileMessage;
import net.aihelp.data.model.rpa.msg.base.Message;
import net.aihelp.utils.RegexUtil;

import java.util.regex.Pattern;

public class AgentMessageFactory extends MessageFactory {

    public static Message getAgentMessage(String msg, long timestamp, String nickname) {
        switch (getAgentMessageType(msg)) {
            case Message.TYPE_AGENT_IMAGE:
                return getAgentImageMessage(msg, timestamp, nickname);
            case Message.TYPE_AGENT_VIDEO:
                return getAgentVideoMessage(msg, timestamp, nickname);
            case Message.TYPE_AGENT_FILE:
                return getAgentFileMessage(msg, timestamp, nickname);
            case Message.TYPE_AGENT_RICH_TEXT:
                return getAgentRichTextMessage(msg, timestamp, nickname);
            default:
                return getAgentTextMessage(msg, timestamp, nickname);
        }
    }

    private static int getAgentMessageType(String message) {
        if (isAgentImage(message)) {
            return Message.TYPE_AGENT_IMAGE;
        }
        if (isAgentVideo(message)) {
            return Message.TYPE_AGENT_VIDEO;
        }
        if (isAgentAttachment(message)) {
            return Message.TYPE_AGENT_FILE;
        }
        if (isAgentRichText(message)) {
            return Message.TYPE_AGENT_RICH_TEXT;
        }
        return Message.TYPE_AGENT_TEXT;
    }

    private static Message getAgentTextMessage(String msg, long timestamp, String nickname) {
        AgentMessage agentMessage = new AgentMessage(nickname);
        agentMessage.setTimestamp(timestamp);
        agentMessage.setContent(getPeeledContent(msg));
        agentMessage.setNormalMessage(true);
        return agentMessage;
    }

    private static Message getAgentRichTextMessage(String message, long timestamp, String nickname) {
        AgentRichMessage agentRichMessage = new AgentRichMessage(nickname);
        agentRichMessage.setTimestamp(timestamp);
        agentRichMessage.setNormalMessage(true);
        agentRichMessage.setMessageList(message);
        return agentRichMessage;
    }

    private static Message getAgentImageMessage(String msg, long timestamp, String nickname) {
        FileMessage imageMsg = new FileMessage(Message.TYPE_AGENT_IMAGE, getPeeledContent(msg));
        if (isAgentAttachment(msg)) {
            imageMsg = getAttachmentMessage(msg, Message.TYPE_AGENT_IMAGE);
        }
        imageMsg.setTimestamp(timestamp);
        imageMsg.setNickname(nickname);
        imageMsg.setNormalMessage(true);
        return imageMsg;
    }

    private static Message getAgentVideoMessage(String msg, long timestamp, String nickname) {
        FileMessage videoMsg = new FileMessage(Message.TYPE_AGENT_VIDEO, getPeeledContent(msg));
        if (isAgentAttachment(msg)) {
            videoMsg = getAttachmentMessage(msg, Message.TYPE_AGENT_VIDEO);
        }
        videoMsg.setTimestamp(timestamp);
        videoMsg.setNickname(nickname);
        videoMsg.setNormalMessage(true);
        return videoMsg;
    }

    private static Message getAgentFileMessage(String msg, long timestamp, String nickname) {
        FileMessage fileMessage = getAttachmentMessage(msg, Message.TYPE_AGENT_FILE);
        fileMessage.setNickname(nickname);
        fileMessage.setTimestamp(timestamp);
        fileMessage.setNormalMessage(true);
        return fileMessage;
    }

    private static boolean isAgentText(String message) {
        return !TextUtils.isEmpty(getPeeledContent(message));
    }

    private static boolean isAgentImage(String message) {
        return Pattern.compile(RegexUtil.REGEX_IMAGE).matcher(getPeeledContent(message)).matches()
                || isAttachmentImage(message);
    }

    private static boolean isAgentVideo(String message) {
        return Pattern.compile(RegexUtil.REGEX_VIDEO).matcher(getPeeledContent(message)).matches()
                || isAttachmentVideo(message);
    }

    private static boolean isAgentRichText(String message) {
        String replyContent = getPeeledContent(message);
        boolean replyWithHyperLinks = Pattern.compile(RegexUtil.REGEX_RICH_TEXT).matcher(replyContent).find();
        boolean replyWithAttachment = !TextUtils.isEmpty(replyContent) && hasAtLeastNAttachments(message, 1);
        boolean multipleAttachments = (TextUtils.isEmpty(replyContent) && hasAtLeastNAttachments(message, 2));
        return replyWithHyperLinks || replyWithAttachment || multipleAttachments;
    }

    private static boolean isAgentAttachment(String message) {
        return TextUtils.isEmpty(getPeeledContent(message)) && hasOnlyOneAttachment(message);
    }

}
