package net.aihelp.data.logic.cs.rpa.factory;

import android.text.TextUtils;

import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.data.logic.cs.rpa.helper.HistoryHelper;
import net.aihelp.data.model.rpa.msg.BotMessage;
import net.aihelp.data.model.rpa.msg.bot.Answer;
import net.aihelp.data.model.rpa.msg.bot.ExternalUrl;
import net.aihelp.data.model.rpa.msg.bot.Faq;
import net.aihelp.data.model.rpa.msg.bot.FormUrl;
import net.aihelp.data.model.rpa.msg.bot.SelfService;
import net.aihelp.data.logic.cs.rpa.helper.ResponseHelper;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.NonNull;

public class BotMessageFactory extends MessageFactory {

    // region - 历史消息类型枚举
    private final static int BOT_REPLY_TYPE_TEXT = 0;
    private final static int BOT_REPLY_TYPE_FORM_CONTENT = 1;
    private final static int BOT_REPLY_TYPE_ATTACHMENT = 2;
    private final static int BOT_REPLY_TYPE_EXTERNAL_URL = 3;
    private final static int BOT_REPLY_TYPE_FAQ = 4;
    private final static int BOT_REPLY_TYPE_SELF_SERVICE = 5;
    private final static int BOT_REPLY_TYPE_FORM_URL = 6;
    // endregion

    public static BotMessage getHistoricBotMessage(String msg, long timestamp, String nickname, int historyType) {
        JSONObject historyMessage = JsonHelper.getJsonObject(msg);
        BotMessage botMessage = new BotMessage(nickname);
        botMessage.setTimestamp(timestamp);
        botMessage.setContent(getPeeledContent(msg));
        botMessage.setExternalUrl(getHistoryExternalUrl(historyMessage));
        botMessage.setFaq(getHistoryFaq(historyMessage));
        botMessage.setSelfService(getHistorySelfService(historyMessage));
        botMessage.setBotAnswers(getHistoryBotAnswers(historyMessage, historyType));
        botMessage.setUserFeedback(getHistoryUserFeedback(historyMessage, historyType));
        return botMessage;
    }

    public static BotMessage getResponseBotMessage(JSONObject bot, long timestamp, String nickname) {
        int status = bot.optInt("status", -1);
        JSONObject botReply = JsonHelper.getJsonObject(bot, "reply");
        BotMessage botMessage = new BotMessage(nickname);
        botMessage.setNormalMessage(status == ResponseHelper.RPA_PROCEDURE_BOT_NORMAL);
        botMessage.setDuringRPAProcedure(status == ResponseHelper.RPA_PROCEDURE_BOT_NORMAL ||
                status == ResponseHelper.RPA_PROCEDURE_BOT_STOP_AND_WAIT);
        botMessage.setTimestamp(timestamp);
        botMessage.setContent(getBotContent(botReply));
        botMessage.setFormUrl(getFormUrl(botReply));
        botMessage.setExternalUrl(getExternalUrl(botReply));
        botMessage.setFaq(getRPAFaq(botReply, botMessage));
        botMessage.setSelfService(getSelfService(botReply));
        botMessage.setBotAnswers(getBotAnswers(botReply));
        return botMessage;
    }

    // region - 客服或机器人回复的消息内容
    public static <T> T getAttachmentContent(String message, String attachmentKey, T fallback) {
        JSONObject contentObject = getContentObject(JsonHelper.getJsonObject(message));
        JSONArray attachments = JsonHelper.getJsonArray(contentObject, "attachments");
        if (attachments.length() > 0) {
            JSONObject attachItem = JsonHelper.getJsonObject(attachments, 0);
            return JsonHelper.opt(attachItem, attachmentKey, fallback);
        }
        return fallback;
    }

    public static String getBotContent(JSONObject jsonObject) {
        return JsonHelper.optString(jsonObject, "botReply");
    }
    // endregion

    // region - 获取 Faq 数据
    public static Faq getHistoryFaq(JSONObject historyMessage) {
        JSONObject contentObject = getContentObject(historyMessage);
        JSONArray historyFaqList = JsonHelper.getJsonArray(contentObject, "faqList");
        return _getFaq(historyFaqList, false);
    }

    public static Faq getRPAFaq(JSONObject jsonObject, BotMessage botMessage) {
        JSONObject faqData = JsonHelper.getJsonObject(jsonObject, "faqData");
        String faqMsg = JsonHelper.optString(faqData, "msg");
        if (!TextUtils.isEmpty(faqMsg) && TextUtils.isEmpty(botMessage.getContent())) {
            botMessage.setContent(faqMsg);
        }
        JSONArray mqttFaqList = JsonHelper.getJsonArray(faqData, "reply");
        boolean isSimilarMatch = faqData.optInt("isSimilarMatch") == 1;
        return _getFaq(mqttFaqList, isSimilarMatch);
    }

    private static Faq _getFaq(JSONArray faqList, boolean isSimilarMatch) {
        List<Faq.FaqData> faqDataList = new ArrayList<>();
        for (int i = 0; i < faqList.length(); i++) {
            faqDataList.add(getFaqData(JsonHelper.getJsonObject(faqList, i), Faq.FAQ_SOURCE_RPA));
        }
        return new Faq(isSimilarMatch, faqDataList);
    }

    public static Faq.FaqData getFaqData(JSONObject faq, int faqSource) {
        long mainId = faq.optLong("mainId");
        long contentId = faq.optLong("contentId");
        String faqTitle = JsonHelper.optString(faq, "faqTitle");
        String faqContent = JsonHelper.optString(faq, "faqContent");
        String template = JsonHelper.optString(faq, "template");
        JSONObject formObject = JsonHelper.getJsonObject(faq, "form");
        String formUrl = JsonHelper.optString(formObject, "url");
        String formTitle = JsonHelper.optString(formObject, "title");
        boolean isSimilarMatch = formObject.optBoolean("isSimilarMatch");
        boolean isShowMore = faq.optBoolean("isShowMore", false);
        return new Faq.FaqData(faqSource, mainId, contentId, faqTitle, faqContent, template, formUrl, formTitle,
                isSimilarMatch, isShowMore);
    }
    // endregion

    // region - 获取自助服务数据
    public static SelfService getHistorySelfService(JSONObject historyMessage) {
        JSONObject contentObject = getContentObject(historyMessage);
        boolean enableSending = contentObject.optInt("allowSend") == 1;
        JSONArray rows = JsonHelper.getJsonArray(contentObject, "apiData");
        return new SelfService(enableSending, getSelfServiceData(rows));
    }

    public static SelfService getSelfService(JSONObject jsonObject) {
        JSONObject apiData = JsonHelper.getJsonObject(jsonObject, "apiData");
        boolean enableSending = apiData.optInt("allowSend") == 1;
        JSONArray rows = JsonHelper.getJsonArray(apiData, "rows");
        return new SelfService(enableSending, getSelfServiceData(rows));
    }

    public static String getSelfServiceData(JSONArray rows) {
        JSONArray selfServiceArray = new JSONArray();
        for (int i = 0; i < rows.length(); i++) {
            JSONArray dataArray = JsonHelper.getJsonArray(JsonHelper.getJsonObject(rows, i), "row");
            try {
                JSONObject selfServiceData = new JSONObject();
                for (int j = 0; j < dataArray.length(); j++) {
                    JSONObject cell = JsonHelper.getJsonObject(dataArray, j);
                    selfServiceData.put(JsonHelper.optString(cell, "key"), cell.opt("value"));
                }
                selfServiceArray.put(selfServiceData);
            } catch (Exception e) {
                // ignored
            }
        }
        return selfServiceArray.toString();
    }
    // endregion

    // region - 链接数据（外链以及表单）
    public static FormUrl getHistoryFormUrl(JSONObject jsonObject) {
        JSONObject contentObject = getContentObject(jsonObject);
        try {
            contentObject.put("msgType", 1);
        } catch (Exception e) {
            // ignored
        }
        return getMessageType(jsonObject) == BOT_REPLY_TYPE_FORM_URL ? getFormUrl(contentObject) : null;
    }

    public static FormUrl getFormUrl(JSONObject jsonObject) {
        boolean isFormUrl = jsonObject.optInt("msgType") == 1;
        String title = JsonHelper.optString(jsonObject, "title");
        String url = JsonHelper.optString(jsonObject, "url");
        return isFormUrl ? new FormUrl(title, url) : null;
    }

    public static ExternalUrl getHistoryExternalUrl(JSONObject jsonObject) {
        JSONObject contentObject = getContentObject(jsonObject);
        return getMessageType(jsonObject) == BOT_REPLY_TYPE_EXTERNAL_URL ? getExternalUrl(contentObject) : null;
    }

    public static ExternalUrl getExternalUrl(JSONObject jsonObject) {
        boolean isExternalUrl = jsonObject.optInt("msgType") != 1;
        String title = JsonHelper.optString(jsonObject, "title");
        String url = JsonHelper.optString(jsonObject, "url");
        return isExternalUrl ? new ExternalUrl(title, url) : null;
    }
    // endregion

    // region - 获取 AnswerBot 列表数据

    /**
     * 所有的 answer bot 的数据都在历史记录中返回：
     * <p>
     * 1、如果当前是 RPA 流程或者人工客服，则点击行为是弹窗查看 FAQ 详情（忽略 RPA 机器人的点击）；
     * 2、如果当前还是 AnswerBot，则点击行为是发送消息；
     */
    public static List<Answer> getHistoryBotAnswers(JSONObject historyMessage, int historyType) {
        ArrayList<Answer> botAnswers = new ArrayList<>();
        JSONArray jsonArray = JsonHelper.getJsonArray(getContentObject(historyMessage), "answerData");
        for (int i = 0; i < jsonArray.length(); i++) {
            JSONObject answerData = JsonHelper.getJsonObject(jsonArray, i);
            Answer answer = new Answer(answerData.optInt("type"), JsonHelper.optString(answerData, "answerTitle"));
            answer.setFaqData(getFaqData(JsonHelper.getJsonObject(answerData, "data"), Faq.FAQ_SOURCE_ANSWER_BOT));
            botAnswers.add(answer);
        }
        return botAnswers;
    }

    public static List<Answer> getBotAnswers(JSONObject jsonObject) {
        ArrayList<Answer> botAnswers = new ArrayList<>();
        JSONArray jsonArray = JsonHelper.getJsonArray(jsonObject, "answerData");
        for (int i = 0; i < jsonArray.length(); i++) {
            JSONObject answerData = JsonHelper.getJsonObject(jsonArray, i);
            Answer answer = new Answer(answerData.optInt("type"), JsonHelper.optString(answerData, "answerTitle"));
            answer.setFaqData(getFaqData(JsonHelper.getJsonObject(answerData, "data"), Faq.FAQ_SOURCE_ANSWER_BOT));
            botAnswers.add(answer);
        }
        return botAnswers;
    }
    // endregion

    // region - 获取历史记录的 feedback 数据
    public static int getHistoryUserFeedback(@NonNull JSONObject historyMessage, int historyType) {
        int feedback = historyMessage.optInt("feedback");
        if (historyType == HistoryHelper.HISTORY_TYPE_REFRESH && feedback == BotMessage.USER_FEEDBACK_NONE) {
            feedback = BotMessage.USER_FEEDBACK_INVISIBLE;
        }
        return feedback;
    }
    // endregion

}
