package net.aihelp.data.logic.cs.rpa.factory;

import android.text.TextUtils;

import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.data.model.rpa.msg.base.FileMessage;
import net.aihelp.utils.RegexUtil;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.regex.Pattern;

public class MessageFactory {

    public static String getPeeledContent(String message) {
        try {
            JSONObject botMessage = new JSONObject(message);
            String contentStr = JsonHelper.optString(botMessage, "content");
            try {
                // RPA 机器人消息 contentStr 也是 JsonString， 里边的 reply 字段才是真正需要的消息
                JSONObject rpaBot = new JSONObject(contentStr);
                return JsonHelper.optString(rpaBot, "reply");
            } catch (Exception e) {
                // 如果 contentStr 无法转为 JSONObject，则说明是普通机器人消息，contentStr 就是消息内容
                return contentStr;
            }
        } catch (Exception e) {
            // 如果 message 无法转换为 JSONObject，则说明当前为客服或自动化消息，message 就是消息内容
            return message;
        }
    }

    public static JSONObject getContentObject(String message) {
        JSONObject botMessage = JsonHelper.getJsonObject(message);
        return JsonHelper.getJsonObject(JsonHelper.optString(botMessage, "content"));
    }

    /**
     * 区分客服回复
     * 客服回复的消息不操作底部输入区域，因为客服回复不会发 2001 的 mqtt
     */
    public static boolean isAgentMsg(String response) {
        try {
            JSONObject jsonObject = new JSONObject(response);
            return jsonObject.optInt("usertype") == 0 && jsonObject.isNull("bot") &&
                    !TextUtils.isEmpty(JsonHelper.optString(jsonObject, "msg"));
        } catch (Exception e) {
            return false;
        }
    }

    public static JSONObject getContentObject(JSONObject historyMessage) {
        if (JsonHelper.hasKey(historyMessage, "content")) {
            String contentJsonString = JsonHelper.optString(historyMessage, "content");
            try {
                return new JSONObject(contentJsonString);
            } catch (Exception e) {
                // If the `content` string can't be initialized properly, try the historyMessage object itself.
                // This `content in historyMessage` thing is only valid in login scenario, and the historyMessage object
                // is valid in refresh scenario, in which the `content` doesn't exist al all.
            }
        }
        return historyMessage == null ? new JSONObject() : historyMessage;
    }

    protected static int getMessageType(JSONObject messageObject) {
        try {
            return messageObject.optInt("type");
        } catch (Exception e) {
            return -1;
        }
    }

    // region - 附件消息的相关判断
    public static FileMessage getAttachmentMessage(String message, int msgType) {
        String fileUrl = getAttachmentContent(message, "url", "");
        String fileName = getAttachmentContent(message, "filename", "");
        int fileSize = getAttachmentContent(message, "size", 0);
        FileMessage fileMessage = new FileMessage(msgType, fileUrl);
        fileMessage.setFileInfo(fileName, fileSize);
        fileMessage.setNormalMessage(true);
        return fileMessage;
    }

    public static <T> T getAttachmentContent(String message, String attachmentKey, T fallback) {
        return getAttachmentContent(message, 0, attachmentKey, fallback);
    }

    public static <T> T getAttachmentContent(String message, int index, String attachmentKey, T fallback) {
        JSONObject contentObject = getContentObject(JsonHelper.getJsonObject(message));
        JSONArray attachments = JsonHelper.getJsonArray(contentObject, "attachments");
        if (attachments.length() > 0 && index <= attachments.length() - 1) {
            JSONObject attachItem = JsonHelper.getJsonObject(attachments, index);
            Object result = attachItem.opt(attachmentKey);
            if (result != null) {
                if (fallback instanceof String && result.toString().equals(JSONObject.NULL.toString())) {
                    return fallback;
                }
                return (T) result;
            }
        }
        return fallback;
    }

    public static boolean isAttachmentImage(String message) {
        if (hasOnlyOneAttachment(message) && TextUtils.isEmpty(getPeeledContent(message))) {
            String type = getAttachmentContent(message, "type", "");
            return Pattern.compile(RegexUtil.ANDROID_SUPPORTED_IMAGE_SUFFIX).matcher(type).matches();
        }
        return false;
    }

    public static boolean isAttachmentVideo(String message) {
        if (hasOnlyOneAttachment(message) && TextUtils.isEmpty(getPeeledContent(message))) {
            String type = getAttachmentContent(message, "type", "");
            return Pattern.compile(RegexUtil.ANDROID_SUPPORTED_VIDEO_SUFFIX).matcher(type).matches();
        }
        return false;
    }

    private static int sizeOfAttachments(String message) {
        JSONObject msgObject = JsonHelper.getJsonObject(message);
        JSONObject contentObject = getContentObject(msgObject);
        JSONArray attachments = JsonHelper.getJsonArray(contentObject, "attachments");
        return attachments.length();
    }

    public static boolean hasOnlyOneAttachment(String message) {
        return sizeOfAttachments(message) == 1;
    }

    public static boolean hasAtLeastNAttachments(String message, int atLeast) {
        return sizeOfAttachments(message) >= atLeast;
    }
    // endregion

}
