package net.aihelp.data.logic.cs.rpa.helper;

import android.text.TextUtils;

import net.aihelp.common.CustomConfig;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.data.model.cs.ChatInfoEntity;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.regex.Pattern;

public class ChatInfoHelper {

    private static final int MESSAGE_FROM_USER = 1;
    private static final int MESSAGE_FROM_AGENT_OR_BOT = 2;
    private static final int CUSTOMER_TYPE_AGENT = 0;
    private static final int CUSTOMER_TYPE_BOT = 1;
    private static final int CUSTOMER_TYPE_AUTOMATION = 2;

    // region - login 历史记录处理
    private static String assigneeInfo;

    private static List<ChatInfoEntity> getListForLoginAndPoll(String jsonString) {
        ArrayList<ChatInfoEntity> chatInfoList = new ArrayList<>();
        Pattern pattern = Pattern.compile("[0-9]*");

        JSONObject loginMsgObject = JsonHelper.getJsonObject(jsonString);

        Iterator<String> keys = loginMsgObject.keys();
        while (keys.hasNext()) {
            String key = keys.next();

            // 提取客诉分配信息
            if ("assigneeinfo".equals(key)) {
                assigneeInfo = loginMsgObject.optString("assigneeinfo");
                continue;
            }

            // 提取时间戳
            if (CustomConfig.CustomerService.isMessageTimestampVisible && "chatTimestamp".equals(key)) {
                prepareTimeStampMessageForLogin(chatInfoList, loginMsgObject.optString("chatTimestamp"));
                continue;
            }

            // 取出所有的消息时间戳，过程中处理好人工客服的昵称
            String[] chatKey = key.split("\\|");
            if (pattern.matcher(chatKey[0]).matches()) {
                ChatInfoEntity entity = new ChatInfoEntity();
                entity.setTimeStamp(Long.parseLong(chatKey[0]));
                if (chatKey.length == 2) {
                    String agentName = chatKey[1];
                    entity.setRole("Bot".equals(agentName) ? ChatInfoEntity.ROLE_BOT : ChatInfoEntity.ROLE_AGENT);
                    entity.setAgentName(agentName);
                } else {
                    entity.setRole(ChatInfoEntity.ROLE_USER);
                    entity.setAgentName("");
                }

                // the message string from the loginMsgObject may be either a normal string or a json string,
                // if it's json string, it has a `type` field in it to mark up current message type,
                // which is the same type as in pull-to-refresh historical data
                // the type field, per se, may be useful in the future, let's extract and save it for now
                String message = loginMsgObject.optString(getConversationJSONObjectKey(entity));
                entity.setMessage(message);

                JSONObject messageJson = JsonHelper.getJsonObject(message);
                entity.setMsgType(JsonHelper.opt(messageJson, "type", -1));

                chatInfoList.add(entity);
            }
        }

        // 依据时间戳对所有的消息进行排序处理
        Collections.sort(chatInfoList);
        return chatInfoList;
    }

    public static void prepareTimeStampMessageForLogin(ArrayList<ChatInfoEntity> chatInfoList, String chatTimeStamp) {
        if (TextUtils.isEmpty(chatTimeStamp)) return;
        String[] timeStampArray = chatTimeStamp.split(",");
        for (String timeStamp : timeStampArray) {
            if (Pattern.compile("[0-9]*").matcher(timeStamp).matches()) {
                ChatInfoEntity entity = new ChatInfoEntity();
                entity.setRole(ChatInfoEntity.ROLE_TIMESTAMP);
                entity.setTimeStamp(Long.parseLong(timeStamp) - 1);
                chatInfoList.add(entity);
            }
        }
    }

    public static void prepareNickname(ChatInfoEntity entity) {
        if (!TextUtils.isEmpty(assigneeInfo)) {
            try {
                JSONArray jsonArray = new JSONArray(assigneeInfo);
                if (jsonArray.length() > 0) {
                    for (int i = 0; i < jsonArray.length(); i++) {
                        JSONObject jsonObject = jsonArray.getJSONObject(i);
                        int userId = jsonObject.optInt("userid");
                        String userName = jsonObject.optString("username");
                        if (entity.getUserId() == userId || userName.equals(entity.getAgentName())) {
                            String nickname = JsonHelper.optString(jsonObject, "nickname");
                            entity.setAgentNickname(nickname);
                            break;
                        }
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    public static String getConversationJSONObjectKey(ChatInfoEntity infoEntity) {
        String key = String.valueOf(infoEntity.getTimeStamp());
        if (!TextUtils.isEmpty(infoEntity.getAgentName())) {
            key = key + "|" + infoEntity.getAgentName();
        }
        return key;
    }
    // endregion

    // region - 下拉刷新获取历史记录
    private static List<ChatInfoEntity> getListForRefresh(String jsonString) {
        List<ChatInfoEntity> chatInfoList = new ArrayList<>();
        JSONArray jsonArray = JsonHelper.getJsonArray(jsonString);
        for (int i = 0; i < jsonArray.length(); i++) {
            JSONObject jsonObject = JsonHelper.getJsonObject(jsonArray, i);

            if (jsonObject.optBoolean("isWithdraw")) {
                continue;
            }

            if (jsonObject.optBoolean("isShowTimeLine")) {
                ChatInfoEntity entity = new ChatInfoEntity();
                entity.setRole(ChatInfoEntity.ROLE_TIMESTAMP);
                entity.setTimeStamp(jsonObject.optLong("messageTime", 0) - 1);
                chatInfoList.add(entity);
            }

            ChatInfoEntity entity = new ChatInfoEntity();
            entity.setTimeStamp(jsonObject.optLong("messageTime", 0));
            entity.setAgentNickname(JsonHelper.optString(jsonObject, "customer"));
            entity.setUserId(jsonObject.optInt("customerId", 0));
            entity.setMsgType(JsonHelper.opt(jsonObject, "messageType", -1));
            entity.setMessage(getMessageString(jsonObject));
            entity.setRole(getRole(jsonObject));
            chatInfoList.add(entity);
        }
        Collections.sort(chatInfoList);
        return chatInfoList;
    }

    private static String getMessageString(JSONObject jsonObject) {
        JSONObject messageObject = new JSONObject();
        try {
            // merge `message`, `messageType` and `feedback` into one json object to take advantage of MessageFactory's getPeelContent()
            messageObject.put("content", JsonHelper.optString(jsonObject, "message"));
            messageObject.put("type", JsonHelper.opt(jsonObject, "messageType", -1));
            messageObject.put("feedback", jsonObject.optInt("feedback"));
        } catch (Exception e) {
            // ignored
        }
        return messageObject.toString();
    }

    private static int getRole(JSONObject jsonObject) {
        switch (jsonObject.optInt("messageFrom")) {
            case MESSAGE_FROM_USER:
                return ChatInfoEntity.ROLE_USER;
            case MESSAGE_FROM_AGENT_OR_BOT:
                switch (jsonObject.optInt("customerType")) {
                    case CUSTOMER_TYPE_AGENT:
                        return ChatInfoEntity.ROLE_AGENT;
                    case CUSTOMER_TYPE_BOT:
                        return ChatInfoEntity.ROLE_BOT;
                    case CUSTOMER_TYPE_AUTOMATION:
                        return ChatInfoEntity.ROLE_AUTOMATION;
                }
        }
        return ChatInfoEntity.ROLE_TIMESTAMP;
    }
    // endregion

    public static List<ChatInfoEntity> getChatInfoList(String jsonString, int historyType) {
        switch (historyType) {
            case HistoryHelper.HISTORY_TYPE_LOGIN:
            case HistoryHelper.HISTORY_TYPE_POLL:
                return getListForLoginAndPoll(jsonString);
            case HistoryHelper.HISTORY_TYPE_REFRESH:
                return getListForRefresh(jsonString);
            default:
                return new ArrayList<>();
        }
    }

}
