package net.aihelp.data.logic.cs.rpa.helper;

import android.text.TextUtils;

import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.data.logic.cs.rpa.factory.AgentMessageFactory;
import net.aihelp.data.model.cs.ChatInfoEntity;
import net.aihelp.data.model.rpa.msg.BotMessage;
import net.aihelp.data.model.rpa.msg.base.Message;
import net.aihelp.data.logic.cs.rpa.factory.BotMessageFactory;
import net.aihelp.data.logic.cs.rpa.factory.TimeMessageFactory;
import net.aihelp.data.logic.cs.rpa.factory.UserMessageFactory;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

public class HistoryHelper {

    public static final int HISTORY_TYPE_LOGIN = 1;
    public static final int HISTORY_TYPE_REFRESH = 2;
    public static final int HISTORY_TYPE_POLL = 3;

    /**
     * 返回一条消息加工成 model 后的数据集合
     *
     * @param chatInfoEntity
     * @param historyType
     * @return 由于图片和视频可能混在 msg 中，需要将其拆出来单独展示成一条消息，所以返回的可能是个列表
     */
    private static List<Message> getListFromMsgString(ChatInfoEntity chatInfoEntity, int historyType) {
        int role = chatInfoEntity.getRole();
        long timeStamp = chatInfoEntity.getTimeStamp();
        String msg = chatInfoEntity.getMessage();
        int msgType = chatInfoEntity.getMsgType();

        ArrayList<Message> messageList = new ArrayList<>();
        switch (role) {
            case ChatInfoEntity.ROLE_TIMESTAMP:
                if (historyType == HISTORY_TYPE_POLL) break; // ignore the timestamp returned by poll api
                messageList.add(TimeMessageFactory.getTimeMessage(timeStamp));
                break;
            case ChatInfoEntity.ROLE_AGENT:
            case ChatInfoEntity.ROLE_AUTOMATION:
                ChatInfoHelper.prepareNickname(chatInfoEntity);
                messageList.add(AgentMessageFactory.getAgentMessage(msg, timeStamp, chatInfoEntity.getAgentNickname()));
                break;
            case ChatInfoEntity.ROLE_BOT:
                JSONObject msgJsonObject = JsonHelper.getJsonObject(msg);
                // 从 login 返回的数据没有 nickname，需要用 customerId 去 assigneeInfo 中查找
                // 但是从下拉刷新接口返回的数据中有 nickname，并且 msg 中也没有 customerId 字段，在这里用 hasKey 区分一下
                // 否则在下拉刷新的场景下，就会出现用 0 把本来已经设置好的昵称覆盖掉，导致最后显示成问答机器人的情况
                if (JsonHelper.hasKey(msgJsonObject, "customerId")) {
                    chatInfoEntity.setUserId(msgJsonObject.optInt("customerId"));
                }
                ChatInfoHelper.prepareNickname(chatInfoEntity);
                BotMessage botMessage = BotMessageFactory.getHistoricBotMessage(msg, timeStamp,
                        chatInfoEntity.getAgentNickname(), historyType);
                if (botMessage.hasSelfService() && botMessage.getSelfService().isEnableSend()) {
                    // 允许发送的自助服务属于 RPA 流程，会显示在 BottomView 的位置，不需要加入列表，直接把列表返回
                    return messageList;
                }
                messageList.add(botMessage);
                break;
            default:
                messageList.addAll(UserMessageFactory.getUserMessage(msgType, msg, timeStamp));
                break;
        }
        return messageList;
    }

    public static List<Message> getHistoryList(String msgJson, int historyType) {
        ArrayList<Message> list = new ArrayList<>();
        if (!TextUtils.isEmpty(msgJson)) {
            for (ChatInfoEntity chatInfoEntity : ChatInfoHelper.getChatInfoList(msgJson, historyType)) {
                list.addAll(getListFromMsgString(chatInfoEntity, historyType));
            }
        }
        return list;
    }

}
