package net.aihelp.data.track.event;

import android.content.Context;
import android.os.Build;
import android.text.TextUtils;

import net.aihelp.BuildConfig;
import net.aihelp.common.API;
import net.aihelp.common.Const;
import net.aihelp.common.UserProfile;
import net.aihelp.config.AIHelpContext;
import net.aihelp.core.net.http.config.HttpConfig;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.data.localize.LocalizeHelper;
import net.aihelp.data.localize.util.LocalizeUtil;
import net.aihelp.data.track.event.utils.EventHelper;
import net.aihelp.data.track.event.utils.EventType;
import net.aihelp.ui.helper.BreakReleaseHelper;
import net.aihelp.utils.ClipboardUtil;
import net.aihelp.utils.DeviceInfoUtil;
import net.aihelp.utils.DeviceUuidFactory;
import net.aihelp.utils.FileUtil;
import net.aihelp.utils.RegexUtil;
import net.aihelp.utils.ToastUtil;
import net.aihelp.utils.UploadFileHelper;

import org.json.JSONArray;
import org.json.JSONObject;

import java.io.File;
import java.util.Date;
import java.util.regex.Pattern;

public enum EventTracker {

    INSTANCE;

    public static long SAMPLING_LOG_SIZE_LIMIT = 2 * 1024 * 1024;
    public static String SAMPLING_LOG_VERSION = "1.0";

    private boolean isTargeted = false;
    private JSONArray targetUserIds = new JSONArray();
    private String matchingRegex = "";

    public void log(int type, Object... extraData) {
        if (isTargeted) {
//            EventHelper.INSTANCE.logEventInfoFile(type, extraData);
        }
    }

    public void syncTrackMessage() {
        isTargeted = false;
        Context context = AIHelpContext.getInstance().getContext();
        String filePath = EventHelper.INSTANCE.getFilePath(context);
        String filename = String.format("%s.txt", DeviceUuidFactory.id(context));
        File originFile = new File(filePath);
        if (originFile.exists() && originFile.length() > 0) {
            File file = FileUtil.copyAndRenameFile(context, filePath, filename);
            if (file != null && file.exists() && file.length() > 0) {
                originFile.delete();
                makeToast(context, "Uploading log, please be patient...");
                UploadFileHelper.INSTANCE.setOnUploadFileListener(new UploadFileHelper.OnUploadFileListener() {
                    @Override
                    public void onFileUploaded(String fileUrl) {
                        file.delete();
                        if (BreakReleaseHelper.isBreak()) {
                            ClipboardUtil.copy(context, fileUrl);
                        }
                    }
                }).performUpload(file);
            }
        } else {
            makeToast(context, "No logs are found.");
        }
    }

    public void prepareDataSource() {
        try {
            String content = FileUtil.getContentFromFile(LocalizeUtil.getFileLocation(LocalizeHelper.MODE_SAMPLING_FILE));
            JSONObject jsonObject = JsonHelper.getJsonObject(content);
            targetUserIds = JsonHelper.getJsonArray(jsonObject, "userIds");
            matchingRegex = JsonHelper.optString(jsonObject, "wildcardSamplingUserId");
            SAMPLING_LOG_VERSION = JsonHelper.optString(jsonObject, "version");
            SAMPLING_LOG_SIZE_LIMIT = JsonHelper.opt(jsonObject, "maxSize", SAMPLING_LOG_SIZE_LIMIT);
        } catch (Exception e) {
            // ignored
        }
        // try matching current user after data is all set
        updateTargetedStatus();
    }

    public void updateTargetedStatus() {
        if (isTargeted) return;
        try {
            // start logging if release is break
            if (!BuildConfig.DEBUG && BreakReleaseHelper.isBreak()) {
                markTargetedAndStartingPoint();
                return;
            }
            // start logging if current user is matched by regex expression
            if (RegexUtil.isRegexValid(matchingRegex) && Pattern.matches(matchingRegex, UserProfile.USER_ID)) {
                markTargetedAndStartingPoint();
                return;
            }
            // start logging if current user is matched by userId
            if (targetUserIds != null) {
                for (int i = 0; i < targetUserIds.length(); i++) {
                    String sample = targetUserIds.optString(i, "").toLowerCase();
                    if (TextUtils.equals(sample, HttpConfig.md5(UserProfile.USER_ID).toLowerCase())) {
                        markTargetedAndStartingPoint();
                        break;
                    }
                }
            }
        } catch (Exception e) {
            // ignored
        }
    }

    private void makeToast(Context context, String toast) {
        if (BreakReleaseHelper.isBreak()) {
            ToastUtil.INSTANCE.makeRawToast(context, toast);
        }
    }

    private void markTargetedAndStartingPoint() {
        isTargeted = true;
        EventHelper.INSTANCE.reset();
        log(EventType.TRACK_STARTED,
                "appKey", Const.APP_KEY,
                "domain", API.HOST_URL,
                "appId", Const.APP_ID,
                "deviceId", DeviceUuidFactory.id(AIHelpContext.getInstance().getContext()),
                "userId", UserProfile.USER_ID,
                "deviceInfo", String.format("%s %s %s", Build.MANUFACTURER, Build.MODEL, Build.VERSION.RELEASE),
                "sdkVersion", BuildConfig.SDK_VERSION,
                "originLanguage", Const.ORIGINAL_LANGUAGE,
                "countryCode", DeviceInfoUtil.getInstance().getSimCountryIso(),
                "timeZone", new Date().toString()
        );
    }
}

