package net.aihelp.data.track.event.utils;

import android.content.Context;

import net.aihelp.config.AIHelpContext;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.data.track.event.EventTracker;
import net.aihelp.utils.DateFormatUtil;
import net.aihelp.utils.DeviceUuidFactory;

import org.json.JSONArray;
import org.json.JSONObject;

import java.io.File;
import java.io.FileOutputStream;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import androidx.annotation.NonNull;

public enum EventHelper {

    INSTANCE;

    private ThreadPoolExecutor threadPoolExecutor;
    private JSONArray jsonArray = new JSONArray();

    EventHelper() {
        if (threadPoolExecutor == null) {
            LinkedBlockingQueue<Runnable> logMessagesQueue = new LinkedBlockingQueue<Runnable>();
            this.threadPoolExecutor = new ThreadPoolExecutor(1, 1, 60L, TimeUnit.SECONDS, logMessagesQueue, new ThreadFactory() {
                public Thread newThread(@NonNull Runnable r) {
                    return new Thread(r, "EventLogger");
                }
            });
        }
    }

    public void submit(Runnable runnable) {
        threadPoolExecutor.submit(runnable);
    }

    public void logEventInfoFile(int type, Object... extraData) {
        threadPoolExecutor.submit(new Runnable() {
            @Override
            public void run() {
                try {
                    jsonArray.put(getLog(type, extraData));
                    File eventLog = new File(getFilePath(AIHelpContext.getInstance().getContext()));
                    if (eventLog.length() < EventTracker.SAMPLING_LOG_SIZE_LIMIT) {
                        if (eventLog.exists()) {
                            eventLog.delete();
                        }
                        FileOutputStream fos = new FileOutputStream(eventLog.getAbsoluteFile(), true);
                        fos.write(jsonArray.toString().getBytes());
                        fos.flush();
                        fos.close();
                    }
                } catch (Exception e) {
                    // ignored
                }
            }
        });
    }

    public String getFilePath(Context context) {
        String dirPath = "";
        File filesDir = context.getFilesDir();
        String fileName = String.format("%s.json", DeviceUuidFactory.id(context));
        if (filesDir != null) {
            dirPath = filesDir.getAbsolutePath() + "/AIHelp/event";
            File dirFile = new File(dirPath);
            if (!dirFile.exists()) {
                if (dirFile.mkdirs()) {
                    return dirFile.getAbsolutePath() + File.separator + fileName;
                }
            }
        }
        return dirPath + File.separator + fileName;
    }

    private JSONObject getLog(int type, Object... extraData) {
        JSONObject jsonObject = JsonHelper.getJsonObject();
        JsonHelper.put(jsonObject, "type", type);
        JsonHelper.put(jsonObject, "time", System.currentTimeMillis());
        JsonHelper.put(jsonObject, "desc", EventType.getEventDesc(type));

        JSONObject metaObject = JsonHelper.getJsonObject();
        for (int i = 0; i < extraData.length - 1; i = i + 2) {
            Object key = extraData[i];
            Object value = extraData[i + 1];
            JsonHelper.put(metaObject, String.valueOf(key), value);
        }
        if (extraData.length % 2 == 1) {
            Object value = extraData[extraData.length - 1];
            if (value instanceof JSONObject) {
                metaObject = JsonHelper.shallowCopy((JSONObject) value);
            } else {
                JsonHelper.put(metaObject, "other", value);
            }
        }

        if (type != EventType.TRACK_STARTED) {
            JsonHelper.put(metaObject, "formattedTime", DateFormatUtil.dateFormat("yyyy-MM-dd HH:mm:ss.SSS"));
        }
        JsonHelper.put(jsonObject, "metadata", metaObject);

        return jsonObject;
    }

    public JSONArray getJsonArray() {
        return jsonArray;
    }

    public void reset() {
        jsonArray = new JSONArray();
    }

}
